#include <types.h>
#include <uart.h>

static inline int sdtr_set_baudrate(uint_t baudrate)
{
	hword_t val;
	byte_t port;

	for (port = 0; port < NR_UART_PORTS; port++) {
		/* set the LCR to allow access to the Divisor Latches */
		*(volatile ulong_t *)(UART_LCR(port)) 
			= ((UART_LCR_DLA | UART_LCR_CHAR_BIT_8) & BE_BYTE_MASK);

		/* Divisor Latch setting
		 * (system clock speed) / (16 * baudrate) */
		val = (CPU_CLK / (16 * baudrate));
	
		/* adjust baudrate: MSB first and the LSB last */
		*(volatile ulong_t *)(UART_DL2(port)) = (val >> 8) & BE_BYTE_MASK;
		*(volatile ulong_t *)(UART_DL1(port)) = val & BE_BYTE_MASK;

		/* disable access to Divisor Latches
		   at this time data can be sent and received */
		*(volatile ulong_t *)(UART_LCR(port)) 
			= UART_LCR_CHAR_BIT_8 & BE_BYTE_MASK;
		nop7();
	}

	return 0;
}

void sdtr_write_cooked(int ch, const byte_t *data, size_t length)
{
	byte_t c;

	while (length--) {
		c = *data++;
		if (c == '\n' || c == '\r') {
			sdtr_put(ch, '\n');
			sdtr_put(ch, '\r');
		} 
		else {
			sdtr_put(ch, c);
		}
		nop7();
	}	
}

size_t sdtr_read_cooked(int ch, byte_t *data, size_t length)
{
	byte_t c;
	size_t count = 0;

	while(count < length) {
		c = sdtr_get(ch);
		switch (c) {
		case '\n':
		case '\r':
			*data++ = '\n';
			count++;
			goto out;
			break;
		default:
			*data++ = c;
			count++;
			break;
		}
		nop7();
	}
out:	

	return count;
}

ssize_t sdtr_write(int ch, const void *input_data, size_t length)
{
	nop7();
	sdtr_write_cooked(ch, (byte_t*)input_data, length);
	nop7();
	return length;
}

ssize_t sdtr_read(int ch, const void *output_data, size_t length)
{
	nop7();
	length = sdtr_read_cooked(ch, (byte_t*)output_data, length);
	nop7();
	return length;
}

void sdtr_setup()
{
	sdtr_set_baudrate(SDTR_INIT_BAUDRATE);
}
