/*
 * dma.h
 *
 * DMA definitions for Responsive Multithreaded Processor
 * Constants name should be begun with DMA_ or DMAC_.
 *
 * Editors:
 *		Shinpei Kato <shinpei@ny.ics.keio.ac.jp>
 */


#ifndef __A_RMT_DMA_H__
#define	__A_RMT_DMA_H__

#include "addrmap.h"
#include "io.h" 

/* # of DMACs */
#define DMAC_NUM 3
/* size of a DMAC */
#define DMAC_SIZE	0x1000
/* #of DMAC channels */
#define DMA_CH_NUM 4

/* DMA BASE ADDRESS */
#define	DMAC0_BASE	DMA0_START	/* all channels for 1394 */
#define	DMAC1_BASE	DMA1_START	/* ch0: USB READ
								   ch1: USB WRITE
								   ch2: PCI DREQ0
								   ch3: PCI DREQ1 */
#define	DMAC2_BASE	DMA2_START	/* ch0: RL_EL_DPM_IN
								   ch1: RL_DL_DPM_IN
								   ch2: RL_EL_DPM_OUT
								   ch3: RL_DL_DPM_OUT */


#define	DMAC_OFFSET(x)		(DMAC_SIZE * (x)) /* DMA0-2 */
#define	DMA_CH_OFFSET(x)	(0x40 * (x)) /* Input channel */

/* @x: # of dmac. 
   Should use DMACX_BASE. */
#define DMAC(x) (DMAC0_BASE + DMAC_SIZE * (x))


/*
 * Registers Offset. 
 */

#define DMA_CTRL_OFFSET	0x800 /* Control register */
#define	DMA_ICLR_OFFSET	0x804 /* Interrupt Clear register */

#define	DMA_PSA_OFFSET	0x04 /* Port/Source Address register */
#define	DMA_MDA_OFFSET	0x08 /* Memory/Source Address register */
#define	DMA_ID_OFFSET	0x0c /* Internal Data register */
#define	DMA_TMC_OFFSET	0x10 /* Transfer Mode Control register */
#define	DMA_ST_OFFSET	0x14 /* STatus register */
#define	DMA_LN_OFFSET	0x18 /* transfer LeNgth register */

/* Control registers */
#define DMAC0_CTRL (DMAC0_BASE + DMA_CTRL_OFFSET)
#define DMAC1_CTRL (DMAC1_BASE + DMA_CTRL_OFFSET)
#define DMAC2_CTRL (DMAC2_BASE + DMA_CTRL_OFFSET)

/* Interrupt Clear registers, @x: channel*/
#define DMAC0_ICLR (DMAC0_BASE + DMA_ICLR_OFFSET)
#define DMAC1_ICLR (DMAC1_BASE + DMA_ICLR_OFFSET)
#define DMAC2_ICLR (DMAC2_BASE + DMA_ICLR_OFFSET)

/* @x: channel */
#define DMAC0_PSA(x)	(DMAC0_BASE + DMA_CH_OFFSET(x) + DMA_PSA_OFFSET)
#define DMAC0_MDA(x)	(DMAC0_BASE + DMA_CH_OFFSET(x) + DMA_MDA_OFFSET)
#define DMAC0_ID(x)		(DMAC0_BASE + DMA_CH_OFFSET(x) + DMA_ID_OFFSET)
#define DMAC0_TMC(x)	(DMAC0_BASE + DMA_CH_OFFSET(x) + DMA_TMC_OFFSET)
#define DMAC0_ST(x)		(DMAC0_BASE + DMA_CH_OFFSET(x) + DMA_ST_OFFSET)
#define DMAC0_LN(x)		(DMAC0_BASE + DMA_CH_OFFSET(x) + DMA_LN_OFFSET)

#define DMAC1_PSA(x)	(DMAC1_BASE + DMA_CH_OFFSET(x) + DMA_PSA_OFFSET)
#define DMAC1_MDA(x)	(DMAC1_BASE + DMA_CH_OFFSET(x) + DMA_MDA_OFFSET)
#define DMAC1_ID(x)		(DMAC1_BASE + DMA_CH_OFFSET(x) + DMA_ID_OFFSET)
#define DMAC1_TMC(x)	(DMAC1_BASE + DMA_CH_OFFSET(x) + DMA_TMC_OFFSET)
#define DMAC1_ST(x)		(DMAC1_BASE + DMA_CH_OFFSET(x) + DMA_ST_OFFSET)
#define DMAC1_LN(x)		(DMAC1_BASE + DMA_CH_OFFSET(x) + DMA_LN_OFFSET)

#define DMAC2_PSA(x)	(DMAC2_BASE + DMA_CH_OFFSET(x) + DMA_PSA_OFFSET)
#define DMAC2_MDA(x)	(DMAC2_BASE + DMA_CH_OFFSET(x) + DMA_MDA_OFFSET)
#define DMAC2_ID(x)		(DMAC2_BASE + DMA_CH_OFFSET(x) + DMA_ID_OFFSET)
#define DMAC2_TMC(x)	(DMAC2_BASE + DMA_CH_OFFSET(x) + DMA_TMC_OFFSET)
#define DMAC2_ST(x)		(DMAC2_BASE + DMA_CH_OFFSET(x) + DMA_ST_OFFSET)
#define DMAC2_LN(x)		(DMAC2_BASE + DMA_CH_OFFSET(x) + DMA_LN_OFFSET)

/* @x: dmac id, @y: channel 
   should use DMACX_HOGE */
#define	DMAC_PSA(x, y)	(DMAC(x) + DMA_CH_OFFSET(y) + DMA_PSA_OFFSET)
#define	DMAC_MDA(x, y)	(DMAC(x) + DMA_CH_OFFSET(y) + DMA_MDA_OFFSET)
#define	DMAC_ID(x, y)	(DMAC(x) + DMA_CH_OFFSET(y) + DMA_ID_OFFSET)
#define	DMAC_TMC(x, y)	(DMAC(x) + DMA_CH_OFFSET(y) + DMA_TMC_OFFSET)
#define	DMAC_ST(x, y)	(DMAC(x) + DMA_CH_OFFSET(y) + DMA_ST_OFFSET)
#define	DMAC_LN(x, y)	(DMAC(x) + DMA_CH_OFFSET(y) + DMA_LN_OFFSET)

/*
 * I/O-specific DMA.
 */

/* DMA #. see DMA BASE ADDRESS area above. */
#define DMAC_1394	0
#define DMAC_PCI		1
#define DMAC_USB		1
#define DMAC_RL		2

#define DMA_CH_USB_RD		0
#define DMA_CH_USB_WR		1
#define DMA_CH_PCI_DREQ0	2
#define DMA_CH_PCI_DREQ1	3
#define DMA_CH_RL_EL_IN		0
#define DMA_CH_RL_DL_IN		1
#define DMA_CH_RL_EL_OUT	2
#define DMA_CH_RL_DL_OUT	3

/* 
 * DMA COMMAND 
 */
/* stop DMA */
#define	DMA_STOP	0x0

/* clear status bit */
#define	DMA_ST_CLR	0x0

/* clear int bit */
#define	DMA_INT_CLR	0x0

/* more precise DMA control register settings */
#define DMA_DAU     0x4000 /* Destination Address Update */
#define DMA_SAU     0x2000 /* Source Address Update */
#define DMA_BM      0x1000 /* Burst Mode */
#define DMA_RL      0x0800 /* Responsive Link */
#define DMA_PCI     0x0400 /* PCI */
#define DMA_MTM     0x0200 /* Memorty To Memory */
#define DMA_MR      0x0100 /* Memory Read */
#define DMA_IO32    0x0080 /* 32-bit I/O Port */
#define DMA_IO16    0x0040 /* 16-bit I/O Port */
#define DMA_IO8     0x0020 /* 8-bit I/O Port */
#define DMA_S16     0x0010 /* Swap at 16-bit*/
#define DMA_S8      0x0008 /* Swap at 8-bit */
#define DMA_IER     0x0004 /* Interrupt enable of ER-bit */
#define DMA_IED     0x0002 /* Interrupt enable of ED-bit */
#define DMA_ST      0x0001 /* Start */

/* status bit */
#define	DMA_ER	0x0002
#define	DMA_ED	0x0001


/*
 * Transfer Modes.
 */

/* for ROM to SDRAM */
#define DMA_ROM2SDRAM \
	(DMA_BM | DMA_MTM | DMA_ST)

/* for Responsive Link */
#define DMA_MODE_RL_READ_SET     \
	(DMA_DAU | DMA_SAU | DMA_RL | DMA_MTM | DMA_IER | DMA_IED)
#define DMA_MODE_RL_WRITE_SET    \
	(DMA_SAU | DMA_RL | DMA_MTM)

#define DMA_MODE_RL_READ     \
	(DMA_DAU | DMA_SAU | DMA_RL | DMA_MTM | DMA_ST)
#define DMA_MODE_RL_WRITE    \
	(/*DMA_BM|*/ DMA_SAU | DMA_RL | DMA_MTM | DMA_ST)

/* for memory to memory transfer */
#define	DMA_M2M	(DMA_DAU | DMA_SAU | DMA_BM | DMA_MTM | DMA_ST )

/* for PCI */
#define	DMA_PCI_READ_SET	\
	(DMA_DAU |  DMA_PCI | /* DMA_BM | */ DMA_IO32)
#define	DMA_PCI_WRITE_SET	\
	(DMA_SAU | DMA_PCI | /* DMA_BM | */ DMA_MR | DMA_IO32)

#define	DMA_PCI_READ	(DMA_PCI_READ_SET|DMA_ST)
#define	DMA_PCI_WRITE	(DMA_PCI_WRITE_SET|DMA_ST)

#define	DMA_PCI_READ_SET_DUMMY	0x0
#define	DMA_PCI_WRITE_SET_DUMMY	0x0

/* set value @x to DMA register @addr*/
#define DMA_REG_SET(addr, x) MEM_WR_W(x, addr)
/* get value from DMA register @addr */
#define DMA_REG_GET(addr) MEM_RD_W(addr)

#endif	/* __RT_FRONTIER_RMT_DMA_H__ */
