#include <stdio.h>
#include "libelf.h"
#include <string.h>

#define OP_RMTNOP 0x03e00000
#define OP_NOP	0x0
#define OP_SYNC	0x0000000f

#define BYTE_GET(field)  byte_get_big_endian(field, sizeof(field))


typedef struct {
	unsigned int offset;
	unsigned int size;
	unsigned int addr;
} section;

/* ELF Header (32-bit implementations) */
typedef struct {
  unsigned char e_ident[16];            /* ELF "magic number" */
  unsigned char e_type[2];              /* Identifies object file type */
  unsigned char e_machine[2];           /* Specifies required architecture */
  unsigned char e_version[4];           /* Identifies object file version */
  unsigned char e_entry[4];             /* Entry point virtual address */
  unsigned char e_phoff[4];             /* Program header table file offset */
  unsigned char e_shoff[4];             /* Section header table file offset */
  unsigned char e_flags[4];             /* Processor-specific flags */
  unsigned char e_ehsize[2];            /* ELF header size in bytes */
  unsigned char e_phentsize[2];         /* Program header table entry size */
  unsigned char e_phnum[2];             /* Program header table entry count */
  unsigned char e_shentsize[2];         /* Section header table entry size */
  unsigned char e_shnum[2];             /* Section header table entry count */
  unsigned char e_shstrndx[2];          /* Section header string table index */
} Elf32_External_Ehdr;

/* Section header */
typedef struct {
  unsigned char sh_name[4];             /* Section name, index in string tbl */
  unsigned char sh_type[4];             /* Type of section */
  unsigned char sh_flags[4];            /* Miscellaneous section attributes */
  unsigned char sh_addr[4];             /* Section virtual addr at execution */
  unsigned char sh_offset[4];           /* Section file offset */
  unsigned char sh_size[4];             /* Size of section in bytes */
  unsigned char sh_link[4];             /* Index of another section */
  unsigned char sh_info[4];             /* Additional section information */
  unsigned char sh_addralign[4];        /* Section alignment */
  unsigned char sh_entsize[4];          /* Entry size if section holds table */
} Elf32_External_Shdr;


int byte_get_big_endian (unsigned char* field, int size)
{
	switch (size) {
    case 1:
		return *field;

    case 2:
		return ((unsigned int) (field[1])) | (((int) (field[0])) << 8);

    case 4:
		return ((unsigned long) (field[3]))
			|   (((unsigned long) (field[2])) << 8)
			|   (((unsigned long) (field[1])) << 16)
			|   (((unsigned long) (field[0])) << 24);

    case 8:
		/* Although we are extracing data from an 8 byte wide field, we
		   are returning only 4 bytes of data.  */
		return ((unsigned long) (field[7]))
			|   (((unsigned long) (field[6])) << 8)
			|   (((unsigned long) (field[5])) << 16)
			|   (((unsigned long) (field[4])) << 24);

    default:
		printf ("Unhandled data length: %d\n", size);
		abort ();
    }

	return 0;
}


void show_usage(void) {
	printf("Usage: respii-nopds-patch [OPTION] SOURCE DEST\n");
	printf("SOURCE is a binary file generated by gcc-respii and \n");
	printf("DEST is a copy of SOURCE but all the instructions \n");
	printf("which were flooded out are back in original positions.\n");
	printf("\n");
//	printf("Options:\n");
//	printf("  -v     Show the swaped instructions.\n");
//	printf("  --help Show how to use this program and quit.\n");
//	printf("\n\n");
}


int main(int argc, char** argv)
{
	Elf32_Ehdr elf_header;
	Elf32_Shdr *section_headers = NULL;
	Elf32_Shdr *section_header = NULL;
	Elf32_External_Ehdr xeh;
	Elf32_External_Shdr *xshdrs = NULL;
	Elf32_Shdr *internal = NULL;
	unsigned int i, j, k;
	int counter, fp_pos;
	char *bakfile = NULL, *srcfile = NULL, *destfile = NULL;
	FILE *src = NULL, *dest = NULL;
	unsigned int word;
	unsigned char big_word[4], little_word[4];
	unsigned char byte[1];
	int text_num = 0;
	section* text_sections = NULL;
	int option_v = 1;

	if (argc < 3) {
		printf("too few arguments.\n\n");
		show_usage();
		return 0;
	}
	else if(argc > 3) {
		printf("too many arguments.\n\n");
		show_usage();
		return 0;
	}

	srcfile = argv[1];
	destfile = argv[2];

	printf("source file name: %s\n", srcfile);
	printf("destination file name: %s\n", destfile);
	printf("\n");

	/* open src file. */
	if (!(src = fopen(srcfile, "r"))) {
		printf("Failed to open %s\n", srcfile);
		goto end;
	}

	/* open dest file. */
	if (!(dest = fopen(destfile, "w"))) {
		printf("Failed to open %s\n", destfile);
		goto end;
	}

	/* read elf header. */
	if (!(fread(&xeh, sizeof(xeh), 1, src))) {
		printf("Failed to read %s\n", srcfile);
	}

	/* transform to little endian byte */
	elf_header.e_type      = BYTE_GET (xeh.e_type);
	elf_header.e_machine   = BYTE_GET (xeh.e_machine);
	elf_header.e_version   = BYTE_GET (xeh.e_version);
	elf_header.e_entry     = BYTE_GET (xeh.e_entry);
	elf_header.e_phoff     = BYTE_GET (xeh.e_phoff);
	elf_header.e_shoff     = BYTE_GET (xeh.e_shoff);
	elf_header.e_flags     = BYTE_GET (xeh.e_flags);
	elf_header.e_ehsize    = BYTE_GET (xeh.e_ehsize);
	elf_header.e_phentsize = BYTE_GET (xeh.e_phentsize);
	elf_header.e_phnum     = BYTE_GET (xeh.e_phnum);
	elf_header.e_shentsize = BYTE_GET (xeh.e_shentsize);
	elf_header.e_shnum     = BYTE_GET (xeh.e_shnum);
	elf_header.e_shstrndx  = BYTE_GET (xeh.e_shstrndx);

	/* get section headers. */
	xshdrs = (Elf32_External_Shdr *) 
		malloc (elf_header.e_shentsize * elf_header.e_shnum);

	/* seek file. */
	if (fseek(src, elf_header.e_shoff, SEEK_SET)) {
		printf("Failed to seek %s\n", srcfile);
		goto end;
	}

	/* read section headers. */
	if (!(fread(xshdrs, elf_header.e_shentsize * elf_header.e_shnum, 1, src))) {
		printf("Failed to read %s\n", srcfile);
		goto end;
	}

	/* allocate section headers. */
	section_headers = (Elf32_Shdr *) 
		malloc (elf_header.e_shnum * sizeof (Elf32_Shdr));

	for (i = 0, section_header = section_headers;
		 i < elf_header.e_shnum;
		 i++, section_header++) {
		section_header->sh_name      = BYTE_GET (xshdrs[i].sh_name);
		section_header->sh_type      = BYTE_GET (xshdrs[i].sh_type);
		section_header->sh_flags     = BYTE_GET (xshdrs[i].sh_flags);
		section_header->sh_addr      = BYTE_GET (xshdrs[i].sh_addr);
		section_header->sh_offset    = BYTE_GET (xshdrs[i].sh_offset);
		section_header->sh_size      = BYTE_GET (xshdrs[i].sh_size);
		section_header->sh_link      = BYTE_GET (xshdrs[i].sh_link);
		section_header->sh_info      = BYTE_GET (xshdrs[i].sh_info);
		section_header->sh_addralign = BYTE_GET (xshdrs[i].sh_addralign);
		section_header->sh_entsize   = BYTE_GET (xshdrs[i].sh_entsize);
    }

	for (i = 0, section_header = section_headers;
		 i < elf_header.e_shnum;
		 i++, section_header++) {
		if (SHF_EXECINSTR & section_header->sh_flags) {
			text_num++;
		} 
	}

	text_sections = (section*)malloc(sizeof(section) * text_num);
	j = 0;
	for (i = 0, section_header = section_headers;
		 i < elf_header.e_shnum;
		 i++, section_header++) {
		if (SHF_EXECINSTR & section_header->sh_flags) {
			if (section_header->sh_size % sizeof(word) != 0) {
				printf("size of text section isn't word times.\n");
				goto end;
			}
			text_sections[j].offset = section_header->sh_offset;
			text_sections[j].size = section_header->sh_size;
			text_sections[j].addr = section_header->sh_addr;
			j++;
		} 
	}

	/* seek to the begining. */
	if (fseek(src, 0, SEEK_SET)) {
		printf("Failed to seek %s\n", srcfile);
		goto end;
	}

	fp_pos = 0; 
	counter = 0;
	for (i = 0; i < text_num; i++) {
		fp_pos = ftell(src);
		/* just copy to dest from src until text section. */
		for (j = fp_pos; j < text_sections[i].offset; j++) {
			fread(&byte, sizeof(byte), 1, src);
			fwrite(&byte, sizeof(byte), 1, dest);
		}

#if 0
		/* now we reached .text section. */
		for (j = 0; j < text_sections[i].size; j++) {
			fread(&byte, sizeof(byte), 1, src);
			fwrite(&byte, sizeof(byte), 1, dest);
		}
#else
		for (j = 0; j < (text_sections[i].size / sizeof(word)); j++) {
			/* read an instruction */
			fread(big_word, sizeof(word), 1, src);
			/* trasform to little endian from big endian */
			word = BYTE_GET(big_word);
			if (word == OP_NOP) {
				((unsigned long*)little_word)[0] = OP_SYNC;
				word = BYTE_GET(little_word);
				/* write the instruction */
				fwrite(&word, sizeof(word), 1, dest);
			}
			else {
				/* write the instruction */
				fwrite(big_word, sizeof(word), 1, dest);
			}
		}
		for (j = j*sizeof(word); j < text_sections[i].size; j++)  {
			fread(&byte, sizeof(byte), 1, src);
			fwrite(&byte, sizeof(byte), 1, dest);
		}
#endif
		if (i == text_num - 1) {
			while (fread(&byte, sizeof(byte), 1, src))
				fwrite(&byte, sizeof(byte), 1, dest);
		}
	}

	fchmod(fileno(dest), 0755);

 end:
	/* free memory */
	if (xshdrs)
		free(xshdrs);
	if (text_sections)
		free(text_sections);
	if (bakfile)
		free (bakfile);

	/* close file stream. */
	if (dest)
		fclose(dest);
	if (src)
		fclose(src);

	
	return 0;
}
