/*
	pcitool.c: shows I/O specific registers of Responsive Processor Board.
		/proc/resproc/pci	PCI registers
*/

#ifndef __KERNEL__
#  define __KERNEL__
#endif
#ifndef MODULE
#  define MODULE
#endif

#if 0
#define __NO_VERSION__ /* don't define kernel_verion in module.h */
#include <linux/version.h>
char kernel_version [] = UTS_RELEASE;

#include <linux/module.h>
#else
#include <linux/autoconf.h>
#if	defined(CONFIG_MODVERSIONS) && !defined(MODVERSIONS)
#	define	MODVERSIONS
#endif

#ifdef	MODVERSIONS
#	include <linux/modversions.h>
#endif
#include <linux/module.h>
#endif

#include <linux/sched.h>
#include <linux/proc_fs.h>
#include <linux/pci.h>

#include <asm/io.h>                     /* in/out byte, word... */

#include	"resproc.h"

/* for driver to be closed even if it malfunctions */
#define B	'b'
#define W	'w'
#define L	'l'

MODULE_DESCRIPTION("proc tool for Responsive Processor");
MODULE_AUTHOR("Hidenori Kobayashi");
MODULE_LICENSE("GPL");

struct pci_dev *pci;
u32		pci_ioaddr;

static struct proc_dir_entry *proc_parent;

/* table for storing triple (length, place, message) */
struct	Table {
	char		type;	/* Byte, Word, Long(dword) */
	unsigned char	place;	/* offset */
	char		*msg;
};

/* PCI configuration space */
struct	Table	pci_conf_tbl[] = {
	{W, PCI_VENDOR_ID,		"Vendor ID        "},
	{W, PCI_DEVICE_ID,		"Device ID        "},
	{W, PCI_COMMAND,		"Command Register "},
	{W, PCI_STATUS,			"Status Register  "},
	{B, PCI_REVISION_ID,		"Revision ID      "},
	{B, PCI_CLASS_PROG,		"Device Base Class"},
	{W, PCI_CLASS_DEVICE,		"Device Class     "},
	{B, PCI_CACHE_LINE_SIZE,	"Cache Line Size  "},
	{B, PCI_LATENCY_TIMER,		"Latency Timer    "},
	{B, PCI_HEADER_TYPE,		"Header Type      "},
	{B, PCI_BIST,			"BIST             "},
	{L, PCI_BASE_ADDRESS_0,		"Base Address 0   "},
	{L, PCI_BASE_ADDRESS_1,		"Base Address 1   "},
	{L, PCI_BASE_ADDRESS_2,		"Base Address 2   "},
	{L, PCI_BASE_ADDRESS_3,		"Base Address 3   "},
	{L, PCI_BASE_ADDRESS_4,		"Base Address 4   "},
	{L, PCI_BASE_ADDRESS_5,		"Base Address 5   "},
	{L, PCI_CARDBUS_CIS,		"CardBUS CIS Ptr  "},
	{W, PCI_SUBSYSTEM_VENDOR_ID,		"Subsys Vendor ID "},
	{W, PCI_SUBSYSTEM_ID,	"Subsys Device ID "},
	{L, PCI_ROM_ADDRESS,		"Exp ROM Base Addr"},
	{B, PCI_INTERRUPT_LINE,		"IRQ Line         "},
	{B, PCI_INTERRUPT_PIN,		"IRQ Pin          "},
	{B, PCI_MIN_GNT,		"Min_Gnt          "},
	{B, PCI_MAX_LAT,		"Max_Lat          "},
	{'\0', 0, ""}
};

/* resproc board configuration space */
struct	Table	pci_spc_tbl[] = {
	{B, OS_CMD1,		"CMD1 [31:24]     "},
	{B, OS_CMD2,		"CMD2 [23:16]     "},
	{B, OS_ASIR,		"ASI Register     "},
	{L, OS_MB_A,		"Mailbox A  PCI->S"},
	{L, OS_MB_B,		"Mailbox B  PCI<-S"},
	{L, OS_L_AD,		"Local AD         "},
	{L, OS_LBAP,		"Local Bus Ac-Port"},
	{L, OS_PBAP,		"PCI   Bus Ac-Port"},
	{L, OS_CAR,		"CAR /BAR      ---"},
	{L, OS_CDCR,		"CDCR/BDCR      | "},
	{B, OS_DMR,		"DMR           DMA"},
	{B, OS_DSR,		"DSR            | "},
	{B, OS_DMSR,		"DMSR/DAMR     ---"},
	{W, OS_FDCR,		"FDCR         ----"},
	{W, OS_FBSR,		"FBSR          |  "},
	{W, OS_WQPR,		"WQPR         FIFO"},
	{W, OS_RQPR,		"RQPR          |  "},
	{B, OS_MRBER,		"MRBER         |  "},
	{B, OS_FSR,		"FSR           |  "},
	{B, OS_FMSR,		"FMSR/FAMR    ----"},
	{'\0', 0, ""}
};

/* sprintf for common configuration (using pci bios) */
int sprt_cnf(struct pci_dev *dev, char *buf, char len,
			unsigned char place, char *msg) {
	static	u8	byte;
	static	u16	word;
	static	u32	dword, j;
	static	int	i,l;
	int		pos = 0;

	switch (len) {
		case B:
			pci_read_config_byte(dev, place, &byte);
			j = byte;
			pos += sprintf(buf+pos, " %s  (%2xH) 0x%02x\t\t",
				msg, place, j);
			l = 7;
			break;
		case W:
			pci_read_config_word(dev, place, &word);
			j = word;
			pos += sprintf(buf+pos, " %s  (%2xH) 0x%04x\t",
				msg, place, j);
			l = 15;
			break;
		case L:
			pci_read_config_dword(dev, place, &dword);
			j = dword;
			pos += sprintf(buf+pos, " %s  (%2xH) 0x%08x\t",
				msg, place, j);
			l = 31;
			break;
	}
	for ( i=l ; i>=0; i-- ) {
		pos +=	(j & (0x1<<i))	? sprintf(buf+pos, "1")
					: sprintf(buf+pos, "0");
		if (i % 8 == 0) pos += sprintf(buf+pos, " ");
	}
	pos += sprintf(buf+pos, "\n");

	return pos;
}

int sprt_io(u32 ioaddr, char *buf, char len, unsigned char offset, char *msg) {
	static	u8	byte;
	static	u16	word;
	static	u32	dword, j;
	static	int	i,l;
	int		pos = 0;

	switch (len) {
		case B:
			byte = inb(ioaddr+offset);
			j = byte;
			l = 7;
			pos += sprintf(buf+pos, " %s  (%2xH) 0x%02x\t\t",
				msg, offset, j);
			break;
		case W:
			word = inw(ioaddr+offset);
			j = word;
			l = 15;
			pos += sprintf(buf+pos, " %s  (%2xH) 0x%04x\t",
				msg, offset, j);
			break;
		case L:
			dword = inl(ioaddr+offset);
			j = dword;
			l = 31;
			pos += sprintf(buf+pos, " %s  (%2xH) 0x%08x\t",
				msg, offset, j);
			break;
	}
	for ( i=l ; i>=0; i-- ) {
		pos +=	(j & (0x1<<i))	? sprintf(buf+pos, "1")
					: sprintf(buf+pos, "0");
		if (i % 8 == 0) pos += sprintf(buf+pos, " ");
	}
	pos += sprintf(buf+pos, "\n");

	return pos;
}

/* entry point functions for PCI */
int reg_pci_read_proc(char *page, char **start, off_t off,
                   int count, int *eof, void *data) {
	int	i, pos = 0;

	pos += sprintf(page+pos, "----- Reg Name ---- Addr - Value ------ Value(2)");
	pos += sprintf(page+pos, "  MSb->LSb\n\t\t\t----- PCI conf ----\n");

	for (i=0; pci_conf_tbl[i].type!='\0'; i++) {
		pos += sprt_cnf(pci, page+pos,
				pci_conf_tbl[i].type,
				pci_conf_tbl[i].place, pci_conf_tbl[i].msg);
	}

	pci_read_config_dword(pci, PCI_BASE_ADDRESS_0, &pci_ioaddr);
	pci_ioaddr &= ~1;	/* clear resource type indicator */

	pos += sprintf(page+pos, "\t\t\t----- I/O space ---\n");
	for (i=0; pci_spc_tbl[i].type!='\0'; i++) {
		pos += sprt_io(pci_ioaddr, page+pos, pci_spc_tbl[i].type,
				pci_spc_tbl[i].place, pci_spc_tbl[i].msg);
	}

	return pos;
}

/*
 * init, cleanup modules
 */
int init_module( void )
{
	if ( !pci_present() ){
		printk(KERN_INFO "No PCI present\n");
		return 0;
	}

	/* PCI */
	if ((pci = pci_find_device(PCI_VENDOR_ID_RESPROC, 
					PCI_DEVICE_ID_RESPROC, pci))){
		printk(KERN_INFO "VID=0x10cf,DEVID=0x200d found at"
			" Bus %x: Fct %x (dev %d, fun %d)\n",
			pci->bus->number, pci->devfn,
			PCI_SLOT(pci->bus->number), PCI_FUNC(pci->devfn));
	}

	/* register /proc file systems */
	proc_parent = proc_mkdir("resproc", NULL);
	if ( pci != NULL )
		create_proc_read_entry("pci", 0, proc_parent, 
				reg_pci_read_proc, NULL );

	/* hide symbols */
	EXPORT_NO_SYMBOLS;	

	return 0;
}

void cleanup_module(void)
{
	/* unregister /proc file systems */
	if ( pci != NULL )
		remove_proc_entry("pci", proc_parent);
	remove_proc_entry("resproc", NULL);
}
