/*
 * resproc.c -- pci device driver module for Responsive Processor Eva. Board
 *			made from Main.c
 *
 * $Id: resproc.c,v 1.2 2001/03/02 08:16:56 kobahide Exp $
 *********/

#ifndef __KERNEL__
#  define __KERNEL__
#endif
#ifndef MODULE
#  define MODULE
#endif

/*
 * to use resproc module with version support
 */

#include <linux/autoconf.h>
#if defined(CONFIG_MODVERSIONS) && !defined(MODVERSIONS)
#	define MODVERSIONS
#endif

#ifdef 	MODVERSIONS
#	include	<linux/modversions.h>
#endif

#include <linux/module.h>

#include <asm/io.h>			/* inb, ... */
#include <linux/ioport.h>	/* check_region() */
#include <linux/kernel.h> /* printk() */
#include <linux/slab.h> /* kmalloc() */
#include <linux/init.h>
#include <linux/fs.h>     /* everything... */
#include <linux/errno.h>  /* error codes */
#include <linux/types.h>  /* size_t */
#include <linux/fcntl.h>        /* O_ACCMODE */
#include <linux/pci.h>
#include <linux/devfs_fs_kernel.h>

#include <linux/wait.h>	

#include <asm/system.h>   /* cli(), *_flags */
#include <asm/uaccess.h>	/* __copy_to_user() */

#include "resproc.h"        /* local definitions */


/*
 * I don't use static symbols here, because I export no symbols
 */
devfs_handle_t resproc_devfs_dir;

DECLARE_WAIT_QUEUE_HEAD(resproc_queue);

MODULE_AUTHOR("Hidenori Kobayashi");

/*
 * Interrupt Handler
 */
void resproc_intr( int irq, void *dev_id, struct pt_regs *regs)
{
	u8 byte;
	struct Resproc_Dev *resproc = (struct Resproc_Dev *)dev_id;
	unsigned long ioaddr = resproc->base;
	unsigned short head = resproc->mail_head;
	unsigned long *mail = &resproc->mail[head];

	PDEBUG("resproc_intr\n");

	byte = inb(ioaddr+OS_CMD1);

	/* PIS==1 then get mailbox B and clear PIS/PIC */
	if ( byte & CMD1_PIS ) {
		*mail = inl(ioaddr+OS_MB_B);
		resproc->mail_head = ( head == (MAIL_NUM-1) ) ? 0 : head + 1;

		PDEBUG("waking up!\n");
		wake_up_interruptible(&resproc_queue);
		outb( byte|CMD1_PIS, ioaddr+OS_CMD1 );
	}
}

/*
 * Open and close
 */

int resproc_open (struct inode *inode, struct file *filp)
{
	unsigned int irq;
	unsigned long ioaddr;
	unsigned long flags;

    struct Resproc_Dev *resproc; /* device information */


	PDEBUG("resproc_open\n");
    /* and use filp->private_data to point to the device data */
	resproc = filp->private_data;

	/* register irq, if opened for the first time */
	if ( !resproc->usage ){
		u8 b;

		PDEBUG("resproc_open: first time\n");
		irq = resproc->irq;
		ioaddr = resproc->base & PCI_BASE_ADDRESS_IO_MASK;

		/* disable and clear interrupt before requesting irq */
		b = inb( ioaddr+OS_CMD1 );
		outb( b | ~CMD1_PIE | CMD1_PIC, ioaddr+OS_CMD1 );
		save_flags(flags);
		cli();	

		/* request irq */
		if( request_irq( irq, resproc_intr, 
					SA_INTERRUPT | SA_SHIRQ, resproc->name,
					resproc) ){
			printk(KERN_WARNING "Can't get IRQ.\n");
		}

		/* enable pci bus interrupt */
		outb( b|CMD1_PIE, ioaddr+OS_CMD1 );

		/* and initialize buffer index */
		resproc->mail_head = resproc->mail_tail = 0;
		restore_flags(flags);
	}
	resproc->usage++;

    MOD_INC_USE_COUNT;
    PDEBUG("resproc_open returns\n");

    return 0;          /* success */
}

int resproc_release (struct inode *inode, struct file *filp)
{
	struct Resproc_Dev *resproc;

	PDEBUG("resproc_release\n");

	resproc = filp->private_data;

	/* free irq, if closed for the last time */
	resproc->usage--;
	if ( !resproc->usage )
		free_irq(resproc->irq, resproc);

    MOD_DEC_USE_COUNT;
    return (0);
}

/*
 * Data management: read and write
 */

int send_mail( struct Resproc_Dev *resproc, unsigned long x )
{
	int timer = TIMER_COUNT;
	unsigned long ioaddr = resproc->base;

	/* wait for LIS down */
	/*
	 * FIXME: this is really stupid.
	 * "up" is need to avoid reading unsettled flag.
	 * 		may be caused by SPARClite reading the mailbox.
	 * 	Using ID1 may help.
	 */
up:
	while ( timer > 0 ){
		if ( (inb(ioaddr+OS_CMD1) & CMD1_LIS) == 0 )
			break;
		timer--;
	}

	if ( timer <= 0 )
		return -ETIMEDOUT;

	if ( (inb(ioaddr+OS_CMD1) & CMD1_LIS) != 0 )
		goto up;

	PDEBUG("send_mail: after OS_CMD1 = (%x)\n", 
			inb(ioaddr+OS_CMD1) & CMD1_LIS);

	/* write to MailBox A */
	outl( x, ioaddr+OS_MB_A );

	PDEBUG("send_mail: x = %lx\n", x);

	return 0;	/* succeed */
}

int recv_mail( struct Resproc_Dev *resproc, unsigned long *x )
{
	unsigned short tail;
	unsigned short head;
	unsigned long flags;

	PDEBUG("resproc_recv_mail.\n");

#if 1
	save_flags(flags);
	cli();
#endif
	head = resproc->mail_head;
	tail = resproc->mail_tail;
	PDEBUG("head = %d, tail = %d\n", head, tail);

	if ( head == tail){
		interruptible_sleep_on(&resproc_queue);
		PDEBUG("woke up\n");
	}
	PDEBUG("broken?\n");
	*x = resproc->mail[tail];

	PDEBUG("recv_mail: *x = %lx\n", *x);

	resproc->mail_tail = (tail == (MAIL_NUM-1)) ? 0 : tail +  1;
#if 1
	restore_flags(flags);
#endif

	return 0;	/* succeed */
}

int wait_ID0_clr( struct Resproc_Dev *resproc )
{
	int timer = TIMER_COUNT;
	unsigned long ioaddr = resproc->base;

id0again:
	while( (inb(ioaddr+OS_CMD1) & CMD1_ID0) && (timer > 0) )
		timer--;

	if( timer <= 0 )
		return -1;	/* fail */
	else{
		if (inb(ioaddr+OS_CMD1) & CMD1_ID0)
			goto id0again;
		return 0;	/* succeed */
	}
}
	
int wait_ID1_clr( struct Resproc_Dev *resproc )
{
	int timer = TIMER_COUNT;
	unsigned long ioaddr = resproc->base;

	while( (inb(ioaddr+OS_CMD1) & CMD1_ID1) && (timer > 0) )
		timer--;

	if( timer <= 0 )
		return -1;	/* fail */
	else
		return 0;	/* succeed */
}
			
ssize_t resproc_read (struct file *filp, char *buf, size_t count,
                loff_t *f_pos)
{
    struct Resproc_Dev *resproc;
	unsigned long ioaddr;
	unsigned long *data;

	int res;
	unsigned int size;
	unsigned int mailsize;
	unsigned long m;
	unsigned long maildata[3];


	resproc = filp->private_data; 
	ioaddr = resproc->base;
	data = (unsigned long*)resproc->Rdma_data;


	if (down_interruptible (&resproc->sem))
		return (-ERESTARTSYS);

	//outb( CMD2_TAIE | CMD2_MAIE, ioaddr+OS_CMD2 );

	/*
	 * interactively setup a connection
	 */
	/* possible to read data? */
	res = send_mail( resproc, C_S2P );
	if ( res < 0 )
		goto timeout;

	/* wait for ack */
	recv_mail( resproc, &m );

	/* tell the board where the buffer is */
	res = send_mail( resproc, virt_to_bus(data));
	if ( res < 0 )
		goto timeout;

	/* determine the size ( in byte )*/
	if ( count <= PAGE_SIZE ){
		size = count;
	} else {
		size = PAGE_SIZE;
	}

	/* tell the board about the size */
	res = send_mail( resproc, size );
	if ( res < 0 )
		goto timeout;

	/*
	 * small data are faster via mailbox
	 */
	if ( (mailsize = size % 0x10) > 0 ){
		int i;

#if 0
		/* fix size */
		if ( size > 0x10 )
			mailsize = size % 0x4;
#endif

		for ( i = 0; i < mailsize; i+= 4){
			recv_mail( resproc, &maildata[i>>2] );
		}

		do{
			int j;
			for ( j = 0; j < mailsize; j+=4 )
				PDEBUG("resproc_read: maildata[%d] = %lx\n", 
						j>>2, maildata[j>>2] );
		}while(0);
	}

	/* 
	 * wait for transfer completion. waiting for dma to complete
	 */
	res = wait_ID0_clr( resproc );
	if ( res < 0 )
		goto timeout;

	do{
		int i;
		for( i = 0; i < (size>>2) && i < 0x40; i ++)
			PDEBUG("resproc_read: data[%d] = %lx\n",i, 
					*(volatile unsigned long*)(data+i));
	}while(0);

	/* copy kernel-space buffer to user-space buffer */
    res = copy_to_user (buf, data, size-mailsize);
	PDEBUG("to user[dma]: %dB, res = %d\n", size-mailsize, res);
	if ( mailsize > 0 )
    	res = copy_to_user (buf+size-mailsize, maildata, mailsize);
	PDEBUG("to user[mail]: %dB, res = %d\n", mailsize, res);

    up (&resproc->sem);
    return size;

timeout:
    up (&resproc->sem);
    return (-ETIMEDOUT);
}

ssize_t resproc_write (struct file *filp, const char *buf, size_t count,
                loff_t *f_pos)
{
    struct Resproc_Dev *resproc;
	unsigned long ioaddr;
	unsigned long *data;

	int res;
	unsigned long m;
	unsigned int size;
	unsigned int mailsize;

	PDEBUG("resproc_write.\n");

	resproc = filp->private_data;
	ioaddr = resproc->base;
	data = (unsigned long *)resproc->Wdma_data;

	/* determine the size ( in byte )*/
	if ( count <= PAGE_SIZE ){
		size = count;
	} else {
		size = PAGE_SIZE;
	}

    if (down_interruptible (&resproc->sem))
            return (-ERESTARTSYS);

	PDEBUG("resproc_write : trying to send %x\n", buf[0]);
	PDEBUG("resproc_write : size %x\n", count);

	//outb( CMD2_TAIE | CMD2_MAIE, ioaddr+OS_CMD2 );

	/*
	 * interactively setup a connection
	 */
	/* possible to write data? */
	res = send_mail( resproc, C_P2S );
	if ( res < 0 )
		goto timeout;

	/* wait for ack */
	recv_mail( resproc, &m );

	/* copy user-space buffer to kernel-space buffer */
    copy_from_user (data, buf, size);

	/* tell the board where the buffer is */
	res = send_mail( resproc, virt_to_bus(data));
	if ( res < 0 )
		goto timeout;

	/* tell the board about the size */
	res = send_mail( resproc, size );
	if ( res < 0 )
		goto timeout;

	/* wait a bit to send data via mailbox (only here) */
	res = wait_ID1_clr( resproc );
	if ( res < 0 )
		goto timeout;

	/*
	 * small data are faster via mailbox
	 */
	if ( (mailsize = size % 0x10) > 0 ) {
		int i;
		unsigned long *p;

#if 0
		/* fix size */
		if ( size > 0x10 )
			mailsize = size % 0x4;
#endif

		//p = data + size - mailsize;
		p = data + ((size - mailsize)>>2);

		do{
			int j;
			for ( j = 0; j < mailsize; j+=4 )
				PDEBUG("resproc_write: mail_data = %lx\n", 
						*(unsigned long *)(p+(j>>2)));
		}while(0);

		for ( i = 0; i < mailsize; i+=4 ){
			send_mail( resproc, *(unsigned long *)(p+(i>>2)));
		}
	}

	/* 
	 * wait for transfer completion. waiting for dma to complete
	 */
	res = wait_ID0_clr( resproc );
	if ( res < 0 )
		goto timeout;

    up (&resproc->sem);
    return size;

timeout:
    up (&resproc->sem);
    return (-ETIMEDOUT);
}

/*
 * The different file operations
 */

struct file_operations resproc_fops = {
    read:       resproc_read,
    write:      resproc_write,
    open:       resproc_open,
    release:    resproc_release,
};
/*
 * Finally, the module stuff
 */
static int __devinit resproc_probe( struct pci_dev *dev, 
		const struct pci_device_id *id )
{
	struct Resproc_Dev *resproc;
	int result;
	static int resproc_sn = 0;
	char resproc_name[1];

	/* 
	 * allocate the devices -- we can't have them static, as the number
	 * can be specified at load time
	 */
	resproc = kmalloc( sizeof (struct Resproc_Dev), GFP_KERNEL);
	if (!resproc) { return -ENOMEM; }
	dev->driver_data = resproc;

	memset(resproc, 0, sizeof (struct Resproc_Dev));
	sema_init (&resproc->sem, 1);

	result = pci_enable_device(dev);
	if (result)
		return result;

	/* configure Responsive Processor (PCI) Evaluation Board  */
	sprintf(resproc_name, "%i", resproc_sn);
	sprintf(resproc->name, "%i", resproc_sn);

#if 0
	pci_write_config_dword(dev, PCI_COMMAND, 
			PCI_COMMAND_SERR | PCI_COMMAND_PARITY 
			| PCI_COMMAND_MASTER | PCI_COMMAND_MEMORY | PCI_COMMAND_IO);
#else
	pci_write_config_dword(dev, PCI_COMMAND,
			PCI_COMMAND_SERR | PCI_COMMAND_PARITY | PCI_COMMAND_IO );
#endif

	pci_set_master( dev );

	/* record the address of the port */
	resproc->base = pci_resource_start(dev, 0);

	/* get irqline ( Resposive Processor is a PnP board ) */
	resproc->irq = dev->irq;

	/* check, request I/O port */
	if ( !resproc->base || 
			(pci_resource_flags(dev, 0) & IORESOURCE_IO) == 0 ){
		PDEBUG("no I/O resource found at PCI BAR #0?\n");
		return -ENODEV;
	}

	if ( request_region(resproc->base, pci_resource_len(dev, 0), resproc->name) 
			== NULL ){
		PDEBUG("I/O resouce 0x%lx @ 0x%lx busy.\n",
				pci_resource_len(dev, 0), resproc->base);
		return -EBUSY;
	}

	//
	//pci_write_config_dword(dev, 0x0c, 0x00004000);

	/* clear master abort status */
	pci_write_config_word( dev, PCI_STATUS, PCI_STATUS_REC_MASTER_ABORT );

	/* allocate a page for the dma transfer buffer */
	if ( !(resproc->Rdma_data = get_free_page(GFP_KERNEL)) ){
		goto Rfail;
	}
	if ( !(resproc->Wdma_data = get_free_page(GFP_KERNEL)) ){
		goto Wfail;
	}

	resproc->handle = devfs_register(resproc_devfs_dir, resproc_name,
			DEVFS_FL_AUTO_DEVNUM, 0, 0,
			S_IFCHR | S_IRUGO | S_IWUGO, 
			&resproc_fops, resproc);

	resproc_sn++;


	return 0;	/* succeed */

Wfail:
	free_page(resproc->Rdma_data);
Rfail:
	printk(KERN_INFO "resproc.o: NO more memory for dma buffer");

	free_page(resproc->Rdma_data);
	free_page(resproc->Wdma_data);

	resproc->base = 0;
	return -ENOMEM;
}

static void __devinit resproc_remove( struct pci_dev *dev )
{
	struct Resproc_Dev *resproc;

	PDEBUG("resproc remove\n");
	
	resproc = pci_get_drvdata(dev);
	if ( resproc->usage )
		return;

	release_region(resproc->base, pci_resource_len(dev, 0));

	free_page( resproc->Rdma_data );
	free_page( resproc->Wdma_data );
    	kfree(resproc);

    	devfs_unregister(resproc->handle);

	pci_set_drvdata(dev, NULL);

	PDEBUG("module resproc cleaned\n");
}

#define	RESPROC_VER4	0
static struct pci_device_id resproc_id_tbl[] __devinitdata = {
	{RESPROC_VENDOR_ID, RESPROC_DEVICE_ID, PCI_ANY_ID, PCI_ANY_ID, 0, 0, RESPROC_VER4},
	{0, }
};

static struct pci_driver resproc_driver = {
	name:	"Responsive Processor version 4",
	id_table:	resproc_id_tbl,
	probe:	resproc_probe,
	remove:	resproc_remove,
};

MODULE_DEVICE_TABLE(pci, resproc_id_tbl);

int __init resproc_init( void )
{

	resproc_devfs_dir = devfs_mk_dir(NULL, "resproc", NULL);
	if (!resproc_devfs_dir) return -EBUSY;

	return pci_module_init(&resproc_driver);
}

void __exit resproc_exit( void )
{
	pci_unregister_driver(&resproc_driver);
}

module_init(resproc_init);
module_exit(resproc_exit);
