#include <stdio.h>
#include <string.h>
#include <elf.h>
#include <stdlib.h>

#define BYTE_GET(field)  byte_get_big_endian(field, sizeof(field))

typedef struct {
	unsigned int offset;
	unsigned int size;
	unsigned int addr;
} segment;

/* ELF Header (32-bit implementations) */
typedef struct {
  unsigned char e_ident[16];            /* ELF "magic number" */
  unsigned char e_type[2];              /* Identifies object file type */
  unsigned char e_machine[2];           /* Specifies required architecture */
  unsigned char e_version[4];           /* Identifies object file version */
  unsigned char e_entry[4];             /* Entry point virtual address */
  unsigned char e_phoff[4];             /* Program header table file offset */
  unsigned char e_shoff[4];             /* Section header table file offset */
  unsigned char e_flags[4];             /* Processor-specific flags */
  unsigned char e_ehsize[2];            /* ELF header size in bytes */
  unsigned char e_phentsize[2];         /* Program header table entry size */
  unsigned char e_phnum[2];             /* Program header table entry count */
  unsigned char e_shentsize[2];         /* Section header table entry size */
  unsigned char e_shnum[2];             /* Section header table entry count */
  unsigned char e_shstrndx[2];          /* Section header string table index */
} Elf32_External_Ehdr;


/* Program header */
typedef struct {
  unsigned char p_type[4];		/* Type of segment */
  unsigned char p_offset[4];	/* Offset of segment entry */
  unsigned char p_vaddr[4];		/* Virtual address of segment entry */
  unsigned char p_paddr[4];		/* Physical address of segment */
  unsigned char p_filesz[4];	/* Size of file image*/
  unsigned char p_memsz[4];		/* Size of memory image */
  unsigned char p_flags[4];		/* Miscellaneous segment attribute */
  unsigned char p_align[4];		/* Segment alignment */
} Elf32_External_Phdr;

int byte_get_big_endian (unsigned char* field, int size)
{
	switch (size) {
    case 1:
		return *field;

    case 2:
		return ((unsigned int) (field[1])) | (((int) (field[0])) << 8);

    case 4:
		return ((unsigned long) (field[3]))
			|   (((unsigned long) (field[2])) << 8)
			|   (((unsigned long) (field[1])) << 16)
			|   (((unsigned long) (field[0])) << 24);

    case 8:
		/* Although we are extracing data from an 8 byte wide field, we
		   are returning only 4 bytes of data.  */
		return ((unsigned long) (field[7]))
			|   (((unsigned long) (field[6])) << 8)
			|   (((unsigned long) (field[5])) << 16)
			|   (((unsigned long) (field[4])) << 24);

    default:
		printf ("Unhandled data length: %d\n", size);
		abort ();
    }

	return 0;
}


void show_usage(void) {
	printf("Usage: mips-rmt-elf-obj2prg SOURCE\n");
	printf("SOURCE is a elf format binary file.\n");
	printf("\n");
}

int main(int argc, char** argv)
{
	Elf32_Ehdr eh;
	Elf32_Phdr *ph = NULL;
	Elf32_Phdr *phdrs = NULL;
	Elf32_External_Ehdr xeh;
	Elf32_External_Phdr *xphdrs = NULL;
	segment *load_segments = NULL;
	unsigned int i, j, k;
	char *binary = NULL;
	FILE *fbin = NULL;
	unsigned int word;
	unsigned char big_word[4];
	int load_seg_num = 0;
	int lines, tmp;

	if (argc < 2) {
		printf("too few arguments.\n\n");
		show_usage();
		goto end;
	}
	else if(argc > 2) {
		printf("too many arguments.\n\n");
		show_usage();
		goto end;
	}

	binary = argv[1];

	/* open binary file. */
	if (!(fbin = fopen(binary, "rb"))) {
		printf("Failed to open %s\n", binary);
		goto end;
	}

	/* read elf header. */
	if (!(fread(&xeh, sizeof(xeh), 1, fbin))) {
		printf("Failed to read %s\n", binary);
		goto end;
	}

	/* transform to little endian byte */
	eh.e_type      = BYTE_GET (xeh.e_type);
	eh.e_machine   = BYTE_GET (xeh.e_machine);
	eh.e_version   = BYTE_GET (xeh.e_version);
	eh.e_entry     = BYTE_GET (xeh.e_entry);
	eh.e_phoff     = BYTE_GET (xeh.e_phoff);
	eh.e_shoff     = BYTE_GET (xeh.e_shoff);
	eh.e_flags     = BYTE_GET (xeh.e_flags);
	eh.e_ehsize    = BYTE_GET (xeh.e_ehsize);
	eh.e_phentsize = BYTE_GET (xeh.e_phentsize);
	eh.e_phnum     = BYTE_GET (xeh.e_phnum);
	eh.e_shentsize = BYTE_GET (xeh.e_shentsize);
	eh.e_shnum     = BYTE_GET (xeh.e_shnum);
	eh.e_shstrndx  = BYTE_GET (xeh.e_shstrndx);

	/* The head message */
	printf("//\n");
	printf("// Generated by mips-rmt-elf-obj2prg tool.\n");
	printf("// The output format is readable by RMT verilog chip.sh.\n");
	printf("//\n");
	printf("// Source file: %s\n", binary);
	/* print the entry point. */
	printf("// Entry point: 0x%08x\n", eh.e_entry);
	printf("//\n");

	/* allocate for external program headers */
	xphdrs = (Elf32_External_Phdr *) 
		malloc (eh.e_phentsize * eh.e_phnum);

	/* seek file. */
	if (fseek(fbin, eh.e_phoff, SEEK_SET)) {
		printf("Failed to seek %s\n", binary);
		goto end;
	}

	/* read program headers. */
	if (!(fread(xphdrs, eh.e_phentsize * eh.e_phnum, 1, fbin))) {
		printf("Failed to read %s\n", binary);
		goto end;
	}

	/* allocate for program headers. */
	phdrs = (Elf32_Phdr *) 
		malloc (eh.e_phnum * sizeof (Elf32_Phdr));

	/* transfer to the little endian */
	for (i = 0, ph = phdrs; i < eh.e_phnum; i++, ph++) {
		ph->p_type		= BYTE_GET (xphdrs[i].p_type);
		ph->p_offset	= BYTE_GET (xphdrs[i].p_offset);
		ph->p_vaddr		= BYTE_GET (xphdrs[i].p_vaddr);
		ph->p_paddr		= BYTE_GET (xphdrs[i].p_paddr);
		ph->p_filesz	= BYTE_GET (xphdrs[i].p_filesz);
		ph->p_memsz		= BYTE_GET (xphdrs[i].p_memsz);
		ph->p_flags		= BYTE_GET (xphdrs[i].p_flags);
		ph->p_align		= BYTE_GET (xphdrs[i].p_align);
    }

	for (i = 0, ph = phdrs; i < eh.e_phnum; i++, ph++) {
		if (PT_LOAD & ph->p_type)
			load_seg_num++;
	}

	load_segments = (segment*) malloc(sizeof(segment) * load_seg_num);
	j = 0;
	for (i = 0, ph = phdrs; i < eh.e_phnum; i++, ph++) {
		if (PT_LOAD & ph->p_type) {
			load_segments[j].offset = ph->p_offset;
			load_segments[j].size = ph->p_filesz;
			load_segments[j].addr = ph->p_paddr;
			j++;
		} 
	}

#if 0
	for (i = 0; i < load_seg_num; i++) {
		printf("// segment address: %08x\n", load_segments[i].addr);
		printf("// segment offset: 0x%x\n", load_segments[i].offset);
		printf("// segment size: %d bytes\n", load_segments[i].size);
		printf("\n");
	}
#endif

	/* seek to the begining. */
	if (fseek(fbin, 0, SEEK_SET)) {
		printf("Failed to seek %s\n", binary);
		goto end;
	}

	for (i = 0; i < load_seg_num; i++) {
		/* seek to the offset of segment */
		fseek(fbin, load_segments[i].offset, SEEK_SET);
		if (load_segments[i].size % sizeof(big_word) != 0) {
			lines = load_segments[i].size / sizeof(big_word) + 1; 
//			printf("// Not aligned\n");
		}
		else
			lines = load_segments[i].size / sizeof(big_word);

		for (j = 0; j < lines; j++) {
			/* read word as the format of big endian */
			fread(big_word, sizeof(big_word), 1, fbin);
			/* trasform to little endian from big endian */
			word = BYTE_GET(big_word);
			printf("%08x // %08x\n", 
				   word, load_segments[i].addr + j * sizeof(big_word));
		}
		if (i < load_seg_num - 1) {
			tmp = load_segments[i+1].addr - 
				(load_segments[i].addr + j * sizeof(big_word) + 1);
			if (tmp % sizeof(big_word) != 0)
				lines = tmp / sizeof(big_word) + 1;
			else
				lines = tmp / sizeof(big_word);
			for (k = 0; k < lines; k++) {
				printf("ffffffff // %08x NO DATA\n", 
					   load_segments[i].addr + (j + k) * sizeof(big_word));
			}
		}
	}

 end:
	/* free memory */
	if (xphdrs)
		free(xphdrs);

	if (phdrs)
		free(phdrs);

	if (load_segments)
		free(load_segments);

	/* close file stream. */
	if (fbin)
		fclose(fbin);

	return 0;
}
