/*
 * Setup pointers to hardware dependent routines.
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file "COPYING" in the main directory of this archive
 * for more details.
 *
 * Copyright (C) 1996, 1997 by Ralf Baechle
 * Copyright (C) 2001, 2002, 2003 by Liam Davies (ldavies@agile.tv)
 *
 */
#include <linux/config.h>
#include <linux/interrupt.h>
#include <linux/pci.h>
#include <linux/init.h>

#include <asm/bootinfo.h>
#include <asm/pci_channel.h>
#include <asm/time.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/processor.h>
#include <asm/reboot.h>
#include <asm/gt64120.h>
#include <asm/rmt.h>

#include <asm/ny0002/ny0002.h>
#include <asm/thread_control.h>
#include <asm/dbgout.h>

extern void rmtsim_machine_restart(char *command);
extern void rmtsim_machine_halt(void);
extern void rmtsim_machine_power_off(void);

int rmtsim_board_id;

static char my_cmdline[CL_SIZE] = {
 "console=ttyS0,38400 "
#ifdef CONFIG_IP_PNP
 "ip=on "
#endif
#ifdef CONFIG_ROOT_NFS
 "root=/dev/nfs "
#endif
 "rw "
 };

const char *get_system_type(void)
{
	return "RMT";
}

void __init rmtsim_timer_setup(void)
{
#ifdef CONFIG_RMTTIMER_NOUSE
	if (smp_processor_id() == 0)
		time_poll_start_now();
	set_c0_status(ST0_IM_TIMER);		/* Mask TimerInturrption */
#else	/* !CONFIG_RMTTIMER_NOUSE */
#ifdef CONFIG_RMT_DEBUG_TIMER_INTERRUPT_RATE_MULTIPLY
	rmt_str_printf("rmtsim_timer_setup() LATCH=");
	rmt_print_hex(LATCH * CONFIG_RMT_DEBUG_TIMER_INTERRUPT_RATE_MULTIPLY_N);
	rmt_str_printf("(");
	rmt_print_hex(LATCH);
	rmt_str_printf(")\n");
#endif

	/* Register our timer interrupt */
#ifdef CONFIG_RMT_DEBUG_TIMER_INTERRUPT_RATE_MULTIPLY
	write_c0_compare(LATCH  * CONFIG_RMT_DEBUG_TIMER_INTERRUPT_RATE_MULTIPLY_N);
#else
	write_c0_compare(LATCH);
#endif
#if defined(CONFIG_RMT_RTTHREAD)
	if (rmt_rtthread(smp_processor_id()))
	    set_c0_status(ST0_IM_TIMER);	/* Mask TimerInturrption */
	else
	    clear_c0_status(ST0_IM_TIMER);	/* Unmask TimerInturrption */
#else	
	clear_c0_status(ST0_IM_TIMER);		/* Unmask TimerInturrption */
#endif
#endif	/* !CONFIG_RMTTIMER_NOUSE */
#if defined(CONFIG_RMT_COUNT_TICK) && !defined(CONFIG_RMTTIMER_NOUSE)
	if (smp_processor_id() == 0) {
		extern void rmt_internal_jiffies_init(void);
		rmt_internal_jiffies_init();
	}
#endif
	set_c0_status(ST0_PE|ST0_TS);
}

extern struct pci_ops gt64111_pci_ops;

static struct resource rmtsim_mem_resource = {
	"GT64111 PCI MEM", GT64111_IO_BASE, 0xffffffffUL, IORESOURCE_MEM
};

static struct resource rmtsim_io_resource = {
	"GT64111 IO MEM", 0x00001000UL, 0x0fffffffUL, IORESOURCE_IO
};

static struct resource rmtsim_io_resources[] = {
	{ "dma1", 0x00, 0x1f, IORESOURCE_BUSY },
	{ "timer", 0x40, 0x5f, IORESOURCE_BUSY },
	{ "keyboard", 0x60, 0x6f, IORESOURCE_BUSY },
	{ "dma page reg", 0x80, 0x8f, IORESOURCE_BUSY },
	{ "dma2", 0xc0, 0xdf, IORESOURCE_BUSY },
};

#define NY0002_IO_RESOURCES (sizeof(rmtsim_io_resources)/sizeof(struct resource))

#ifdef CONFIG_PCI
static struct pci_controller rmtsim_pci_controller = {
	.pci_ops	= &gt64111_pci_ops,
	.mem_resource	= &rmtsim_mem_resource,
	.mem_offset	= 0,
	.io_resource	= &rmtsim_io_resource,
	.io_offset	= 0x00001000UL - GT64111_IO_BASE
};
#endif

static void __init rmtsim_setup(void)
{
	unsigned int devfn = PCI_DEVFN(NY0002_PCICONF_VIA, 0);
	int i;

	rmt_str_printf("rmtsim_setup() #1\n");
	_machine_restart = rmtsim_machine_restart;
	_machine_halt = rmtsim_machine_halt;
	_machine_power_off = rmtsim_machine_power_off;

	board_timer_setup = rmtsim_timer_setup;

	add_memory_region(ROM_BASE, ROM_SIZE, BOOT_MEM_ROM_DATA);
	add_memory_region(RAM_BASE, RAM_SIZE, BOOT_MEM_RAM);

#ifdef CONFIG_EARLY_PRINTK
	/*
	 * This is a prom style console. We just poke at the
	 *  UART to make it talk.
	 * Only use this console if you really screw up and can't
	 *  get to the stage of setting up a real serial console.
	 */
	rmtsim_setup_console();
#endif

#ifdef CONFIG_PCI
	register_pci_controller(&rmtsim_pci_controller);
#endif
}
early_initcall(rmtsim_setup);

/*
 * Prom init. We read our one and only communication with the firmware.
 * Grab the amount of installed memory
 */
void __init prom_init(void)
{
}

unsigned long __init prom_free_prom_memory(void)
{
	/* Nothing to do! */
	return 0;
}

void disable_early_printk(void)
{
	rmtsim_disable_console();
}
