/*
  This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file "COPYING" in the main directory of this archive
 * for more details.
 *
 * Copyright (C) 1994 - 1999, 2000, 01 Ralf Baechle
 * Copyright (C) 1995, 1996 Paul M. Antoine
 * Copyright (C) 1998 Ulf Carlsson
 * Copyright (C) 1999 Silicon Graphics, Inc.
 * Kevin D. Kissell, kevink@mips.com and Carsten Langgaard, carstenl@mips.com
 * Copyright (C) 2000, 01 MIPS Technologies, Inc.
 * Copyright (C) 2002, 2003  Maciej W. Rozycki
 */
#include <linux/config.h>
#include <linux/init.h>
#include <linux/mm.h>
#include <linux/module.h>
#include <linux/sched.h>
#include <linux/smp.h>
#include <linux/smp_lock.h>
#include <linux/spinlock.h>
#include <linux/interrupt.h>
#include <linux/kallsyms.h>

#include <asm/bootinfo.h>
#include <asm/branch.h>
#include <asm/cpu.h>
#include <asm/fpu.h>
#include <asm/module.h>
#include <asm/pgtable.h>
#include <asm/ptrace.h>
#include <asm/sections.h>
#include <asm/system.h>
#include <asm/tlbdebug.h>
#include <asm/traps.h>
#include <asm/uaccess.h>
#include <asm/mmu_context.h>
#include <asm/watch.h>
#include <asm/types.h>
#include <asm/rmtregs.h>
#include <asm/thread_control.h>
#include <asm/sharereg_lock.h>

#include <asm/dbgout.h>

#ifdef CONFIG_RMT_TLBFLUSH_UACCESS
#include <asm/page-32.h>
#endif

#undef CHECK_DELAY_SLOT_INST

extern asmlinkage void except_vec_itlb_spraddr_miss(void);
extern asmlinkage void except_vec_itlb_allentry_locked(void);
extern asmlinkage void except_vec_itlb_noentry_matched(void);
extern asmlinkage void except_vec_itlb_threadmode_err(void);
extern asmlinkage void except_vec_itlb_protection_err(void);
extern asmlinkage void except_vec_dtlb_spraddr_miss(void);
extern asmlinkage void except_vec_dtlb_allentry_locked(void);
extern asmlinkage void except_vec_dtlb_noentry_matched(void);
extern asmlinkage void except_vec_dtlb_threadmode_err(void);
extern asmlinkage void except_vec_dtlb_protection_err(void);
extern asmlinkage void except_vec_coproc_unusable(void);
extern asmlinkage void except_vec_invalid_instruction(void);
extern asmlinkage void except_vec_system_call(void);
extern asmlinkage void except_vec_break_point(void);
extern asmlinkage void except_vec_int_overflow(void);
extern asmlinkage void except_vec_divide_byzero(void);
extern asmlinkage void except_vec_trap(void);
extern asmlinkage void except_vec_load_dataaddr_missalign(void);
extern asmlinkage void except_vec_store_dataaddr_missalign(void);
extern asmlinkage void except_vec_float_overflow(void);
extern asmlinkage void except_vec_float_underflow(void);
extern asmlinkage void except_vec_float_divide_byzero(void);
extern asmlinkage void except_vec_float_inexact(void);
extern asmlinkage void except_vec_float_invalid_operation(void);
extern asmlinkage void except_vec_reserved(void);
extern asmlinkage void except_vec_vectorint_exp(void);
extern asmlinkage void except_vec_vectorfloat_exp(void);
extern asmlinkage void except_vec_timer_interruption(void);
extern asmlinkage void except_vec_hardware_interruption(void);
extern asmlinkage void except_vec_software_interruption1(void);
extern asmlinkage void except_vec_software_interruption0(void);

extern int fpu_emulator_cop1Handler(int xcptno, struct pt_regs *xcp,
	struct mips_fpu_soft_struct *ctx);

#ifdef CONFIG_RMT_TLBFLUSH_UACCESS        
atomic_t flag_wptlb_exist;
#endif

void (*board_be_init)(void);
int (*board_be_handler)(struct pt_regs *regs, int is_fixup);

/*
 * These constant is for searching for possible module text segments.
 * MODULE_RANGE is a guess of how much space is likely to be vmalloced.
 */
#define MODULE_RANGE (8*1024*1024)

/*
 * This routine abuses get_user()/put_user() to reference pointers
 * with at least a bit of error checking ...
 */
void show_stack(struct task_struct *task, unsigned long *sp)
{
	const int field = 2 * sizeof(unsigned long);
	long stackdata;
	int i;

	sp = sp ? sp : (unsigned long *) &sp;

	printk("Stack :");
	i = 0;
	while ((unsigned long) sp & (PAGE_SIZE - 1)) {
		if (i && ((i % (64 / field)) == 0))
			printk("\n       ");
		if (i > 39) {
			printk(" ...");
			break;
		}

		if (__get_user(stackdata, sp++)) {
			printk(" (Bad stack address)");
			break;
		}

		printk(" %0*lx", field, stackdata);
		i++;
	}
	printk("\n");
}

void show_trace(struct task_struct *task, unsigned long *stack)
{
	const int field = 2 * sizeof(unsigned long);
	unsigned long addr;

	if (!stack)
		stack = (unsigned long*)&stack;


#ifdef CONFIG_AXE_DEBUG
	printk("show_trace() %d: pid=0x%x cmd=%s\n",
		smp_processor_id(),  task->pid, task->comm);
#endif

	printk("Call Trace:");
#ifdef CONFIG_KALLSYMS
	printk("\n");
#endif
	while (!kstack_end(stack)) {
		addr = *stack++;
		if (kernel_text_address(addr)) {
			printk(" [0x%x <%0*lx>] ", stack, field, addr);
			print_symbol("%s\n", addr);
		}
	}
	printk("\n");
}

void show_trace_task(struct task_struct *tsk)
{
	show_trace(tsk, (long *)tsk->thread.reg29);
}

/*
 * The architecture-independent dump_stack generator
 */
void dump_stack(void)
{
	unsigned long stack;

	show_trace(current, &stack);
}

EXPORT_SYMBOL(dump_stack);

void show_code(unsigned int *pc)
{
	long i;

	printk("\nCode:");

	for(i = -3 ; i < 6 ; i++) {
		unsigned int insn;
		if (__get_user(insn, pc + i)) {
			printk(" (Bad address in epc)\n");
			break;
		}
		printk("%c%08x%c", (i?' ':'<'), insn, (i?' ':'>'));
	}
}

static void print_cp0status(unsigned cp0status)
{
	char *msg;

	switch (cp0status & ST0_KSU) {
	case KSU_USER:
		msg = " USER";
		break;
	case KSU_SUPERVISOR:
		msg = " SUPERVISOR";
		break;
	case KSU_KERNEL:
		msg = " KERNEL";
		break;
	default:
		msg = " ?";
	}

	printk("cp0status: %08x%s%s%s IM(%s%s%s%s)%s%s%s\n",
	       cp0status,
	       (cp0status & ST0_TS) ? " TS" : "",
	       (cp0status & ST0_PE) ? " PE" : "",
	       (cp0status & ST0_EV) ? " EV" : "",
	       (cp0status & ST0_IM_TIMER) ? " TIMER" : "",
	       (cp0status & ST0_IM_HW) ? " HW" : "",
	       (cp0status & ST0_IM_SW1) ? " SW1" : "",
	       (cp0status & ST0_IM_SW2) ? " SW2" : "",
	       (cp0status & ST0_EB) ? " EB" : "",
	       msg,
	       (cp0status & ST0_IE) ? " IE" : "",
	       (cp0status & ST0_EL) ? " EL" : "");
}

static void print_cp0cause(unsigned cp0cause)
{
	printk("cp0cause: %08x %s HIRL=%02x%s%s%s%s CODE=%02x\n",
	       cp0cause, 
	       (cp0cause & 0x80000000) ? " D" : "",
	       (cp0cause >> 12) & 0x1f,
	       (cp0cause & CAUSE_PND_TI) ? " TI" : "",
	       (cp0cause & CAUSE_PND_HI) ? " HI" : "",
	       (cp0cause & CAUSE_PND_S1) ? " S1" : "",
	       (cp0cause & CAUSE_PND_S0) ? " S0" : "",
	       (cp0cause >> 2) & 0x1f);
}

void show_regs(struct pt_regs *regs)
{
	const int field = 2 * sizeof(unsigned long);
	unsigned int cause = regs->cp0_cause;
	int i;

	printk("Cpu %d\n", smp_processor_id());

#ifdef NEVER
	/*
	 * Saved main processor registers
	 */
	for (i = 0; i < 32; ) {
		if ((i % 4) == 0)
			printk("$%2d   :", i);
		if (i == 0)
			printk(" %0*lx", field, 0UL);
		else if (i == 26 || i == 27)
			printk(" %*s", field, "");
		else
			printk(" %0*lx", field, regs->regs[i]);

		i++;
		if ((i % 4) == 0)
			printk("\n");
	}

	/*
	 * Saved cp0 registers
	 */
	printk("epc   : %0*lx ", field, regs->cp0_epc);
	print_symbol("%s ", regs->cp0_epc);
	printk("    %s\n", print_tainted());
	printk("ra    : %0*lx ", field, regs->regs[31]);
	print_symbol("%s\n", regs->regs[31]);

	printk("Status: %08x    ", (uint32_t) regs->cp0_status);

	switch (regs->cp0_status & ST0_KSU) {
	case KSU_USER:
		printk("USER ");
		break;
	case KSU_SUPERVISOR:
		printk("SUPERVISOR ");
		break;
	case KSU_KERNEL:
		printk("KERNEL ");
		break;
	default:
		printk("BAD_MODE ");
		break;
	}
	if (regs->cp0_status & ST0_EL)
		printk("EL ");
	if (regs->cp0_status & ST0_IE)
		printk("IE ");
	printk("\n");

	printk("Cause : %08x\n", cause);

#ifndef CONFIG_AXE_AVOID
	cause = (cause & CAUSEF_EXCCODE) >> CAUSEB_EXCCODE;
	if (0x1 <= cause && cause <= 0xa)
		printk("ExpAddr : %0*lx\n", field, regs->mmu_spr_exp_addr);
#endif
#else /* !NEVER */
	printk("[%d:%s] ", current->pid, current->comm);
	print_cp0status(regs->cp0_status);
	printk("epc: %08x  ", regs->cp0_epc);
	print_cp0cause(regs->cp0_cause);
	printk("$0: %08x  at: %08x  v0: %08x  v1: %08x\n",
	       regs->regs[0], regs->regs[1], regs->regs[2], regs->regs[3]);

	printk("a0: %08x  a1: %08x  a2: %08x  a3: %08x\n",
	       regs->regs[4], regs->regs[5], regs->regs[6], regs->regs[7]);

	printk("t0: %08x  t1: %08x  t2: %08x  t3: %08x\n",
	       regs->regs[8], regs->regs[9], regs->regs[10], regs->regs[11]);

	printk("t4: %08x  t5: %08x  t6: %08x  t7: %08x\n",
	       regs->regs[12], regs->regs[13], regs->regs[14], regs->regs[15]);

	printk("s0: %08x  s1: %08x  s2: %08x  s3: %08x\n",
	       regs->regs[16], regs->regs[17], regs->regs[18], regs->regs[19]);

	printk("s4: %08x  s5: %08x  s6: %08x  s7: %08x\n",
	       regs->regs[20], regs->regs[21], regs->regs[22], regs->regs[23]);

	printk("t8: %08x  t9: %08x  k0: %08x  k1: %08x\n",
	       regs->regs[24], regs->regs[25], regs->regs[26], regs->regs[27]);

	printk("gp: %08x  sp: %08x  fp: %08x  ra: %08x\n",
	       regs->regs[28], regs->regs[29], regs->regs[30], regs->regs[31]);
#endif
}

void show_memdump(unsigned char *base, int len, unsigned offset)
{
        int i;

        for (i=0; i<len; i++) {
                if (i%16 == 0)
                        printk("%08x:", base + offset + i);
                if (i%16 == 15)
			printk(" %02x\n", base[i]);
		else
			printk(" %02x", base[i]);
        }
        if (i%16 != 0)
		printk("\n");
}

void show_registers(struct pt_regs *regs)
{
	show_regs(regs);
	printk("Process %s (pid: %d, threadinfo=%p, task=%p)\n",
	        current->comm, current->pid, current_thread_info(), current);
	show_stack(current, (long *) regs->regs[29]);
	show_trace(current, (long *) regs->regs[29]);
	show_code((unsigned int *) regs->cp0_epc);
	printk("\n");
}

void show_pgd_user(struct mm_struct *mm)
{
	int i, j, k;
	pgd_t *pgd;
	pmd_t *pmd;
	pte_t *pte;
	unsigned long addr_g, addr_m, addr_p, paddr, prot;

	for (i=0; i<USER_PTRS_PER_PGD; i++) {
		addr_g = i << PGDIR_SHIFT;
		pgd = pgd_offset(mm, addr_g);
		if (!pgd_present(*pgd))
			continue;
		for (j=0; j < PTRS_PER_PMD; j++) {
			addr_m = addr_g + (j << PMD_SHIFT);
			pmd = pmd_offset(pgd, addr_m);
			if (!pmd_present(*pmd))
				continue;
			for (k=0; k < PTRS_PER_PTE; k++) {
				addr_p = addr_m + (k << PAGE_SHIFT);
				pte = pte_offset(pmd, addr_p);
				if (!pte_present(*pte))
					continue;
				paddr = pfn_to_phys(pte_pfn(*pte));
				prot = pte_val(*pte) & ~PAGE_MASK;
				printk(" PTE: %08x -> %08x prot:%03x\n",
				       addr_p, paddr, prot);
			}
		}
	}
}

void show_mm(struct mm_struct *mm)
{
	struct vm_area_struct *vma;

	printk("mm=0x%08x\n", mm);
	if (mm == NULL)
		return;
	printk("mm_users=%d mm_count=%d map_count=%d\n",
	       mm->mm_users, mm->mm_count, mm->map_count);
	printk("code:%08x-%08x data:%08x-%08x BRK:%08x-%08x\n",
	       mm->start_code, mm->end_code, mm->start_data, mm->end_data,
	       mm->start_brk, mm->brk);
	printk("stack:%08x ARG:%08x-%08x ENV:%08x-%08x\n",
	       mm->start_stack, mm->arg_start, mm->arg_end,
	       mm->env_start, mm->env_end);
	printk("rss:%08x total_vm:%08x locked_vm:%08x\n",
	       mm->rss, mm->total_vm, mm->locked_vm);

	for (vma = mm->mmap; vma != NULL; vma = vma->vm_next) {
		printk(" %08x-%08x vm_page_prot=%08x vm_flags=%08x\n",
		       vma->vm_start, vma->vm_end,
		       vma->vm_page_prot, vma->vm_flags);
	}
	show_pgd_user(mm);
}

static spinlock_t die_lock = SPIN_LOCK_UNLOCKED;

NORET_TYPE void __die(const char * str, struct pt_regs * regs,
	const char * file, const char * func, unsigned long line)
{
	static int die_counter;

	console_verbose();
	spin_lock_irq(&die_lock);
	printk("%s", str);
	if (file && func)
		printk(" in %s:%s, line %ld", file, func, line);
	printk("[#%d]:\n", ++die_counter);
	show_registers(regs);
	spin_unlock_irq(&die_lock);
	do_exit(SIGSEGV);
}

void __die_if_kernel(const char * str, struct pt_regs * regs,
		     const char * file, const char * func, unsigned long line)
{
	if (!user_mode(regs))
		__die(str, regs, file, func, line);
}

extern const struct exception_table_entry __start___dbe_table[];
extern const struct exception_table_entry __stop___dbe_table[];

void __declare_dbe_table(void)
{
	__asm__ __volatile__(
	".section\t__dbe_table,\"a\"\n\t"
	".previous"
	);
}

#ifdef CONFIG_MDULES

/* Given an address, look for it in the module exception tables. */
const struct exception_table_entry *search_module_dbetables(unsigned long addr)
{
	unsigned long flags;
	const struct exception_table_entry *e = NULL;
	struct module *mod;

	spin_lock_irqsave(&modlist_lock, flags);
	list_for_each_entry(mod, &modules, list) {
		if (mod->arch.num_dbeentries == 0)
			continue;
				
		e = search_extable(mod->arch.dbe_table_start,
				   mod->arch.dbe_table_end +
		                   mod->arch.num_dbeentries - 1,
				   addr);
		if (e)
			break;
	}
	spin_unlock_irqrestore(&modlist_lock, flags);

	/* Now, if we found one, we are running inside it now, hence
           we cannot unload the module, hence no refcnt needed. */
	return e;
}

#else

/* Given an address, look for it in the exception tables. */
static inline const struct exception_table_entry *
search_module_dbetables(unsigned long addr)
{
	return NULL;
}

#endif

/* Given an address, look for it in the exception tables. */
const struct exception_table_entry *search_dbe_tables(unsigned long addr)
{
	const struct exception_table_entry *e;

	e = search_extable(__start___dbe_table, __stop___dbe_table - 1, addr);
	if (!e)
		e = search_module_dbetables(addr);
	return e;
}

/*
 * ll/sc emulation
 */

#define OPCODE 0xfc000000
#define BASE   0x03e00000
#define RT     0x001f0000
#define OFFSET 0x0000ffff
#define LL     0xc0000000
#define SC     0xe0000000

#ifdef CONFIG_RMT
asmlinkage void do_itlb_spraddr_miss(struct pt_regs *regs)
{
	die("ITLB-Addr", regs);
}

asmlinkage void do_itlb_allentry_locked(struct pt_regs *regs)
{
	die("ITLB-AllEntry", regs);
}

asmlinkage void do_itlb_threadmode_err(struct pt_regs *regs)
{
	die("ITLB-Mode", regs);
}

asmlinkage void do_itlb_protection_err(struct pt_regs *regs)
{
	die("ITLB-Pro", regs);
}

asmlinkage void do_dtlb_spraddr_miss(struct pt_regs *regs)
{
	die("DTLB-Addr", regs);
}

asmlinkage void do_dtlb_allentry_locked(struct pt_regs *regs)
{
	die("DTLB-AllEntry", regs);
}

asmlinkage void do_dtlb_threadmode_err(struct pt_regs *regs)
{
	die("DTLB-Mode", regs);
}

/*
 * TLB entry miss and page protection error handler
 */

static pte_t *find_pte_offset(pgd_t *pgd, unsigned long addr)
{
	int offset = __pgd_offset(addr);
	pmd_t *pmd;

	pmd = pmd_offset(pgd + offset, addr);
	return pte_offset(pmd, addr);
}

static void setup_tlb_entry(unsigned va, pte_t pte, int w,
			    unsigned *entry1, unsigned *entry2)
{
	unsigned e1, e2;
	int active_thread_id, group;

	e1 = va & PAGE_MASK;
	e2 = (pte_val(pte) & PAGE_MASK);

	active_thread_id = rmt_mycnum();
	if (va < VMALLOC_START) {
		if (pte_val(pte) & _PAGE_GLOBAL) {
			/* kernel space */
			e1 |= w ? TLB_ENT1_PRO_KERRW : TLB_ENT1_PRO_KERR;
			group =  TLB_GRP_KERNEL;
		} else {
			/* user space */
			e1 |= w ? TLB_ENT1_PRO_USRRW : TLB_ENT1_PRO_USRR;
			group = cpu_fix_context(active_thread_id);
		}
	} else {
		/* vmalloc */
		e1 |= TLB_ENT1_PRO_KERRW;
		group = TLB_GRP_VMALLOC;
	}
	e1 |= ((1 << active_thread_id) << TLB_ENT1_SHR_SHIFT);
	e2 |= (group << TLB_ENT2_GRP_SHIFT) |
		TLB_ENT2_UNC | TLB_ENT2_BRT_32B;

	*entry1 = e1;
	*entry2 = e2;
}

static void invalidate_dtlb_entry(unsigned va, pte_t pte)
{
	int active_thread_id, group;

	active_thread_id = rmt_mycnum();
	if (va < VMALLOC_START) {
		if (pte_val(pte) & _PAGE_GLOBAL) {
			group = (pte_val(pte) & _PAGE_GLOBAL);
		} else {
			group = cpu_fix_context(active_thread_id);
		}
	} else {
		group = TLB_GRP_VMALLOC;
	}
	write_dmmu_register(MMU_SPR_GRP_FLUSH, group);
}

static void load_itlb_entry(unsigned entry1, unsigned entry2)
{
	write_immu_register(MMU_SPR_ENTRY1, entry1);
	write_immu_register(MMU_SPR_ENTRY2, entry2);
	write_immu_register(MMU_SPR_ENTRY_LRU, 0);
}

static void load_dtlb_entry(unsigned entry1, unsigned entry2)
{
	write_dmmu_register(MMU_SPR_ENTRY1, entry1);
	write_dmmu_register(MMU_SPR_ENTRY2, entry2);
	write_dmmu_register(MMU_SPR_ENTRY_LRU, 0);
}

static unsigned load_pc_insn(unsigned va, int *fault)
{
	unsigned pa, insn;
	pte_t *pte;

	rmt_print_val_hex("load_pc_insn", va);
	pte = find_pte_offset((pgd_t *)pgd_current[smp_processor_id()], va);

	if (pte_present(*pte)) {
		pa = (pte_val(*pte) & PAGE_MASK) | (va & ~PAGE_MASK);
		rmt_print_val_hex("PC(user)", pa);
		insn = *(unsigned *)pa;
	} else {
		if (0x80000000UL <= va && va < VMALLOC_START ||
		    VMALLOC_END <= va) {
			rmt_print_val_hex("PC(kernel)", va);
			insn = *(unsigned *)va;
		} else {
			if (fault) {
				*fault = 1;
				return 0;
			} else
				panic("Bad PC. no entry on page table (0x%08x)", va);
		}
	}
	if (fault)
		*fault = 0;

	return insn;
}

#define GET_OPCODE(x)                 ((x) & 0xfc000000)
#define GET_OPCODE_SPECIAL(x)         ((x) & 0xfc00003f)
#define GET_OPCODE_REGIMM(x)          ((x) & 0xfc1f0000)
#define GET_BASE(x)                   (((x) >> 21) & 0x1f)
#define GET_IMMEDIATE(x)              \
 (((x) & 0x00008000)? (x) | 0xffff0000 : (x) & 0x0000ffff )

#define OPCODE_COP0                   0x40000000
#define OPCODE_VINT                   0x78000000
#define OPCODE_VFP                    0x7c000000
#define OPCODE_J                      0x08000000
#define OPCODE_JAL                    0x0c000000
#define OPCODE_SPECIAL                0x00000000
#define OPCODE_REGIMM                 0x04000000

#define OPCODE_SPECIAL_JR            0x00000008
#define OPCODE_SPECIAL_JALR          0x00000009

#ifdef CHECK_DELAY_SLOT_INST
static int have_delay_slot(unsigned insn)
{
	switch (GET_OPCODE(insn)) {
	case OPCODE_SPECIAL:
		switch (GET_OPCODE_SPECIAL(insn)) {
		case OPCODE_SPECIAL_JR:
		case OPCODE_SPECIAL_JALR:
			return 1;
		default:
			return 0;
		}
	case OPCODE_REGIMM:
		if ((insn & 0xfc0c0000) == 0x04000000)
			return 1; /* BLTZ, BLTEZ, BLTZL, BGEZL, BLTZAL,
				     BLTZALL, BGEZALL */
		else
			return 0;
	case OPCODE_J:
	case OPCODE_JAL:
		return 1;
	default:
		if ((insn & 0xb0000000) == 0x10000000)
			return 1;
		else
			return 0;
	}
}
#endif /* CHECK_DELAY_SLOT_INST */

static unsigned get_fault_epc(struct pt_regs *regs, int dtlb)
{
#ifdef CHECK_DELAY_SLOT_INST
	unsigned insn;
	int ds, fault;

	if (regs->cp0_cause & CAUSEF_BD) {
		insn = load_pc_insn(regs->cp0_epc, &fault);
		if (!fault && have_delay_slot(insn)) {
			unsigned mask;
			mask = PAGE_SIZE-1;
			if (!dtlb && (regs->cp0_epc & mask) != (mask & ~3))
				ds = 0;
			else
				ds = 1;
		} else
			ds = 0;
		rmt_print_val_dec("Detect exception on delay slot", ds);
		rmt_print_val_hex("insn", insn);
		rmt_show_pt_regs(regs);

#ifdef CONFIG_RMTBUG_FIX_CAUSE_D
		if (!ds) {
			rmt_str_printf("FIX: clear CAUSE D bit\n");
			regs->cp0_cause &= ~CAUSEF_BD;
		}
#endif
		return ds ? regs->cp0_epc + 4 : regs->cp0_epc;
	} else
		return regs->cp0_epc;
#else /* !CHECK_DELAY_SLOT_INST */
	/* see include/asm/stackframe.h  SAVE_SOME */
	if (regs->cp0_cause & CAUSEF_BD)
		return regs->cp0_epc + 4;
	else
		return regs->cp0_epc;
#endif /* !CHECK_DELAY_SLOT_INST */
}

static unsigned get_data_fault_address(struct pt_regs *regs, unsigned insn,
				       int *w)
{
	unsigned address, epc, opcode;
	int w_op;

	rmt_print_val_hex("get_data_fault_address insn", insn);
	w_op = 0;
	opcode = GET_OPCODE(insn);
	if( opcode == OPCODE_COP0 ){ 		/* IOLW, IOLH, IOLB */
		address = regs->regs[GET_BASE(insn)];
	} else if( opcode == OPCODE_VINT ){	/* VILW, VISW */
		address = regs->regs[GET_BASE(insn)];
		if (insn & 4)
			w_op = 1;
	} else if( opcode == OPCODE_VFP ){	/* VFLW, VFSW, VFLD, VFSD */
		address = regs->regs[GET_BASE(insn)];
		if (insn & 4)
			w_op = 1;
	} else {				/* other */
		address = regs->regs[GET_BASE(insn)];
		address += GET_IMMEDIATE(insn);
		if (insn & 0x20000000)
			w_op = 1;
	}

	rmt_print_val_hex("data fault address", address);
	rmt_print_val_hex("data fault write", w_op);

	if (w)
		*w = w_op;
	return address;
}

//static unsigned get_fault_address_dtlb(struct pt_regs *regs, int *w)
unsigned get_fault_address(struct pt_regs *regs, int *w)
{
	unsigned pc, insn;

	pc = get_fault_epc(regs, 1);
	insn = load_pc_insn(pc, NULL);
	return get_data_fault_address(regs, insn, w);
}

asmlinkage void do_itlb_noentry(struct pt_regs *regs)
{
	unsigned address;
	unsigned entry1, entry2;
	pte_t *pte;

	rmt_print_val_hex("do_itlb_noentry epc", regs->cp0_epc);
	sharereg_lock_tlb();
	address = get_fault_epc(regs, 0);
	rmt_print_val_hex("do_itlb_noentry", address);
	pte = find_pte_offset((pgd_t *)pgd_current[smp_processor_id()],
			      address);
	if (!pte_present(*pte))
		goto noentry;
	pte_val(*pte) |= _PAGE_VALID | _PAGE_ACCESSED;

	setup_tlb_entry(address, *pte, 0, &entry1, &entry2);
	load_itlb_entry(entry1, entry2);

	sharereg_unlock_tlb();
	rmt_print_val_hex("do_itlb_noentry load_entry", entry2);
	rmt_show_pt_regs(regs);
	return;
 noentry:
	sharereg_unlock_tlb();

	rmt_print_val_hex("do_itlb_noentry page_fault", address);

	local_irq_enable();
	do_page_fault(regs, 0, address);
}

asmlinkage void do_dtlb_noentry(struct pt_regs *regs)
{
	unsigned address;
	unsigned entry1, entry2;
	int w;
	pte_t *pte;

	rmt_print_val_hex("do_dtlb_noentry epc", regs->cp0_epc);
	sharereg_lock_tlb();
	address = get_fault_address(regs, &w);
	rmt_print_val_hex("do_dtlb_noentry", address);
	pte = find_pte_offset((pgd_t *)pgd_current[smp_processor_id()],
			      address);
	if (!pte_present(*pte))
		goto noentry;
	pte_val(*pte) |= _PAGE_VALID | _PAGE_ACCESSED;

#ifdef CONFIG_RMT_TLBFLUSH_UACCESS        
        if (regs->cp0_epc >= __UA_LIMIT
            && !pte_write(*pte) && w != 0)
                goto noentry;
        if (!pte_write(*pte)) {
                atomic_set(&flag_wptlb_exist, 1);
        }
#endif

	setup_tlb_entry(address, *pte, 0, &entry1, &entry2);
	load_dtlb_entry(entry1, entry2);

	sharereg_unlock_tlb();
	rmt_print_val_hex("do_dtlb_noentry load_entry", entry2);
	rmt_show_pt_regs(regs);
	return;
 noentry:
	sharereg_unlock_tlb();

	rmt_print_val_hex("do_dtlb_noentry page_fault", address);

	local_irq_enable();
	do_page_fault(regs, w, address);
}

asmlinkage void do_dtlb_protection_err(struct pt_regs *regs)
{
	unsigned address;
	unsigned entry1, entry2;
	pte_t *pte;
	int w, group;

	rmt_print_val_hex("do_dtlb_protection_err epc", regs->cp0_epc);
	sharereg_lock_tlb();
	address = get_fault_address(regs, &w);
	//address = get_fault_address_dtlb(regs, &w);
	rmt_print_val_hex("do_dtlb_protection_err", address);

	if (!w)
		goto page_fault;	/* mode error */

	pte = find_pte_offset((pgd_t *)pgd_current[smp_processor_id()],
			      address);
	if (!pte_present(*pte))
		goto page_fault;

	if ((pte_val(*pte) & (_PAGE_PRESENT|_PAGE_WRITE)) !=
	    (_PAGE_PRESENT|_PAGE_WRITE))
		goto page_fault;	/* no write permission */

	pte_val(*pte) |= _PAGE_VALID | _PAGE_ACCESSED |
		_PAGE_MODIFIED | _PAGE_DIRTY;

	if (address >= VMALLOC_START) {
		panic("write protection error on vmalloc area (fault=%08x)",
		      address);
	}
	invalidate_dtlb_entry(address, *pte);
	setup_tlb_entry(address, *pte, 1, &entry1, &entry2);
#ifdef CONFIG_RMT_TLBFLUSH_UACCESS        
             if ((entry1 & TLB_ENT1_PRO_ALLRW) == 
                TLB_ENT1_PRO_USRR ){
                    atomic_set(&flag_wptlb_exist, 1);
             }
#endif
	load_dtlb_entry(entry1, entry2);

	sharereg_unlock_tlb();

	rmt_print_val_hex("do_dtlb_protection_err load_entry", entry2);
	rmt_show_pt_regs(regs);
	return;
 page_fault:
	sharereg_unlock_tlb();

	rmt_print_val_hex("do_dtlb_protection_err page_fault", address);

	local_irq_enable();
	do_page_fault(regs, w, address);
}

/*****/

asmlinkage void do_coproc_unusable(struct pt_regs *regs)
{
	die("Coprocessor Unusable", regs);
}

static  unsigned long get_phys_page (unsigned long addr,
					   struct mm_struct *mm)
{
	pgd_t *pgd;
	pmd_t *pmd;
	pte_t *pte;
	unsigned long physpage;

	pgd = pgd_offset(mm, addr);
	pmd = pmd_offset(pgd, addr);
	pte = pte_offset(pmd, addr);

	if ((physpage = pte_val(*pte)) & _PAGE_VALID)
		return KSEG0ADDR(physpage & PAGE_MASK);

	return 0;
}

asmlinkage void do_invalid_instruction(struct pt_regs *regs)
{
#ifdef CONFIG_KERNEL_VERBOSE
	printk("[%d] invalid_instruction epc=%x\n",
		rmt_mycnum(), regs->cp0_epc);
	show_regs(regs);
#endif

	die_if_kernel("Reserved instruction in kernel code", regs);

	if (!cpu_has_llsc)
		if (!simulate_llsc(regs))
			return;

	force_sig(SIGILL, current);
}

#ifdef CONFIG_RMTDEBUG_REGSDUMP
static struct pt_regs break_point_last_regs;
#endif

asmlinkage void do_break_point(struct pt_regs *regs)
{
	unsigned int opcode, bcode;
	siginfo_t info;

#ifdef CONFIG_RMTDEBUG_REGSDUMP
	break_point_last_regs = *regs;
	rmt_print_val_hex("break_point_last_regs start",
			  (unsigned)&break_point_last_regs);
	rmt_print_val_hex("break_point_last_regs end",
			  (unsigned)(&break_point_last_regs + 1) -1);
#endif
	rmt_str_printf("### EXP: DO_BREAK_POINT ###\n");
	rmt_show_pt_regs(regs);

	die_if_kernel("Break instruction in kernel code", regs);

#ifdef CONFIG_RMTDEBUG_REGSDUMP
	regs->cp0_epc += 4;
	rmt_print_val_hex("new epc",
			  regs->cp0_epc);
#else
	if (get_insn_opcode(regs, &opcode))
		return;

	/*
	 * There is the ancient bug in the MIPS assemblers that the break
	 * code starts left to bit 16 instead to bit 6 in the opcode.
	 * Gas is bug-compatible ...
	 */
	bcode = ((opcode >> 16) & ((1 << 20) - 1));

	/*
	 * (A short test says that IRIX 5.3 sends SIGTRAP for all break
	 * insns, even for break codes that indicate arithmetic failures.
	 * Weird ...)
	 * But should we continue the brokenness???  --macro
	 */
	switch (bcode) {
	case 6:
	case 7:
		if (bcode == 7)
			info.si_code = FPE_INTDIV;
		else
			info.si_code = FPE_INTOVF;
		info.si_signo = SIGFPE;
		info.si_errno = 0;
		info.si_addr = (void *)regs->cp0_epc;
		force_sig_info(SIGFPE, &info, current);
		break;
	default:
		force_sig(SIGTRAP, current);
	}
#endif
}

asmlinkage void do_int_overflow(struct pt_regs *regs)
{
	siginfo_t info;

	info.si_code = FPE_INTOVF;
	info.si_signo = SIGFPE;
	info.si_errno = 0;
	info.si_addr = (void *)regs->cp0_epc;
	force_sig_info(SIGFPE, &info, current);
}

asmlinkage void do_divide_byzero(struct pt_regs *regs)
{
	siginfo_t info;
	die_if_kernel("DivZero", regs);

	info.si_code = FPE_INTDIV;
	info.si_signo = SIGFPE;
	info.si_errno = 0;
	info.si_addr = (void *)regs->cp0_epc;
	force_sig_info(SIGFPE, &info, current);
}

asmlinkage void do_trap(struct pt_regs *regs)
{
	rmt_str_printf("### EXP: TRAP ###\n");
	
	unsigned int opcode, tcode = 0;
	siginfo_t info;

	die_if_kernel("Trap instruction in kernel code", regs);

	if (get_insn_opcode(regs, &opcode))
		return;

	/* Immediate versions don't provide a code.  */
	if (!(opcode & OPCODE))
		tcode = ((opcode >> 6) & ((1 << 20) - 1));

	/*
	 * (A short test says that IRIX 5.3 sends SIGTRAP for all trap
	 * insns, even for trap codes that indicate arithmetic failures.
	 * Weird ...)
	 * But should we continue the brokenness???  --macro
	 */
	switch (tcode) {
	case 6:
	case 7:
		if (tcode == 7)
			info.si_code = FPE_INTDIV;
		else
			info.si_code = FPE_INTOVF;
		info.si_signo = SIGFPE;
		info.si_errno = 0;
		info.si_addr = (void *)regs->cp0_epc;
		force_sig_info(SIGFPE, &info, current);
		break;
	default:
		force_sig(SIGTRAP, current);
	}
}

/*
asmlinkage void do_load_dataaddr_missalign(struct pt_regs *regs, unsigned int opcode)
{
	rmt_str_printf("### EXP: LOAD_DATAADDR_MISSALIGN ###\n");
}

asmlinkage void do_store_dataaddr_missalign(struct pt_regs *regs, unsigned int opcode)
{
	rmt_str_printf("### EXP: STORE_DATAADDR_MISSALIGN ###\n");
}
*/

asmlinkage void do_float_overflow(struct pt_regs *regs)
{
	siginfo_t info;

	info.si_code = FPE_FLTOVF;
	info.si_signo = SIGFPE;
	info.si_errno = 0;
	info.si_addr = (void *)regs->cp0_epc;
	force_sig_info(SIGFPE, &info, current);
}

asmlinkage void do_float_underflow(struct pt_regs *regs)
{
	siginfo_t info;

	info.si_code = FPE_FLTUND;
	info.si_signo = SIGFPE;
	info.si_errno = 0;
	info.si_addr = (void *)regs->cp0_epc;
	force_sig_info(SIGFPE, &info, current);
}

asmlinkage void do_float_divide_byzero(struct pt_regs *regs)
{
	siginfo_t info;

	info.si_code = FPE_FLTDIV;
	info.si_signo = SIGFPE;
	info.si_errno = 0;
	info.si_addr = (void *)regs->cp0_epc;
	force_sig_info(SIGFPE, &info, current);
}

asmlinkage void do_float_inexact(struct pt_regs *regs)
{
	siginfo_t info;

	info.si_code = FPE_FLTRES;
	info.si_signo = SIGFPE;
	info.si_errno = 0;
	info.si_addr = (void *)regs->cp0_epc;
	force_sig_info(SIGFPE, &info, current);
}

asmlinkage void do_float_invalid_operation(struct pt_regs *regs)
{
	siginfo_t info;

	info.si_code = FPE_FLTINV;
	info.si_signo = SIGFPE;
	info.si_errno = 0;
	info.si_addr = (void *)regs->cp0_epc;
	force_sig_info(SIGFPE, &info, current);
}

asmlinkage void do_reserved(struct pt_regs *regs)
{
	__asm__ __volatile__ (
		"lui	$26, 0x0300\n"
		"ori	$27, $0, 0xff\n"
		"mtc0	$26, $27\n"
	);
	rmt_str_printf("### EXP: RESERVED ###\n");
	
	/*
	 * Game over - no way to handle this if it ever occurs.  Most probably
	 * caused by a new unknown cpu type or after another deadly
	 * hard/software error.
	 */
	show_regs(regs);
	panic("Caught reserved exception %ld - should not happen.",
	      (regs->cp0_cause & 0x7f) >> 2);
}

asmlinkage void do_vectorint_exp(struct pt_regs *regs)
{
	die("VINT-EXP", regs);
}

asmlinkage void do_vectorfloat_exp(struct pt_regs *regs)
{
	die("VFLT-EXP", regs);
}

asmlinkage void do_hardware_interruption(struct pt_regs *regs)
{
	die("HW-Intrpt", regs);
}

asmlinkage void do_software_interruption1(struct pt_regs *regs)
{
	rmt_str_printf("### EXP: SOFTWARE_INTERRUPTION1 ###\n");
#ifdef CONFIG_SMP
	int cause, tmp;

	tmp = CP0_OWN_CAUSE;
	__asm__ __volatile__ (
	"mfc0	%0, %1\n"
	"ori	%0, 0x200\n"
	"xori	%0, %0, 0x200\n"
	"mtc0	%0, %1\n"
	: "=&r" ((int)cause)
	: "r" ((int)tmp)
	);
	clear_c0_status(ST0_IE);
	smp_call_function_interrupt();
#endif
}

asmlinkage void do_software_interruption0(struct pt_regs *regs)
{
	rmt_str_printf("### EXP: SOFTWARE_INTERRUPTION0 ###\n");
#ifdef CONFIG_SMP
	int cause, tmp;

	tmp = CP0_OWN_CAUSE;
	__asm__ __volatile__ (
	"mfc0	%0, %1\n"
	"ori	%0, 0x100\n"
	"xori	%0, %0, 0x100\n"
	"mtc0	%0, %1\n"
	: "=&r" ((int)cause)
	: "r" ((int)tmp)
	);
	clear_c0_status(ST0_IE);
	update_process_times(user_mode(regs));
#endif
	return;
}
#else
asmlinkage void do_be(struct pt_regs *regs)
{
	const int field = 2 * sizeof(unsigned long);
	const struct exception_table_entry *fixup = NULL;
	int data = regs->cp0_cause & 4;
	int action = MIPS_BE_FATAL;

	/* XXX For now.  Fixme, this searches the wrong table ...  */
	if (data && !user_mode(regs))
		fixup = search_dbe_tables(exception_epc(regs));

	if (fixup)
		action = MIPS_BE_FIXUP;

	if (board_be_handler)
		action = board_be_handler(regs, fixup != 0);

	switch (action) {
	case MIPS_BE_DISCARD:
		return;
	case MIPS_BE_FIXUP:
		if (fixup) {
			regs->cp0_epc = fixup->nextinsn;
			return;
		}
		break;
	default:
		break;
	}

	/*
	 * Assume it would be too dangerous to continue ...
	 */
	printk(KERN_ALERT "%s bus error, epc == %0*lx, ra == %0*lx\n",
	       data ? "Data" : "Instruction",
	       field, regs->cp0_epc, field, regs->regs[31]);
	die_if_kernel("Oops", regs);
	force_sig(SIGBUS, current);
}
#endif

static inline int get_insn_opcode(struct pt_regs *regs, unsigned int *opcode)
{
	unsigned int *epc;

	epc = (unsigned int *) regs->cp0_epc +
	      ((regs->cp0_cause & CAUSEF_BD) != 0);
	if (!get_user(*opcode, epc))
		return 0;

	force_sig(SIGSEGV, current);
	return 1;
}

/*
 * The ll_bit is cleared by r*_switch.S
 */

unsigned long ll_bit;

static struct task_struct *ll_task = NULL;

static inline void simulate_ll(struct pt_regs *regs, unsigned int opcode)
{
	unsigned long value, *vaddr;
	long offset;
	int signal = 0;

	/*
	 * analyse the ll instruction that just caused a ri exception
	 * and put the referenced address to addr.
	 */

	/* sign extend offset */
	offset = opcode & OFFSET;
	offset <<= 16;
	offset >>= 16;

	vaddr = (unsigned long *)((long)(regs->regs[(opcode & BASE) >> 21]) + offset);

	if ((unsigned long)vaddr & 3) {
		signal = SIGBUS;
		goto sig;
	}
	if (get_user(value, vaddr)) {
		signal = SIGSEGV;
		goto sig;
	}

	if (ll_task == NULL || ll_task == current) {
		ll_bit = 1;
	} else {
		ll_bit = 0;
	}
	ll_task = current;

	regs->regs[(opcode & RT) >> 16] = value;

	compute_return_epc(regs);
	return;

sig:
	force_sig(signal, current);
}

static inline void simulate_sc(struct pt_regs *regs, unsigned int opcode)
{
	unsigned long *vaddr, reg;
	long offset;
	int signal = 0;

	/*
	 * analyse the sc instruction that just caused a ri exception
	 * and put the referenced address to addr.
	 */

	/* sign extend offset */
	offset = opcode & OFFSET;
	offset <<= 16;
	offset >>= 16;

	vaddr = (unsigned long *)((long)(regs->regs[(opcode & BASE) >> 21]) + offset);
	reg = (opcode & RT) >> 16;

	if ((unsigned long)vaddr & 3) {
		signal = SIGBUS;
		goto sig;
	}
	if (ll_bit == 0 || ll_task != current) {
		regs->regs[reg] = 0;
		compute_return_epc(regs);
		return;
	}

	if (put_user(regs->regs[reg], vaddr)) {
		signal = SIGSEGV;
		goto sig;
	}

	regs->regs[reg] = 1;

	compute_return_epc(regs);
	return;

sig:
	force_sig(signal, current);
}

/*
 * ll uses the opcode of lwc0 and sc uses the opcode of swc0.  That is both
 * opcodes are supposed to result in coproc unusable exceptions if
 * executed on ll/sc-less processors.  That's the theory.  In practice a
 * few processors such as NEC's VR4100 throw reserved instruction exceptions
 * instead, so we're doing the emulation thing in both exception handlers.
 */
static inline int simulate_llsc(struct pt_regs *regs)
{
	unsigned int opcode;

	if (unlikely(get_insn_opcode(regs, &opcode)))
		return -EFAULT;

	if ((opcode & OPCODE) == LL) {
		simulate_ll(regs, opcode);
		return 0;
	}
	if ((opcode & OPCODE) == SC) {
		simulate_sc(regs, opcode);
		return 0;
	}

	return -EFAULT;			/* Strange things going on ... */
}

asmlinkage void cache_parity_error(void)
{
	const int field = 2 * sizeof(unsigned long);
	unsigned int reg_val;

	/* For the moment, report the problem and hang. */
	printk("Cache error exception:\n");

#if 0 /* MIPS */
	printk("cp0_errorepc == %0*lx\n", field, read_c0_errorepc());
	reg_val = read_c0_cacheerr();
	printk("c0_cacheerr == %08x\n", reg_val);

	printk("Decoded c0_cacheerr: %s cache fault in %s reference.\n",
	       reg_val & (1<<30) ? "secondary" : "primary",
	       reg_val & (1<<31) ? "data" : "insn");
	printk("Error bits: %s%s%s%s%s%s%s\n",
	       reg_val & (1<<29) ? "ED " : "",
	       reg_val & (1<<28) ? "ET " : "",
	       reg_val & (1<<26) ? "EE " : "",
	       reg_val & (1<<25) ? "EB " : "",
	       reg_val & (1<<24) ? "EI " : "",
	       reg_val & (1<<23) ? "E1 " : "",
	       reg_val & (1<<22) ? "E0 " : "");
	printk("IDX: 0x%08x\n", reg_val & ((1<<22)-1));

#if defined(CONFIG_CPU_RMT)
	if (reg_val & (1<<22))
		printk("DErrAddr0: 0x%0*lx\n", field, read_c0_derraddr0());

	if (reg_val & (1<<23))
		printk("DErrAddr1: 0x%0*lx\n", field, read_c0_derraddr1());
#endif
#endif
	panic("Can't handle the cache error!");
}

void *set_except_vector(int n, void *addr)
{
	static unsigned long exception_handlers[32];
	unsigned long handler = (unsigned long) addr;
	unsigned long old_handler = exception_handlers[n];

	exception_handlers[n] = handler;
	memcpy((void *)(EXP_BASE_ADDR + n*0x10), handler, 0x10);
	
	return (void *)old_handler;
}

/*
 * This is used by native signal handling
 */
asmlinkage int (*save_fp_context)(struct sigcontext *sc);
asmlinkage int (*restore_fp_context)(struct sigcontext *sc);

extern asmlinkage int _save_fp_context(struct sigcontext *sc);
extern asmlinkage int _restore_fp_context(struct sigcontext *sc);

extern asmlinkage int fpu_emulator_save_context(struct sigcontext *sc);
extern asmlinkage int fpu_emulator_restore_context(struct sigcontext *sc);

static inline void signal_init(void)
{
	if (cpu_has_fpu) {
		save_fp_context = _save_fp_context;
		restore_fp_context = _restore_fp_context;
	} else {
		save_fp_context = fpu_emulator_save_context;
		restore_fp_context = fpu_emulator_restore_context;
	}
}

#ifdef CONFIG_MIPS32_COMPAT

/*
 * This is used by 32-bit signal stuff on the 64-bit kernel
 */
asmlinkage int (*save_fp_context32)(struct sigcontext32 *sc);
asmlinkage int (*restore_fp_context32)(struct sigcontext32 *sc);

extern asmlinkage int _save_fp_context32(struct sigcontext32 *sc);
extern asmlinkage int _restore_fp_context32(struct sigcontext32 *sc);

extern asmlinkage int fpu_emulator_save_context32(struct sigcontext32 *sc);
extern asmlinkage int fpu_emulator_restore_context32(struct sigcontext32 *sc);

static inline void signal32_init(void)
{
	if (cpu_has_fpu) {
		save_fp_context32 = _save_fp_context32;
		restore_fp_context32 = _restore_fp_context32;
	} else {
		save_fp_context32 = fpu_emulator_save_context32;
		restore_fp_context32 = fpu_emulator_restore_context32;
	}
}
#endif

extern void cpu_cache_init(void);
extern void tlb_init(void);
extern void enable_mmu(void);

void __init per_cpu_trap_init(void)
{
	unsigned int cpu = smp_processor_id();

	if(!rmt_mycnum())
		asid_cache(cpu) = ASID_FIRST_VERSION;
	
	rmt_str_printf("per_cpu_trap_init() #1\n");
	rmt_str_printf("swapper_pg_dir=0x"); rmt_hex_printf(swapper_pg_dir);
	rmt_str_printf("TLBMISS_HANDLER_SETUP()\n");
	TLBMISS_HANDLER_SETUP();
	rmt_str_printf("pgd_current=0x"); rmt_hex_printf(pgd_current[smp_processor_id()]);
	rmt_str_printf("&pgd_current=0x"); rmt_hex_printf(&pgd_current[smp_processor_id()]);

	atomic_inc(&init_mm.mm_count);
	rmt_str_printf("per_cpu_trap_init() #2\n");
	current->active_mm = &init_mm;
	rmt_str_printf("per_cpu_trap_init() #3\n");
	BUG_ON(current->mm);
	rmt_str_printf("per_cpu_trap_init() #4\n");
	enter_lazy_tlb(&init_mm, current);
	rmt_str_printf("per_cpu_trap_init() #5\n");

	cpu_cache_init();
	rmt_str_printf("tlb_init()\n");
	tlb_init();
	rmt_str_printf("per_cpu_trap_init() #0\n");
}

void __init trap_init(void)
{
	rmt_str_printf("per_cpu_trap_init()\n");
	per_cpu_trap_init();

	/*
	 * Copy the generic exception handlers to their final destination.
	 * This will be overriden later as suitable for a particular
	 * configuration.
	 */
	set_except_vector(0x01, except_vec_itlb_spraddr_miss);
	set_except_vector(0x02, except_vec_itlb_allentry_locked);
	set_except_vector(0x03, except_vec_itlb_noentry_matched);
	set_except_vector(0x04, except_vec_itlb_threadmode_err);
	set_except_vector(0x05, except_vec_itlb_protection_err);
	set_except_vector(0x06, except_vec_dtlb_spraddr_miss);
	set_except_vector(0x07, except_vec_dtlb_allentry_locked);
	set_except_vector(0x08, except_vec_dtlb_noentry_matched);
	set_except_vector(0x09, except_vec_dtlb_threadmode_err);
	set_except_vector(0x0a, except_vec_dtlb_protection_err);
	set_except_vector(0x0b, except_vec_coproc_unusable);
	set_except_vector(0x0c, except_vec_invalid_instruction);
	set_except_vector(0x0d, except_vec_system_call);
	set_except_vector(0x0e, except_vec_break_point);
	set_except_vector(0x0f, except_vec_int_overflow);
	set_except_vector(0x10, except_vec_divide_byzero);
	set_except_vector(0x11, except_vec_trap);
	set_except_vector(0x12, except_vec_load_dataaddr_missalign);
	set_except_vector(0x13, except_vec_store_dataaddr_missalign);
	set_except_vector(0x14, except_vec_float_overflow);
	set_except_vector(0x15, except_vec_float_underflow);
	set_except_vector(0x16, except_vec_float_divide_byzero);
	set_except_vector(0x17, except_vec_float_inexact);
	set_except_vector(0x18, except_vec_float_invalid_operation);
	set_except_vector(0x19, except_vec_reserved);
	set_except_vector(0x1a, except_vec_vectorint_exp);
	set_except_vector(0x1b, except_vec_vectorfloat_exp);
	set_except_vector(0x1c, except_vec_timer_interruption);
	set_except_vector(0x1d, except_vec_hardware_interruption);
	set_except_vector(0x1e, except_vec_software_interruption0);
	set_except_vector(0x1f, except_vec_software_interruption1);
	AXE_MEMDUMP(0x80000300);
	AXE_MEMDUMP(0x80000380);

	rmt_str_printf("signal_init()\n");
	signal_init();
#ifdef CONFIG_MIPS32_COMPAT
	rmt_str_printf("signal32_init()\n");
	signal32_init();
#endif

#ifndef CONFIG_AXE_AVOID
	rmt_str_printf("flush_icashe_range()\n");
	flush_icache_range(CAC_BASE, CAC_BASE + 0x400);
#endif
}

void set_temporary_exception_handlers(void)
{
	extern char except_vec_temporary;
	int i;
	
	for(i=0x01; i<=0x1f; i++) {
		memcpy((void *)(EXP_BASE_ADDR + i*0x10),
			&except_vec_temporary, 0x10);
	}
}

void print_temporary_exception(void)
{
	rmt_str_printf("Exception Occurs!\n");
	rmt_str_printf("Status Register: 0x");
	rmt_hex_printf(read_c0_status());
	
	rmt_str_printf("Exceptoin PC Register: 0x");
	rmt_hex_printf(read_c0_epc());
	
	rmt_str_printf("Exception Code: 0x");
	rmt_hex_printf((read_c0_cause()>>2) & 0x1f);

	AXE_EXIT();
	while(1) ;
}

