#include <linux/smp.h>
#include <linux/spinlock.h>
#include <linux/sched.h>
#include <asm/ioctls.h>
#include <asm/thread_control.h>

#define DBGOUT_C_SOURCE
#include <asm/dbgout.h>


spinlock_t axeputchar_lock = SPIN_LOCK_UNLOCKED;

#ifdef CONFIG_RMT_UARTIO
#define PUTCHAR(x) rmtsim_cons_put_char((x), RMT_UART_BASE)
#else
#define PUTCHAR(x) AXE_PUTCHAR(x)
#endif

static void print_thread_info(void)
{
#ifdef CONFIG_SMP
	const char digit[] = "0123456789abcdef";
	int i, stat;
	
	stat = read_c0_status();

	/* smp processor id */
	PUTCHAR('s');
	for(i=1; i>=0; i--) PUTCHAR(digit[(smp_processor_id()>>(i*4))&0xf]);
	/* thread id */
	PUTCHAR(':');
	PUTCHAR('t');
	for(i=1; i>=0; i--) PUTCHAR(digit[(rmt_getotid()>>(i*4))&0xf]);
	/* context id */
	PUTCHAR(':');
	PUTCHAR('c');
	for(i=3; i>=0; i--) PUTCHAR(digit[(rmt_mycnum()>>(i*4))&0xf]);
	/* status register (lower 8 bits) */
	PUTCHAR(':');
	PUTCHAR('r');
	for(i=3; i>=0; i--) PUTCHAR(digit[(stat>>(i*4))&0xf]);
	/* process id */
	//PUTCHAR(':');
	//PUTCHAR('p');
	//for(i=7; i>=0; i--) PUTCHAR(digit[(current->pid>>(i*4))&0xf]);

	PUTCHAR(':');
	PUTCHAR(' ');
#endif
}

void _rmt_str_printf(char *str)
{	
	unsigned long flags;

	spin_lock_irqsave(&axeputchar_lock, flags);
	print_thread_info();
	while(*str) {
		if (*str == '\n')
			PUTCHAR('\r');
		PUTCHAR(*str++);
	}
	spin_unlock_irqrestore(&axeputchar_lock, flags);
}

void _rmt_hex_printf(unsigned long val)
{
	int	i;
	const char digit[] = "0123456789abcdef";
	unsigned long flags;
	
	spin_lock_irqsave(&axeputchar_lock, flags);
	
	for(i=7; i>=0; i--) PUTCHAR(digit[(val>>(i*4))&0xf]);
	PUTCHAR('\r');
	PUTCHAR('\n');

	spin_unlock_irqrestore(&axeputchar_lock, flags);
}

void _rmt_hex_print(unsigned long val)
{
	int	i;
	const char digit[] = "0123456789abcdef";
	unsigned long flags;
	
	spin_lock_irqsave(&axeputchar_lock, flags);
	
	for(i=7; i>=0; i--) PUTCHAR(digit[(val>>(i*4))&0xf]);
	
	spin_unlock_irqrestore(&axeputchar_lock, flags);
}

static void print_str(char *str)
{
	while(*str) {
		if (*str == '\n')
			PUTCHAR('\r');
		PUTCHAR(*str++);
	}
}

static void println(void)
{
	PUTCHAR('\r');
	PUTCHAR('\n');
}

static void print_hex_n(unsigned hex, int n)
{
	const char hextbl[] = "0123456789abcdef";

	for (n--; n >= 0; n--)
		PUTCHAR(hextbl[(hex >> (4 * n)) & 0xf]);
}

static void print_hex(unsigned hex)
{
	print_hex_n(hex, 8);
}

static void print_dec(unsigned dec)
{
	char buf[16];
	int n;
	
	n = 0;
	while (n < 15 & dec != 0) {
		buf[n++] = '0' + dec % 10;
		dec /= 10;
	}
	if (n == 0) {
		buf[n++] = '0';
	}
	while (n-- > 0)
		PUTCHAR(buf[n]);
}

void _rmt_print_val_hex(char *msg, unsigned hex)
{
	unsigned long flags;
	
	spin_lock_irqsave(&axeputchar_lock, flags);
	print_thread_info();
	print_str(msg);
	print_str(": ");
	print_hex(hex);
	print_str("\n");
	spin_unlock_irqrestore(&axeputchar_lock, flags);
}

void _rmt_print_val_dec(char *msg, unsigned dec)
{
	unsigned long flags;
	
	spin_lock_irqsave(&axeputchar_lock, flags);
	print_thread_info();
	print_str(msg);
	print_str(": ");
	print_dec(dec);
	print_str("\n");
	spin_unlock_irqrestore(&axeputchar_lock, flags);
}

void _rmt_print_hex(unsigned hex)
{
	unsigned long flags;
	
	spin_lock_irqsave(&axeputchar_lock, flags);
	print_hex(hex);
	spin_unlock_irqrestore(&axeputchar_lock, flags);
}

void _rmt_print_dec(unsigned dec)
{
	unsigned long flags;
	
	spin_lock_irqsave(&axeputchar_lock, flags);
	print_dec(dec);
	spin_unlock_irqrestore(&axeputchar_lock, flags);
}

static void print_task(struct task_struct *tsk)
{
	print_str("[");
	print_dec(tsk->pid);
	print_str(":");
	print_str(tsk->comm);
	print_str("]");
}

static void print_cp0status(unsigned cp0status)
{
	print_str("cp0status:");
	if (cp0status & ST0_TS)
		print_str(" TS");
	if (cp0status & ST0_PE)
		print_str(" PE");
	if (cp0status & ST0_EV)
		print_str(" EV");
	print_str(" IM(");
	if (cp0status & ST0_IM_TIMER)
		print_str(" TIMER");
	if (cp0status & ST0_IM_HW)
		print_str(" HW");
	if (cp0status & ST0_IM_SW1)
		print_str(" SW1");
	if (cp0status & ST0_IM_SW2)
		print_str(" SW2");
	print_str(")");

	if (cp0status & ST0_EB)
		print_str(" EB");

	switch (cp0status & ST0_KSU) {
	case KSU_USER:
		print_str(" USER");
		break;
	case KSU_SUPERVISOR:
		print_str(" SUPERVISOR");
		break;
	case KSU_KERNEL:
		print_str(" KERNEL");
		break;
	default:
		print_str(" ?");
	}

	if (cp0status & ST0_IE)
		print_str(" IE");
	if (cp0status & ST0_EL)
		print_str(" EL");
}

static void print_cp0cause(unsigned cp0cause)
{
	print_str("cp0cause:");
	if (cp0cause & 0x80000000)
		print_str(" D");
	print_str(" HIRL=");
	print_hex_n((cp0cause >> 12) & 0x1f, 2);
	if (cp0cause & CAUSE_PND_TI)
		print_str(" TI");
	if (cp0cause & CAUSE_PND_HI)
		print_str(" HI");
	if (cp0cause & CAUSE_PND_S1)
		print_str(" S1");
	if (cp0cause & CAUSE_PND_S0)
		print_str(" S0");
	print_str(" CODE=");
	print_hex_n((cp0cause >> 2) & 0x1f, 2);
}

void _rmt_show_pt_regs(struct pt_regs *regs)
{
	unsigned long flags;

	spin_lock_irqsave(&axeputchar_lock, flags);
	print_thread_info();
	print_task(current);
	print_str(" ");
	print_cp0status(regs->cp0_status);
	print_str(" ");
	print_cp0cause(regs->cp0_cause);
	print_str("\n");

	print_str("status: ");
	print_hex(regs->cp0_status);
	print_str("  epc: ");
	print_hex(regs->cp0_epc);
	print_str("  cause: ");
	print_hex(regs->cp0_cause);
	println();

#ifdef NEVER
	print_str("hi: ");
	print_hex(regs->hi);
	print_str("  lo: ");
	print_hex(regs->lo);
	print_str("  (mmu_spr_exp_addr: ");
	print_hex(regs->mmu_spr_exp_addr);
	print_str(")\n");
#endif

	print_str("$0: ");	print_hex(regs->regs[0]);
	print_str("  at: "); print_hex(regs->regs[1]);
	print_str("  v0: "); print_hex(regs->regs[2]);
	print_str("  v1: "); print_hex(regs->regs[3]);
	println();

	print_str("a0: "); print_hex(regs->regs[4]);
	print_str("  a1: "); print_hex(regs->regs[5]);
	print_str("  a2: "); print_hex(regs->regs[6]);
	print_str("  a3: "); print_hex(regs->regs[7]);
	println();

	print_str("t0: "); print_hex(regs->regs[8]);
	print_str("  t1: "); print_hex(regs->regs[9]);
	print_str("  t2: "); print_hex(regs->regs[10]);
	print_str("  t3: "); print_hex(regs->regs[11]);
	println();

	print_str("t4: "); print_hex(regs->regs[12]);
	print_str("  t5: "); print_hex(regs->regs[13]);
	print_str("  t6: "); print_hex(regs->regs[14]);
	print_str("  t7: "); print_hex(regs->regs[15]);
	println();

	print_str("s0: "); print_hex(regs->regs[16]);
	print_str("  s1: "); print_hex(regs->regs[17]);
	print_str("  s2: "); print_hex(regs->regs[18]);
	print_str("  s3: "); print_hex(regs->regs[19]);
	println();

	print_str("s4: "); print_hex(regs->regs[20]);
	print_str("  s5: "); print_hex(regs->regs[21]);
	print_str("  s6: "); print_hex(regs->regs[22]);
	print_str("  s7: "); print_hex(regs->regs[23]);
	println();

	print_str("t8: "); print_hex(regs->regs[24]);
	print_str("  t9: "); print_hex(regs->regs[25]);
	print_str("  k0: "); print_hex(regs->regs[26]);
	print_str("  k1: "); print_hex(regs->regs[27]);
	println();

	print_str("gp: "); print_hex(regs->regs[28]);
	print_str("  sp: "); print_hex(regs->regs[29]);
	print_str("  fp: "); print_hex(regs->regs[30]);
	print_str("  ra: "); print_hex(regs->regs[31]);
	println();
	spin_unlock_irqrestore(&axeputchar_lock, flags);
}

void _rmt_print_task(struct task_struct *tsk)
{
	unsigned long flags;
	
	spin_lock_irqsave(&axeputchar_lock, flags);
	print_thread_info();
	print_task(tsk);
	spin_unlock_irqrestore(&axeputchar_lock, flags);
}

void _rmt_show_regs(void)
{
	unsigned long flags;

	spin_lock_irqsave(&axeputchar_lock, flags);
	print_str("show_regs:\n");
	print_thread_info();
	print_task(current);
	print_str(" ");
	print_cp0status(read_c0_status());
	print_str(" ");
	print_cp0cause(read_c0_cause());
	print_str("\n");

	print_str("status: ");
	print_hex(read_c0_status());
	print_str("  epc: ");
	print_hex(read_c0_epc());
	print_str("  cause: ");
	print_hex(read_c0_cause());
	println();

	print_str("$0: ");	print_hex(read_gp_register($0));
	print_str("  at: "); print_hex(read_gp_register($1));
	print_str("  v0: "); print_hex(read_gp_register($2));
	print_str("  v1: "); print_hex(read_gp_register($3));
	println();

	print_str("a0: "); print_hex(read_gp_register($4));
	print_str("  a1: "); print_hex(read_gp_register($5));
	print_str("  a2: "); print_hex(read_gp_register($6));
	print_str("  a3: "); print_hex(read_gp_register($7));
	println();

	print_str("t0: "); print_hex(read_gp_register($8));
	print_str("  t1: "); print_hex(read_gp_register($9));
	print_str("  t2: "); print_hex(read_gp_register($10));
	print_str("  t3: "); print_hex(read_gp_register($11));
	println();

	print_str("t4: "); print_hex(read_gp_register($12));
	print_str("  t5: "); print_hex(read_gp_register($13));
	print_str("  t6: "); print_hex(read_gp_register($14));
	print_str("  t7: "); print_hex(read_gp_register($15));
	println();

	print_str("s0: "); print_hex(read_gp_register($16));
	print_str("  s1: "); print_hex(read_gp_register($17));
	print_str("  s2: "); print_hex(read_gp_register($18));
	print_str("  s3: "); print_hex(read_gp_register($19));
	println();

	print_str("s4: "); print_hex(read_gp_register($20));
	print_str("  s5: "); print_hex(read_gp_register($21));
	print_str("  s6: "); print_hex(read_gp_register($22));
	print_str("  s7: "); print_hex(read_gp_register($23));
	println();

	print_str("t8: "); print_hex(read_gp_register($24));
	print_str("  t9: "); print_hex(read_gp_register($25));
	print_str("  k0: "); print_hex(read_gp_register($26));
	print_str("  k1: "); print_hex(read_gp_register($27));
	println();

	print_str("gp: "); print_hex(read_gp_register($28));
	print_str("  sp: "); print_hex(read_gp_register($29));
	print_str("  fp: "); print_hex(read_gp_register($30));
	print_str("  ra: "); print_hex(read_gp_register($31));
	println();

	print_str("icache-on: ");	print_hex(__read_32bit_c0_register(0x80, 0));
	print_str("  icache-off: "); print_hex(__read_32bit_c0_register(0x86, 0));
	println();

#ifdef NEVER
	print_str("MMU_SPR_EXP_LOG: ");
	print_hex(read_immu_register(MMU_SPR_EXP_LOG));
	print_str(" ");
	print_hex(read_dmmu_register(MMU_SPR_EXP_LOG));
	print_str("  MMU_SPR_EXP_ADDR: ");
	print_hex(read_immu_register(MMU_SPR_EXP_ADDR));
	print_str(" ");
	print_hex(read_dmmu_register(MMU_SPR_EXP_ADDR));
	println();
#endif
	spin_unlock_irqrestore(&axeputchar_lock, flags);

#ifdef NEVER
	_rmt_str_printf("rmt_show_regs:\n");
	_rmt_str_printf("icache-on: "); _rmt_hex_printf(__read_32bit_c0_register(0x80, 0));
	_rmt_str_printf("dcache-on: "); _rmt_hex_printf(__read_32bit_c0_register(0x86, 0));
	
	_rmt_str_printf("status: "); _rmt_hex_printf(read_c0_status());
	_rmt_str_printf("epc: "); _rmt_hex_printf(read_c0_epc());
	_rmt_str_printf("cause: "); _rmt_hex_printf(read_c0_cause());

	_rmt_str_printf("$0: "); _rmt_hex_printf(read_gp_register($0));
	//_rmt_str_printf("at: "); _rmt_hex_printf(read_gp_register($1));
	_rmt_str_printf("v0: "); _rmt_hex_printf(read_gp_register($2));
	_rmt_str_printf("v1: "); _rmt_hex_printf(read_gp_register($3));
	_rmt_str_printf("a0: "); _rmt_hex_printf(read_gp_register($4));
	_rmt_str_printf("a1: "); _rmt_hex_printf(read_gp_register($5));
	_rmt_str_printf("a2: "); _rmt_hex_printf(read_gp_register($6));
	_rmt_str_printf("a3: "); _rmt_hex_printf(read_gp_register($7));
	_rmt_str_printf("t0: "); _rmt_hex_printf(read_gp_register($8));
	_rmt_str_printf("t1: "); _rmt_hex_printf(read_gp_register($9));
	_rmt_str_printf("t2: "); _rmt_hex_printf(read_gp_register($10));
	_rmt_str_printf("t3: "); _rmt_hex_printf(read_gp_register($11));
	_rmt_str_printf("t4: "); _rmt_hex_printf(read_gp_register($12));
	_rmt_str_printf("t5: "); _rmt_hex_printf(read_gp_register($13));
	_rmt_str_printf("t6: "); _rmt_hex_printf(read_gp_register($14));
	_rmt_str_printf("t7: "); _rmt_hex_printf(read_gp_register($15));
	_rmt_str_printf("s0: "); _rmt_hex_printf(read_gp_register($16));
	_rmt_str_printf("s1: "); _rmt_hex_printf(read_gp_register($17));
	_rmt_str_printf("s2: "); _rmt_hex_printf(read_gp_register($18));
	_rmt_str_printf("s3: "); _rmt_hex_printf(read_gp_register($19));
	_rmt_str_printf("s4: "); _rmt_hex_printf(read_gp_register($20));
	_rmt_str_printf("s5: "); _rmt_hex_printf(read_gp_register($21));
	_rmt_str_printf("s6: "); _rmt_hex_printf(read_gp_register($22));
	_rmt_str_printf("s7: "); _rmt_hex_printf(read_gp_register($23));
	_rmt_str_printf("t8: "); _rmt_hex_printf(read_gp_register($24));
	_rmt_str_printf("t9: "); _rmt_hex_printf(read_gp_register($25));
	_rmt_str_printf("k0: "); _rmt_hex_printf(read_gp_register($26));
	_rmt_str_printf("k1: "); _rmt_hex_printf(read_gp_register($27));
	_rmt_str_printf("gp: "); _rmt_hex_printf(read_gp_register($28));
	_rmt_str_printf("sp: "); _rmt_hex_printf(read_gp_register($29));
	_rmt_str_printf("fp: "); _rmt_hex_printf(read_gp_register($30));
	_rmt_str_printf("ra: "); _rmt_hex_printf(read_gp_register($31));
#endif /* NEVER */
}
