#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/init.h>

#include <linux/kernel.h>	/* printk() */
#include <linux/slab.h>		/* kmalloc() */
#include <linux/fs.h>		/* everything... */
#include <linux/errno.h>	/* error codes */
#include <linux/types.h>	/* size_t */
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/cdev.h>

#include <asm/uaccess.h>	/* copy_*_user */


//#include <asm-rmt/rtthread.h>
#include <linux/unistd.h>
#include <asm/sysmips.h>
//#include <machine/rtthread.h>
#include <linux/poll.h>
#include "rldata.h"		/* local definitions */


/*
 * Our parameters which can be set at load time.
 */

int rldata_major = RLDATA_MAJOR;
int rldata_minor = 0;

int rldata_nr_devs = RLDATA_NR_DEVS;	/* number of bare rldata devices */

module_param(rldata_major, int, S_IRUGO);
module_param(rldata_minor, int, S_IRUGO);
module_param(rldata_nr_devs, int, S_IRUGO);

MODULE_AUTHOR("Axe");
MODULE_LICENSE("");

struct rldata_dev *rldata_devices;	/* allocated in rldata_init_module */

#define INSTRUCTION_LEVEL
#define SERIAL_MODE

#define RTBL_ADDR_BASE		0xfffe2000
#define RTBL_LINK_BASE		0xfffe3000
#define RTBL_SIZE		(256*4)

#define EVENT_DPM_IN_BASE	0xc0000000
#define EVENT_DPM_OUT_BASE	0xc4000000
#define DATA_DPM_IN_BASE	0xc8000000
#define DATA_DPM_OUT_BASE	0xcc000000
#define EVENT_DPM_IN_CTRL	0xfffef000
#define EVENT_DPM_OUT_CTRL	0xfffef400
#define DATA_DPM_IN_CTRL	0xfffef800
#define DATA_DPM_OUT_CTRL	0xfffefc00
#define RL_SDRAM_MODE		0xfffe0000
#define RL_S_REG		0xfffe0004
#define RL_INIT_REG		0xfffe0008
#define RL_IC_REG		0xfffe000c
#define RL_WAIT_IC_REG		0xfffe0010
#define RL_CONT_IC_REG		0xfffe0014
#define RL_FATAL_IC_REG		0xfffe0018
#define RL_RTBL_IC_REG		0xfffe001c
#define RL_TBL_BREQ_REG		0xfffe0028
#define RL_TBL_BGRNT_REG	0xfffe002c
#define RL_SD_BREQ_REG		0xfffe0020
#define RL_SD_BGRNT_REG		0xfffe0024
#define RL_STF			0xfffe0048
#define RL_PSEL			0xfffe0050
#define	RL_SD_INIT_REG		0xffffe000
#define	RL_SD_SIZE_REG		0xffffe004
#define	SD_INIT_REG		0xfffff000
#define	SD_SIZE_REG		0xfffff004

#define EVENT_IN_CNTL_REG	(EVENT_DPM_IN_CTRL + 0x0)
#define EVENT_IN_PACKET_NUM	(EVENT_DPM_IN_CTRL + 0x8)
#define EVENT_IN_PACKET_VALID	(EVENT_DPM_IN_CTRL + 0xc)
#define EVENT_OUT_CNTL_REG	(EVENT_DPM_OUT_CTRL + 0x0)
#define EVENT_OUT_COUNT_REG	(EVENT_DPM_OUT_CTRL + 0x4)
#define EVENT_OUT_PACKET_NUM	(EVENT_DPM_OUT_CTRL + 0x8)
#define DATA_IN_CNTL_REG	(DATA_DPM_IN_CTRL + 0x0)
#define DATA_IN_PACKET_NUM	(DATA_DPM_IN_CTRL + 0x8)
#define DATA_IN_PACKET_VALID	(DATA_DPM_IN_CTRL + 0xc)
#define DATA_OUT_CNTL_REG	(DATA_DPM_OUT_CTRL + 0x0)
#define DATA_OUT_COUNT_REG	(DATA_DPM_OUT_CTRL + 0x4)
#define DATA_OUT_PACKET_NUM	(DATA_DPM_OUT_CTRL + 0x8)

#define EVENT_PACKET_SIZE	16
#define EVENT_PACKET_WSIZE	4
#define DATA_PACKET_SIZE	64
#define DATA_PACKET_WSIZE	16
#define EVENT_DPM_DMA_DEST_ADDR	0x80100000
#define DATA_DPM_DMA_DEST_ADDR	0x80200000

#define SYSTEM_REG_OWN_ICOUNT	0xe3
#define AXE_PUTCHAR(a) __axe_write_c0_register(0x02000000 | (a))
#define AXE_STOP() __axe_write_c0_register(0x07000000)

#define __axe_write_c0_register(value)                                  \
        do {                                                            \
                __asm__ __volatile__(                                   \
                                     "nop\n"                            \
                                     "mtc0\t%0, %1\n\t"                 \
                                     "nop"                              \
                                     : : "r" ((unsigned int)value), "r"((unsigned int)0xff)); \
        } while (0)

#define TMC_ST	(1<<0)
#define TMC_IED	(1<<1)
#define TMC_IER	(1<<2)
#define TMC_S8	(1<<3)
#define TMC_S16	(1<<4)
#define TMC_8P	(1<<5)
#define TMC_16P	(1<<6)
#define TMC_32P	(1<<7)
#define TMC_MR	(1<<8)
#define TMC_MTM	(1<<9)
#define TMC_PCI	(1<<10)
#define TMC_RL	(1<<11)
#define TMC_BM	(1<<12)
#define TMC_SAU	(1<<13)
#define TMC_DAS	(1<<14)

#define DMA_SR_ER (1<<1)
#define DMA_SR_ED (1<<0)

#define EXP_BASE_ADDR 0x82000000
#define EXP_HANDLER_ADDR 0x82001000
#define SYSREG_EXP_BASE_ADDR 0x6a
#define SYSREG_OWN_STATUS 0xe0
#define SYSREG_OWN_INT_WAIT 0xea
#define STATUS_EB (1<<6)
#define STATUS_IE (1<<0)

#define	DMA_BASE	0xffff2000
#define	DMA_PSA(x)	(DMA_BASE + 0x40*(x) + 0x04)
#define	DMA_MDA(x)	(DMA_BASE + 0x40*(x) + 0x08)
#define	DMA_IDR(x)	(DMA_BASE + 0x40*(x) + 0x0c)
#define	DMA_TMR(x)	(DMA_BASE + 0x40*(x) + 0x10)
#define	DMA_SR(x)	(DMA_BASE + 0x40*(x) + 0x14)
#define	DMA_LR(x)	(DMA_BASE + 0x40*(x) + 0x18)
#define	DMA_PRI		(DMA_BASE + 0x800)
#define	DMA_IRQ		(DMA_BASE + 0x804)
/*
  DMAC0: Event In-DPM
  DMAC1: Data In-DPM
  DMAC2: Event Out-DPM
  DMAC3: Data Out-DPM
*/

/*
 *  for IRC
 */
#define IRQ_DMAC2	26
#define IRQ_LINK	28
#define	MAIN_IRC_BASE	0xffff9000	
#define	RL_IRC_BASE	0xfffe1000	
#define IRC_TMR0(base)  (base + 0x00)
#define IRC_TMR1(base)  (base + 0x04)
#define IRC_RSR(base)   (base + 0x08)
#define IRC_RCR(base)   (base + 0x0c)
#define IRC_MR(base)    (base + 0x10)
#define IRC_ICR(base)   (base + 0x14)
#define IRC_MOD(base)   (base + 0x18)

#define RLIRQ_DA_OUTEOP	1
#define RLIRQ_EV_OUTEOP	2
#define RLIRQ_DA_INEOP	3
#define RLIRQ_EV_INEOP	4
#define RLIRQ_DAP_IN	5
#define RLIRQ_EVP_IN	6
#define RLIRQ_CONT	7
#define RLIRQ_WAIT	8
#define RLIRQ_TABLE	9
#define RLIRQ_FATAL	10
#define RLIRQ_WAKEUP	11
#define RLIRQ_DOWN	12

#define IRL_CLEAR       0x00000010
#define REQ_ALL_CLEAR   0xfffffffe
#define INTR_DISABLE    0xffffffff
#define INTR_MASKALL    0xfffffffe

#define RTBL_NUM  256
#define RTBL_SIZE RTBL_NUM * 4
#define DPM_EVENT_PACKET_NUM 8
#define DPM_DATA_PACKET_NUM 32

#define DPM_EVENT_AMASK DPM_EVENT_PACKET_NUM * EVENT_PACKET_SIZE - 1
#define DPM_DATA_AMASK DPM_DATA_PACKET_NUM * DATA_PACKET_SIZE - 1

#define DPM_CTLREG_FROMTO 0
#define DPM_CTLREG_UNSUSED 4
#define DPM_CTLREG_PCKTNUM 8

#define writel( d, a )	(*(volatile unsigned long *)(a) = (unsigned long)(d))
#define readl( a )	(*(volatile unsigned long *)(a))

/*
 * Responsive Link Register control functions 
 */
void axe_table_bus_request(void)
{
	unsigned long flag;

	writel(0x00000000, RL_TBL_BREQ_REG);

	do {
		flag = *(volatile unsigned long *) RL_TBL_BGRNT_REG;
		flag >>= 31;
	} while (flag != 0x0);
}

void axe_table_bus_release(void)
{
	writel(0x00000001, RL_TBL_BREQ_REG);
}

void axe_table_null_clear(void)
{
	int i;
	int *j;

	for (j = RTBL_LINK_BASE; j < RTBL_LINK_BASE + RTBL_NUM; j += 1)
		*j = 0x00000000;

	for (j = RTBL_ADDR_BASE; j < RTBL_ADDR_BASE + RTBL_NUM; j += 1)
		*j = 0x00000000;

	return;
}

int entry_route_num = 0;
int axe_rl_add_route(unsigned long address, unsigned short link)
{
	int i, j, k = -1, cnt = 0;

	if (entry_route_num > 255)
		return -1;
	for (i = RTBL_ADDR_BASE; i < RTBL_ADDR_BASE + RTBL_SIZE; i += 4) {
		j = readl(i);
		if (j == address) {
			writel(link, i + 0x1000);
			return 0;
		} else if (!j) {
			if (k == -1)
				k = i;
		} else {
			cnt++;
		}
		if ((cnt >= entry_route_num) && (k != -1)) {
			break;
		}
	}
	writel(address, k);
	writel(link, k + 0x1000);
	entry_route_num++;
	return 0;
}

void axe_rl_init(void)
{
	int i;
#if 0
	/* Link SDRAM  16bit x 2 */
	writel(0x00010d09, RL_SD_SIZE_REG);
	/* Link SDRAM Initialization */
	writel(0x0, RL_SD_INIT_REG);
#else
	/* Link SDRAM Mode Setting (Not use) */
	writel(0x0, RL_SDRAM_MODE);
#endif

#ifdef SERIAL_MODE
	writel(0x00000000, RL_PSEL);
#else
	writel(0x0000001f, RL_PSEL);
#endif
	writel(0x00000000, RL_S_REG);	//400M
	writel(0x00010001, RL_INIT_REG);
	writel(0x00000000, RL_INIT_REG);	
	writel(0x001e001e, RL_INIT_REG);
	writel(0x00000000, RL_INIT_REG);
	writel(0x00000000, RL_STF);
	//set IRC Registers
	writel(1, IRC_MOD(MAIN_IRC_BASE));
	writel(0, IRC_TMR0(MAIN_IRC_BASE));
	writel(0, IRC_TMR1(MAIN_IRC_BASE));
	writel((~((1 << IRQ_LINK)) & 0xfffffffe), IRC_MR(MAIN_IRC_BASE));
	writel(1, IRC_MOD(RL_IRC_BASE));
	writel(0, IRC_TMR0(RL_IRC_BASE));
	writel(0, IRC_TMR1(RL_IRC_BASE));
	writel(~0x3ff, IRC_MR(RL_IRC_BASE));	

	/* Make Routing Table */
	axe_table_bus_request();
	axe_table_null_clear();

	/* clockwise */
	/* L0 -> L1 *//* 0x00010001 -> 0x00011001 */
	axe_rl_add_route(0x00010001, 0x0000c062);
	/* L2 -> L3 *//* 0x00011001 -> 0x00012001 */
	axe_rl_add_route(0x00011001, 0x0000c0a8);
	/* L4 -> L0 *//* 0x00012001 -> 0x00013001 */
	axe_rl_add_route(0x00012001, 0x0000c0e1);

	/* counterclockwise */
	/* L0 -> L4 *//* 0x00020002 -> 0x00021002 */
	axe_rl_add_route(0x00020002, 0x0000c070);
	/* L3 -> L2 *//* 0x00021002 -> 0x00022002 */
	axe_rl_add_route(0x00021002, 0x0000c0a4);
	/* L1 -> L0 *//* 0x00022002 -> 0x00023002 */
	axe_rl_add_route(0x00022002, 0x0000c0e1);

	axe_table_bus_release();

	writel(0x00000000, DATA_DPM_OUT_CTRL);
}

/*
 * FIFO operation functions
 */
static inline unsigned int fifo_datasize(struct dpm_fifo *fifo)
{
	return (fifo->in - fifo->out);
}

int fifo_puts_wio2k(struct dpm_fifo *fifo,
		    unsigned char *buffer, unsigned int len)
{
	unsigned int l;
	unsigned int j;
	unsigned int *pnt;

	if (len & 3)
		return -1;

	len = min(len, fifo->size - fifo->in + fifo->out);
	l = min(len, fifo->size - (fifo->in & (fifo->size - 1)));

	for(j = 0;j < l;j += sizeof(unsigned long)){
		*((unsigned long *)(fifo->buffer 
                                    + (fifo->in & (fifo->size - 1)) + j)) = readl(buffer + j);
        }
        
        for(j = 0;j < (len - l);j += sizeof(unsigned long)){
                *((unsigned long *)(fifo->buffer + j)) =  readl(buffer + j + l);
        }

#ifdef CONFIG_SMP
        smp_wmb();
#else
        wmb();
#endif

	fifo->in += len;
        
	return len;

}

int fifo_puts(struct dpm_fifo *fifo,
	      unsigned char *buffer, unsigned int len)
{
	unsigned int l;
        
	len = min(len, fifo->size - fifo->in + fifo->out);
	l = min(len, fifo->size - (fifo->in & (fifo->size - 1)));
        
        if (copy_from_user(fifo->buffer + (fifo->in & (fifo->size - 1)), buffer, l)) {
                return -1;	
        }
        
	if (copy_from_user(fifo->buffer, buffer + l, len - l)) {
		return -1;
	}

#ifdef CONFIG_SMP
        smp_wmb();
#else
        wmb();
#endif

	fifo->in += len;

	return len;
}

unsigned int fifo_outpackets(struct dpm_fifo *fifo, int packetsnum)
{
	unsigned int l, j;
	unsigned int len;
	unsigned long *tmp;
        
        
	len = packetsnum * DATA_PACKET_SIZE;
        
	//len = min(len, fifo->in - fifo->out);	
        
	l = min(len, fifo->size - (fifo->out & (fifo->size - 1)));
        
	for (j = 0; j < l; j += sizeof(unsigned long)) {
		writel(*((unsigned long *) (fifo->buffer + (fifo->out & (fifo->size - 1)) + j)),
                       DATA_DPM_OUT_BASE + j);
	}
        
	for (j = 0; j < (len - l); j += sizeof(unsigned long)) {
		writel(*((unsigned long *) (fifo->buffer + j)),
		       DATA_DPM_OUT_BASE + j + l);
	}
        
#ifdef CONFIG_SMP
        smp_rmb();
#else
        rmb();
#endif

	fifo->out += len;
        
	return len;
}

unsigned int fifo_gets_k2u(struct dpm_fifo *fifo, unsigned char *buffer,
			   int len)
{
	unsigned int l, j;
	unsigned long *tmp;
        
	len = min(len, fifo->in - fifo->out);
        
	l = min(len, fifo->size - (fifo->out & (fifo->size - 1)));
        
	if (copy_to_user(buffer, 
                         fifo->buffer + (fifo->out & (fifo->size - 1)), l)) {
		return -1;
	}
        
	if (copy_to_user(buffer + l, fifo->buffer, len - l)) {
		return -1;
	}
        
#ifdef CONFIG_SMP
        smp_rmb();
#else
        rmb()
#endif

	fifo->out += len;	
        
	return len;
}

void fifo_free(struct dpm_fifo *fifo)
{
	kfree(fifo->buffer);
	kfree(fifo);
}

/*
 * File operation methods
 */
ssize_t rldata_read(struct file *filp, char __user * buf, size_t count,
		    loff_t * f_pos)
{
	struct rldata_dev *dev = filp->private_data;
	ssize_t retval = 0;
	int tmp_len;

	if (count & (DATA_PACKET_SIZE - 1)) {
		return  -EIO;
	}

	if (down_interruptible(&dev->sem_in))
		return -ERESTARTSYS;


	if (!count) {
		retval = 0;
		goto out;
	}

	while (fifo_datasize(dev->fifo_dpmin) == 0) {	
		up(&dev->sem_in);
		if (filp->f_flags & O_NONBLOCK) {
			return -EAGAIN;	
		}
		if (wait_event_interruptible(dev->read_waitq,
					     dev->fifo_dpmin->in != dev->fifo_dpmin->out)) {
			return -ERESTARTSYS;
		}
		if (down_interruptible(&dev->sem_in)) {
			return -ERESTARTSYS;	
		}
	}
        
	tmp_len = fifo_gets_k2u(dev->fifo_dpmin, buf, count);
	if (tmp_len < 0) {
		retval = -EFAULT;
		goto out;
	}

	retval = tmp_len;

out:
	up(&dev->sem_in);
	return retval;
}

ssize_t rldata_write(struct file * filp, const char __user * buf,
		     size_t count, loff_t * f_pos)
{
	struct rldata_dev *dev = filp->private_data;
	ssize_t retval = -ENOMEM;
	int *tmp;
	int err;
	unsigned int i, j;
	int tmp_len;
	int tmp_packetnum;

	if (count & (DATA_PACKET_SIZE - 1)) {
		return -EIO;
	}

	if (down_interruptible(&dev->sem_out))
		return -ERESTARTSYS;

        while (!(dev->fifo_dpmout->size 
                 - fifo_datasize(dev->fifo_dpmout))) {
		up(&dev->sem_out);
		if (filp->f_flags & O_NONBLOCK) {
			return -EAGAIN;
		}
		if (wait_event_interruptible(dev->write_waitq,
                                             (dev->fifo_dpmout->size 
                                               - fifo_datasize(dev->fifo_dpmout)))) {
                        return -ERESTARTSYS;
                }
                if (down_interruptible(&dev->sem_out)) {
                        return -ERESTARTSYS;
                }
	}
        
	tmp_len = fifo_puts(dev->fifo_dpmout, buf, count);
        
	if (tmp_len < 0) {
		retval = -EFAULT;
		goto out;
	}
	count = (size_t) tmp_len;
        
	*f_pos = 0;
	dev->size = *f_pos;
        
	retval = count;
        
 out:
	up(&dev->sem_out);
	return retval;
}

unsigned int rldata_poll(struct file *filp, poll_table * wait)
{
	struct rldata_dev *dev = filp->private_data;
	unsigned int mask;

	poll_wait(filp, &dev->write_waitq, wait);
	poll_wait(filp, &dev->read_waitq, wait);

	mask = 0;
	if (dev->fifo_dpmout->size > fifo_datasize(dev->fifo_dpmout)) {
		mask |= POLLOUT | POLLWRNORM;
	}
	if (fifo_datasize(dev->fifo_dpmin)) {
		mask |= POLLIN | POLLRDNORM;
	}
	return mask;
}

int rldata_open(struct inode *inode, struct file *filp)
{
	struct rldata_dev *dev;	
        
	dev = container_of(inode->i_cdev, struct rldata_dev, cdev);
	filp->private_data = dev;

	return 0;
}

int rldata_release(struct inode *inode, struct file *filp)
{
	return 0;
}

static int rldata_ioctl(struct inode *inode, struct file *filp,
			unsigned int cmd, unsigned long arg)
{
	struct rldata_dev *dev = filp->private_data;
	int retval;
        int err = 0;

        if (_IOC_TYPE(cmd) != RLDATA_IOC_MAGIC) return -ENOTTY;
        if (_IOC_NR(cmd) > RLDATA_IOC_MAXNR) return -ENOTTY;

        if(_IOC_DIR(cmd) & _IOC_READ){
                err = !access_ok(VERYFY_WRITE, 
                                 (void __user *)arg, _IOC_SIZE(cmd));
        }
        if (err) return -EFAULT;

	retval = 0;
	switch (cmd) {
	case RLDATA_PUTOSIZE:
		if (arg < RLDATA_FIFO_SZMIN
                    || arg > RLDATA_FIFO_SZMAX) { 
			return -EINVAL;
		}

		if (down_interruptible(&dev->sem_out))
			return -ERESTARTSYS;
                
		if (fifo_datasize(dev->fifo_dpmout) == 0) {
			if (resize_fifo(dev->fifo_dpmout, arg)) {
				retval = -ENOMEM;
			}else{
                                dev->sz_fifoout = arg;
                        }
		} else {
			retval = -EBUSY;
		}

		up(&dev->sem_out);
		break;
	case RLDATA_PUTISIZE:
		if (arg < RLDATA_FIFO_SZMIN
                    || arg > RLDATA_FIFO_SZMAX) { 
			return -EINVAL;
		}

		if (down_interruptible(&dev->sem_in))
			return -ERESTARTSYS;

		spin_lock(&dev->fifo_dpmin->lock);

		if (fifo_datasize(dev->fifo_dpmin) == 0) {
			if (resize_fifo(dev->fifo_dpmin, arg)) {
				retval = -ENOMEM;
			}else{
                                dev->sz_fifoin = arg;
                        }
		} else {
			retval = -EBUSY;
		}
		spin_unlock(&dev->fifo_dpmin->lock);
                
		up(&dev->sem_in);
		break;
	case RLDATA_GETOSIZE:
		if (down_interruptible(&dev->sem_out))
			return -ERESTARTSYS;

                retval = __put_user(dev->sz_fifoout, (int __user *)arg);
		up(&dev->sem_out);
		break;
	case RLDATA_GETISIZE:
		if (down_interruptible(&dev->sem_in))
			return -ERESTARTSYS;

                retval = __put_user(dev->sz_fifoin, (int __user *)arg);
		up(&dev->sem_in);
		break;

	default:
		return -ENOSYS;
	}

	return retval;
}

struct file_operations rldata_fops = {
	.owner = THIS_MODULE,
	.ioctl = rldata_ioctl,
	.read = rldata_read,
	.write = rldata_write,
	.open = rldata_open,
	.release = rldata_release,
	.poll = rldata_poll
};

static void add_tv(struct timeval *tv, long sec, long usec)
{
	int carry;

	if (usec < 0 || sec < 0) {
		printk("add_tv: bad add value (sec=%d usec=%06d)\n",
		       sec, usec);
		return;
	}

	usec += tv->tv_usec;
	sec += usec / 1000000;
	usec = usec % 1000000;

	tv->tv_sec += sec;
	tv->tv_usec = usec;
}

//#define MON_NEXT_AVERAGE
static int rldata_rtthd_poldpm(void *unused)
{
	int curpacknum_read;
	int curpacknum_write;
	int nxtpacknum_read;
	int nxtpacknum_write;
	int snd_endpoint;
	int rval;
	int tmp;
	int tmp_len;
	struct timeval start, end;
	unsigned period;
	int max_next_rval = 0;
#ifdef MON_NEXT_AVERAGE
	int counter = 0;
	int next_rval = 0;
	int average_next_rval = 0;
#endif

	//printk("Entered to rldata_rtthd_poldpm.\n", rval);
	curpacknum_read = readl(DATA_DPM_IN_CTRL + DPM_CTLREG_PCKTNUM);
	curpacknum_write = readl(DATA_DPM_OUT_CTRL + DPM_CTLREG_PCKTNUM);
	snd_endpoint = curpacknum_write;

 restart:
	do_gettimeofday(&start);
	end = start;
	add_tv(&start, 0, 0);	
	add_tv(&end, 86400, 5);	//1Day
	period = 10000; 
	rval = do_rtthread_start_periodic_thread(&start, &end, period);
	//printk("rldata: real time thread is waken up. rval= %d\n", rval);
#ifdef MON_NEXT_AVERAGE
        printk("&counter = 0x%08x\n", &counter);
        printk("&average_next_rval = 0x%08x\n", &average_next_rval);
        printk("&max_next_rval = 0x%08x\n", &max_next_rval);        
#endif
	do {
		nxtpacknum_read =
                        readl(DATA_DPM_IN_CTRL + DPM_CTLREG_PCKTNUM);
                
		if (nxtpacknum_read != curpacknum_read) {
			tmp = nxtpacknum_read - curpacknum_read;

			if (tmp > 0) {
				tmp = tmp * DATA_PACKET_SIZE;
			} else {
				tmp = (DPM_DATA_PACKET_NUM + tmp) * DATA_PACKET_SIZE;
			}
                        
			spin_lock(&rldata_devices->fifo_dpmin->lock);
                        
			rval = fifo_puts_wio2k(rldata_devices->fifo_dpmin,
                                               DATA_DPM_IN_BASE + curpacknum_read * DATA_PACKET_SIZE, tmp);
                        
			spin_unlock(&rldata_devices->fifo_dpmin->lock);
                        
			if (rval != tmp) {
				printk("fifo_dpmin is full! rval = %d, tmp=%d\n", rval, tmp);
			}
                        
			if (rval > 0) {
				wake_up_interruptible(&rldata_devices->read_waitq);
			}
			curpacknum_read = nxtpacknum_read;
		}
                
		nxtpacknum_write = readl(DATA_DPM_OUT_CTRL + DPM_CTLREG_PCKTNUM);
                
		tmp_len = (signed int) fifo_datasize(rldata_devices->fifo_dpmout);

		if (tmp_len >= DATA_PACKET_SIZE) {
			tmp_len = tmp_len / DATA_PACKET_SIZE;
			if (nxtpacknum_write == snd_endpoint) {
                                // tmp_len <= nxtpacknum_write) {	
                                tmp_len = min(tmp_len, DPM_DATA_PACKET_NUM / 2 - 1);
				writel(tmp_len * 0x10, DATA_DPM_OUT_CTRL);
				fifo_outpackets(rldata_devices->fifo_dpmout, tmp_len);
				snd_endpoint = tmp_len;
				wake_up_interruptible(&rldata_devices->write_waitq);
			}
		}
                
		curpacknum_write = nxtpacknum_write;

#ifdef MON_NEXT_AVERAGE
		tmp = do_rtthread_next_period();	
		if (tmp < 0)
			goto restart;
		next_rval += tmp;
		if (tmp > max_next_rval) {
			max_next_rval = next_rval;
		}
		if (!(counter++ & 63)) {
			average_next_rval = next_rval >> 5;
			next_rval = 0;
		}

#else
		if (do_rtthread_next_period() < 0)
			goto restart;
#endif

	} while (1);

}

static int resize_fifo(struct dpm_fifo *tag_fifo, int sz_pow)
{
	int tmp_size;
	char *tmp_buf;


	tmp_size = sizeof(char) * (1 << sz_pow);

	tmp_buf = kmalloc(tmp_size, GFP_KERNEL);
	if (!tmp_buf) {
		return -1;
	}

	kfree(tag_fifo->buffer);
	tag_fifo->size = tmp_size;
	tag_fifo->buffer = tmp_buf;
	tag_fifo->in = tag_fifo->out = 0;

	return 0;
}



void rldata_cleanup_module(void)
{
	int i;
	dev_t devno = MKDEV(rldata_major, rldata_minor);

	if (rldata_devices) {
                //free FIFO( DPM_IN/_OUT)
                fifo_free(&rldata_devices->fifo_dpmout);
                fifo_free(&rldata_devices->fifo_dpmin);
                //
		cdev_del(&rldata_devices->cdev);
		kfree(rldata_devices);
	}

	unregister_chrdev_region(devno, rldata_nr_devs);
}


/*
 * Set up the char_dev structure for this device.
 */
static int rldata_setup_cdev(struct rldata_dev *dev, int index)
{
	int err, devno = MKDEV(rldata_major, rldata_minor + index);
	struct dpm_fifo *tmp_dpm_fifo;

	cdev_init(&dev->cdev, &rldata_fops);
	dev->cdev.owner = THIS_MODULE;
	dev->cdev.ops = &rldata_fops;

        /* initialize FIFO(for DPM_OUT)*/
	dev->sz_fifoout = RLDATA_FIFO_SZPOW;
	dev->fifo_dpmout = 
                kmalloc(sizeof(struct dpm_fifo), GFP_KERNEL);
	if (!dev->fifo_dpmout) {
		err = -ENOMEM;
		goto fail;
	}
	dev->fifo_dpmout->size =
                sizeof(char) * (1 << dev->sz_fifoout);
	dev->fifo_dpmout->buffer =
                kmalloc(dev->fifo_dpmout->size, GFP_KERNEL);
	if (!dev->fifo_dpmout->buffer) {
		err = -ENOMEM;
		goto fail;
	}
	dev->fifo_dpmout->in = dev->fifo_dpmout->out = 0;
	spin_lock_init(&dev->fifo_dpmout->lock);

        /* initialize FIFO(for DPM_IN) */
	dev->sz_fifoin = RLDATA_FIFO_SZPOW;
	dev->fifo_dpmin = 
                kmalloc(sizeof(struct dpm_fifo), GFP_KERNEL);
	if (!dev->fifo_dpmin) {
		err = -ENOMEM;
		goto fail;
	}
	dev->fifo_dpmin->size =
                sizeof(char) * (1 << dev->sz_fifoin);
	dev->fifo_dpmin->buffer =
                kmalloc(dev->fifo_dpmin->size, GFP_KERNEL);
	if (!dev->fifo_dpmin->buffer) {
		err = -ENOMEM;
		goto fail;
	}
	dev->fifo_dpmin->in = dev->fifo_dpmin->out = 0;
	spin_lock_init(&dev->fifo_dpmin->lock);

	err = cdev_add(&dev->cdev, devno, 1);

fail:
	if (err) printk(KERN_NOTICE "Error %d adding rldata%d", err, index);

        return err;
}


int rldata_init_module(void)
{
	int result, i;
	int pid;
	dev_t dev = 0;

        /*
         * Get a range of minor numbers to work with, asking for a dynamic
         * major unless directed otherwise at load time.
         */
	if (rldata_major) {
		dev = MKDEV(rldata_major, rldata_minor);
		result = register_chrdev_region(dev, rldata_nr_devs, "rldata");
	} else {
		result = alloc_chrdev_region(&dev, rldata_minor, 
                                             rldata_nr_devs, "rldata");
		rldata_major = MAJOR(dev);
	}

	if (result < 0) {
		printk(KERN_WARNING "rldata: can't get major %d\n",
		       rldata_major);
		return result;
	}

	/* 
	 * allocate the devices -- we can't have them static, as the number
	 * can be specified at load time
	 */

	rldata_devices =
                kmalloc(rldata_nr_devs * sizeof(struct rldata_dev),
                        GFP_KERNEL);
	if (!rldata_devices) {
		result = -ENOMEM;
		goto fail;
	}

	memset(rldata_devices, 0,
	       rldata_nr_devs * sizeof(struct rldata_dev));

        /*
         * initialize Responsive Link registers.
         */
	axe_rl_init();
        
	init_waitqueue_head(&rldata_devices->write_waitq);
	init_waitqueue_head(&rldata_devices->read_waitq);

	init_MUTEX(&rldata_devices->sem_in);
	init_MUTEX(&rldata_devices->sem_out);

	result = rldata_setup_cdev(rldata_devices, 0);
        if(result){
                goto fail;
        }

        /* start RT-Thread */
	pid = kernel_thread(rldata_rtthd_poldpm, NULL, CLONE_KERNEL);
	if (!pid) {
		printk(KERN_WARNING
		       "rldata: Failed to start rtthd_poldpm\n");
                result = pid;
		goto fail;
	}

	return 0;

 fail:
	rldata_cleanup_module();
	return result;
}

module_init(rldata_init_module);
module_exit(rldata_cleanup_module);

