#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/init.h>

#include <linux/kernel.h>	/* printk() */
#include <linux/slab.h>		/* kmalloc() */
#include <linux/fs.h>		/* everything... */
#include <linux/errno.h>	/* error codes */
#include <linux/types.h>	/* size_t */
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/cdev.h>
#include <linux/string.h>

#include <asm/uaccess.h>	/* copy_*_user */


#include <linux/unistd.h>
#include <asm/sysmips.h>
#include <linux/poll.h>
#include "rlmx.h"		/* local definitions */


/*
 * Our parameters which can be set at load time.
 */

int rldata_major = RLDATA_MAJOR;
int rldata_minor = 0;

int rldata_nr_devs = RLDATA_NR_DEVS;	/* number of bare rldata devices */
int rlmx_num_port = RLMX_PORT_MAX;

module_param(rldata_major, int, S_IRUGO);
module_param(rldata_minor, int, S_IRUGO);
module_param(rldata_nr_devs, int, S_IRUGO);

MODULE_AUTHOR("AXE,Inc.");
MODULE_LICENSE("");

struct rldata_dev *rldata_devices;	/* allocated in rldata_init_module */

struct rlmx_dev_info *rlmx_dev_info; 

#define INSTRUCTION_LEVEL
#define SERIAL_MODE

#define RTBL_ADDR_BASE		0xfffe2000
#define RTBL_LINK_BASE		0xfffe3000
#define RTBL_SIZE		(256*4)

#define EVENT_DPM_IN_BASE	0xc0000000
#define EVENT_DPM_OUT_BASE	0xc4000000
#define DATA_DPM_IN_BASE	0xc8000000
#define DATA_DPM_OUT_BASE	0xcc000000
#define EVENT_DPM_IN_CTRL	0xfffef000
#define EVENT_DPM_OUT_CTRL	0xfffef400
#define DATA_DPM_IN_CTRL	0xfffef800
#define DATA_DPM_OUT_CTRL	0xfffefc00
#define RL_SDRAM_MODE		0xfffe0000
#define RL_S_REG		0xfffe0004
#define RL_INIT_REG		0xfffe0008
#define RL_IC_REG		0xfffe000c
#define RL_WAIT_IC_REG		0xfffe0010
#define RL_CONT_IC_REG		0xfffe0014
#define RL_FATAL_IC_REG		0xfffe0018
#define RL_RTBL_IC_REG		0xfffe001c
#define RL_TBL_BREQ_REG		0xfffe0028
#define RL_TBL_BGRNT_REG	0xfffe002c
#define RL_SD_BREQ_REG		0xfffe0020
#define RL_SD_BGRNT_REG		0xfffe0024
#define RL_STF			0xfffe0048
#define RL_PSEL			0xfffe0050
#define	RL_SD_INIT_REG		0xffffe000
#define	RL_SD_SIZE_REG		0xffffe004
#define	SD_INIT_REG		0xfffff000
#define	SD_SIZE_REG		0xfffff004


#define EVENT_DPM_DMA_DEST_ADDR	0x80100000
#define DATA_DPM_DMA_DEST_ADDR	0x80200000

#define SYSTEM_REG_OWN_ICOUNT	0xe3
#define AXE_PUTCHAR(a) __axe_write_c0_register(0x02000000 | (a))
#define AXE_STOP() __axe_write_c0_register(0x07000000)
#define AXE_MON() __axe_write_c0_register(0x01234567)

#define __axe_write_c0_register(value)                                  \
        do {                                                            \
                __asm__ __volatile__(                                   \
                                     "nop\n"                            \
                                     "mtc0\t%0, %1\n\t"                 \
                                     "nop"                              \
                                     : : "r" ((unsigned int)value), "r"((unsigned int)0xff)); \
        } while (0)

#define TMC_ST	(1<<0)
#define TMC_IED	(1<<1)
#define TMC_IER	(1<<2)
#define TMC_S8	(1<<3)
#define TMC_S16	(1<<4)
#define TMC_8P	(1<<5)
#define TMC_16P	(1<<6)
#define TMC_32P	(1<<7)
#define TMC_MR	(1<<8)
#define TMC_MTM	(1<<9)
#define TMC_PCI	(1<<10)
#define TMC_RL	(1<<11)
#define TMC_BM	(1<<12)
#define TMC_SAU	(1<<13)
#define TMC_DAS	(1<<14)

#define DMA_SR_ER (1<<1)
#define DMA_SR_ED (1<<0)

#define EXP_BASE_ADDR 0x82000000
#define EXP_HANDLER_ADDR 0x82001000
#define SYSREG_EXP_BASE_ADDR 0x6a
#define SYSREG_OWN_STATUS 0xe0
#define SYSREG_OWN_INT_WAIT 0xea
#define STATUS_EB (1<<6)
#define STATUS_IE (1<<0)

#define	DMA_BASE	0xffff2000
#define	DMA_PSA(x)	(DMA_BASE + 0x40*(x) + 0x04)
#define	DMA_MDA(x)	(DMA_BASE + 0x40*(x) + 0x08)
#define	DMA_IDR(x)	(DMA_BASE + 0x40*(x) + 0x0c)
#define	DMA_TMR(x)	(DMA_BASE + 0x40*(x) + 0x10)
#define	DMA_SR(x)	(DMA_BASE + 0x40*(x) + 0x14)
#define	DMA_LR(x)	(DMA_BASE + 0x40*(x) + 0x18)
#define	DMA_PRI		(DMA_BASE + 0x800)
#define	DMA_IRQ		(DMA_BASE + 0x804)
/*
  DMAC0: Event In-DPM
  DMAC1: Data In-DPM
  DMAC2: Event Out-DPM
  DMAC3: Data Out-DPM
*/


#define RTBL_NUM  256
#define RTBL_SIZE RTBL_NUM * 4
#define DPM_EVENT_PACKET_NUM 8
#define DPM_DATA_PACKET_NUM 32

#define DPM_EVENT_AMASK DPM_EVENT_PACKET_NUM * EVENT_PACKET_SIZE - 1
#define DPM_DATA_AMASK DPM_DATA_PACKET_NUM * DATA_PACKET_SIZE - 1

#define DPM_CTLREG_FROMTO 0
#define DPM_CTLREG_UNSUSED 4
#define DPM_CTLREG_PCKTNUM 8
#define RLMX_PROC
#define RLMX_CONNECT_PKT
#define RLMX_NOACK_ANYPORT
#define RLMX_ACKCONSEQ_CHECK
#define RLMX_CLSCONSEQ_CHECK
#define DEBUG

#define writel( d, a )	(*(volatile unsigned long *)(a) = (unsigned long)(d))
#define readl( a )	(*(volatile unsigned long *)(a))
#define AXE_EXIT() __axe_write_c0_register(0x07000000);

#ifdef DEBUG
#define RLMX_ASSERT(b,str)
#else
#define RLMX_ASSERT(b,str) {if(!b){printk("RLMX_ASSERT : %s \n%s\n(line:%d)\n", #b, str, __LINE__); AXE_EXIT(); }}
#endif

/*
 * Responsive Link Register control functions 
 */
void axe_table_bus_request(void)
{
	unsigned long flag;

	writel(0x00000000, RL_TBL_BREQ_REG);

	do {
		flag = *(volatile unsigned long *) RL_TBL_BGRNT_REG;
		flag >>= 31;
	} while (flag != 0x0);
}

void axe_table_bus_release(void)
{
	writel(0x00000001, RL_TBL_BREQ_REG);
}

void axe_table_null_clear(void)
{
	int i;
	int *j;

	for (j = RTBL_LINK_BASE; j < RTBL_LINK_BASE + RTBL_NUM; j += 1)
		*j = 0x00000000;

	for (j = RTBL_ADDR_BASE; j < RTBL_ADDR_BASE + RTBL_NUM; j += 1)
		*j = 0x00000000;

	return;
}

int entry_route_num = 0;
int axe_rl_add_route(unsigned long address, unsigned short link)
{
	int i, j, k = -1, cnt = 0;

	if (entry_route_num > 255)
		return -1;
	for (i = RTBL_ADDR_BASE; i < RTBL_ADDR_BASE + RTBL_SIZE; i += 4) {
		j = readl(i);
		if (j == address) {
			writel(link, i + 0x1000);
			return 0;
		} else if (!j) {
			if (k == -1)
				k = i;
		} else {
			cnt++;
		}
		if ((cnt >= entry_route_num) && (k != -1)) {
			break;
		}
	}
	writel(address, k);
	writel(link, k + 0x1000);
	entry_route_num++;
	return 0;
}

void axe_rl_init(void)
{
	int i;
#if 0
	/* Link SDRAM  16bit x 2 */
	writel(0x00010d09, RL_SD_SIZE_REG);
	/* Link SDRAM Initialization */
	writel(0x0, RL_SD_INIT_REG);
#else
	/* Link SDRAM Mode Setting (Not use) */
	writel(0x0, RL_SDRAM_MODE);
#endif

#ifdef SERIAL_MODE
	writel(0x00000000, RL_PSEL);
#else
	writel(0x0000001f, RL_PSEL);
#endif
	writel(0x00000000, RL_S_REG);	/*400M*/
	writel(0x00010001, RL_INIT_REG);
	writel(0x00000000, RL_INIT_REG);	
	writel(0x001e001e, RL_INIT_REG);
	writel(0x00000000, RL_INIT_REG);
	writel(0x00000000, RL_STF);
	/*set IRC Registers*/
	writel(1, IRC_MOD(MAIN_IRC_BASE));
	writel(0, IRC_TMR0(MAIN_IRC_BASE));
	writel(0, IRC_TMR1(MAIN_IRC_BASE));
	writel((~((1 << IRQ_LINK)) & 0xfffffffe), IRC_MR(MAIN_IRC_BASE));
	writel(1, IRC_MOD(RL_IRC_BASE));
	writel(0, IRC_TMR0(RL_IRC_BASE));
	writel(0, IRC_TMR1(RL_IRC_BASE));
	writel(~0x3ff, IRC_MR(RL_IRC_BASE));	

	/* Make Routing Table */
	axe_table_bus_request();
	axe_table_null_clear();

	/* Loop Back */
	/*L0 -> L0  0x00010001 -> 0x00013001*/
	axe_rl_add_route(0x00010001, 0x0000c0e1);

	/* clockwise */
	/* L0 -> L1 *//* 0x00010001 -> 0x00011001 */
	/*axe_rl_add_route(0x00010001, 0x0000c062);*/
	/*axe_rl_add_route(0x00010001, 0x0000c021);*/

	/* L2 -> L3 *//* 0x00011001 -> 0x00012001 */
  	axe_rl_add_route(0x00011001, 0x0000c0a8);
	/* L4 -> L0 *//* 0x00012001 -> 0x00013001 */
	axe_rl_add_route(0x00012001, 0x0000c0e1);

	/* counterclockwise */
	/*L0 -> L4 *//* 0x00020002 -> 0x00021002 */
	axe_rl_add_route(0x00020002, 0x0000c070);
	/*L3 -> L2 *//* 0x00021002 -> 0x00022002 */
	axe_rl_add_route(0x00021002, 0x0000c0a4);
	/*L1 -> L0 *//* 0x00022002 -> 0x00023002 */
	axe_rl_add_route(0x00022002, 0x0000c0e1);

	axe_table_bus_release();

	writel(0x00000000, DATA_DPM_OUT_CTRL);
}

/*
 * FIFO operation functions
 */
static inline unsigned int fifo_datasize(struct dpm_fifo *fifo)
{
	return (fifo->in - fifo->out);
}

static inline unsigned int fifo_leftsize(struct dpm_fifo *fifo){
        return (fifo->size - fifo->in + fifo->out);
}
static inline unsigned int fifo_btmspc(struct dpm_fifo *fifo){
        return (fifo->size - (fifo->in & (fifo->size - 1)));
}

void dumpout_packet(unsigned int *buffer) 
{
        int j;
        
        for (j = 0; j < DATA_PACKET_WSIZE; j++) {
                printk("[%d]: 0x%08x\n", j, readl(buffer + j));
        }
}


unsigned int copydpm_and_countdata(struct rldata_dev *dev,
                                     unsigned int *buffer, 
                                     unsigned int len)
{
	unsigned int i, j, k;
        unsigned int tmp, tmp2;
        unsigned int port, seq, pkttyp;
        unsigned int src_port, src_host;
        unsigned int bufuse;
        unsigned int *ack, *rbuf;
        unsigned int num_nonack = 0;
       
        struct rlmx_dev_info *dev_mx;

        for (i = 0; 
             i < (len / 4); i += DATA_PACKET_WSIZE) {
                /* printk("Receive Pakcet:\n"); */
                /* dumpout_packet(buffer + i);  */
                             
                tmp = readl(buffer + i + RLMX_PKTIX_CTLSTAT);
                if (tmp & (1 << RL_CSBIT_FATAL)) {
                        rldata_devices->err_cnt.invlpkt_cnt ++;
                        continue;
                }

                tmp = readl(buffer + i + RLMX_PKTIX_ADDR);
                src_host = (tmp >> RLDATA_ADRBIT) & RLDATA_ADRSIGMASK;      
                tmp = readl(buffer + i + RLMX_PKTIX_PORT) ; 
                port = (tmp & RLMX_PORTMASK);
                src_port = (tmp >> RLMX_PORTBIT) & RLMX_PORTMASK;

                if (port >= RLMX_PORT_MAX) {
                        rldata_devices->err_cnt.port_num ++;
                        printk("Illegal port number(%d) is found.\n", port);
                        dumpout_packet(buffer + i);
                        continue;
                }      

                dev_mx = rlmx_dev_info + port; 
                
                /*Check this port is active?*/
                tmp = atomic_read(&dev_mx->active);
                if (tmp != RLMX_CONACTIVE) {
                        continue;
                }                        
                        
                if (!dev_mx->any_port
                    && atomic_read(&dev_mx->active) == RLMX_CONACTIVE) {
                        /*Check src addr/port */
                        tmp = 0;
                        if (dev_mx->rl_peerhost != src_host) {
                                tmp ++;
                        }
                        if (dev_mx->rl_peerport != src_port) {
                                tmp ++;
                        }
                        if (tmp) {
                                dev_mx->err_cnt.unexpect ++;
                                printk("rl_mx_port[%d]:", port);
                                printk("Packet from unexpected host or port.(host: 0x%08x, port: 0x%08x)\n", 
                                       src_host, src_port);
                                printk("peerhost: 0x%08x, peerport: 0x%08x\n",
                                       dev_mx->rl_peerhost, dev_mx->rl_peerport);
                                dumpout_packet(buffer + i);
                                continue;
                        }
                }

                tmp = readl(buffer + i + RLMX_PKTIX_TYPSEQ); 
                seq = tmp & RLMX_SEQMASK ;
                pkttyp = tmp & RLMX_TYPMASK;

                switch (pkttyp) {
                case (RLMX_TYPDAT << RLMX_SEQBIT): /*Data_PKT.*/
                        /* Check this port is active? */
                        /* tmp = atomic_read(&dev_mx->active); */
                        /* if (tmp != RLMX_CONACTIVE) { */
                        /* continue; */
                        /* }                         */
                        
                        /*Check left space of buf. */
                        bufuse = atomic_read(&dev_mx->rl_rcv.cnt);
                        if (bufuse >= dev_mx->rl_rcv.buf_size) {
                                dev_mx->err_cnt.buf_full ++;
                                printk("rl_mx port[%d] : Buffer full!\n", port);
                                continue;
                        }
                        
                        /*Check packet sequential number.*/
                        if (dev_mx->rl_rcv.nxt != seq && !dev_mx->any_port) {
                                dev_mx->err_cnt.pkt_ord ++;
                                printk("rl_mx port[%d] : Illegal DATA_PKT order!\n", port);
                                printk("expected 0x%08x, arrival 0x%08x\n", 
                                       dev_mx->rl_rcv.nxt, seq);
                                dumpout_packet(buffer + i);
                        }
                        dev_mx->rl_rcv.nxt = (seq + 1) & RLMX_SEQMASK;
                        
                        /*Copy packets to each port buffer.*/
                        rbuf = dev_mx->rl_rbuf[dev_mx->rl_rcv.tpos];
                        spin_lock(&dev_mx->rl_rcv.lock);
                        for (j = 0; j < DATA_PACKET_WSIZE; j++) {
                                *rbuf = readl(buffer + j + i);
                                rbuf ++;
                        }
                        dev_mx->rl_rcv.tpos =
                                (dev_mx->rl_rcv.tpos + 1) % dev_mx->rl_rcv.buf_size;
                        atomic_inc(&dev_mx->rl_rcv.cnt);
                        dev_mx->rl_rcv.pnum ++;
                        spin_unlock(&dev_mx->rl_rcv.lock);
                        
                        wake_up_interruptible(&dev_mx->rl_rcv.rwait);
                        
                        num_nonack++;
                        break;
                        
                case (RLMX_TYPACK << RLMX_SEQBIT): /*Ack packet*/

#ifdef RLMX_ACKCONSEQ_CHECK
                        /*Ignore ack packet whose connection seq-numbers are invalid.*/
                        tmp = readl(buffer + i + RLMX_PKTIX_CONSEQ); 
                        if (tmp != dev_mx->rl_peerseq && !dev_mx->any_port) {
                                printk("rcvseq = 0x%08x, peerseq = 0x%08x\n",
                                       tmp, dev_mx->rl_peerseq); 
                                printk("Receive Pakcet:\n");
                                dumpout_packet(buffer + i);
                                continue;
                        }
#endif

                        if (seq != dev_mx->rl_snd.nxt_ack && !dev_mx->any_port) {
                                dev_mx->err_cnt.ack_ord ++;
                                printk("rl_mx port[%d] : Illegal ACK_PKT order!\n", port);
                                printk("expected 0x%08x, arrival 0x%08x\n",
                                       dev_mx->rl_snd.nxt_ack, seq);
                                printk("Received ACK packet\n");
                                dumpout_packet(buffer + i);
                                }
                        dev_mx->rl_snd.nxt_ack = (seq + 1) & RLMX_SEQMASK;


#ifdef RLMX_NOACK_ANYPORT
                        if (!dev_mx->any_port) {
                                atomic_inc(&dev_mx->rl_snd.winsize);
                        }
#else 
                        atomic_inc(&dev_mx->rl_snd.winsize);
#endif
                        RLMX_ASSERT(atomic_read(&dev_mx->rl_snd.winsize) 
                                    >= dev_mx->rl_snd.max_winsize,
                                    "Invalid windowsie.(> max_size)");

                        wake_up_interruptible(&dev_mx->rl_snd.enqwait);
                        break;
                        
                case (RLMX_TYPCTL << RLMX_SEQBIT): /*Ctrl packet(close);*/
#ifdef RLMX_CLSCONSEQ_CHECK
                        tmp = readl(buffer + i + RLMX_PKTIX_CONSEQ); 
                        if (!dev_mx->any_port && tmp == dev_mx->rl_peerseq) {
#else
                        if (!dev_mx->any_port) {
#endif
                                atomic_set(&dev_mx->active, RLMX_CONDISABLE);
                                atomic_set(&dev_mx->rl_snd.winsize,
                                           dev_mx->rl_snd.max_winsize);
                                wake_up_interruptible(&dev_mx->rl_snd.enqwait);
                                wake_up_interruptible(&dev_mx->rl_rcv.rwait);
                        }
                        break;

#ifdef RLMX_CONNECT_PKT
                case (RLMX_TYPCNT << RLMX_SEQBIT): /*Connection Packet.*/
                        /*If not any-mode, packet is ignored.*/
                        if (!dev_mx->any_port) {
                                continue;
                        }
                        /*Check left space of buf. */
                        bufuse = atomic_read(&dev_mx->rl_rcv.cnt);
                        if (bufuse >= dev_mx->rl_rcv.buf_size) {
                                dev_mx->err_cnt.buf_full ++;
                                printk("rl_mx port[%d] : Buffer full!\n", port);
                                continue;
                        }
                        
                        /*Check sequential number.*/
                        if (dev_mx->rl_rcv.nxt != seq && !dev_mx->any_port) {
                                dev_mx->err_cnt.pkt_ord ++;
                                printk("rl_mx port[%d] : Illegal DATA_PKT order!\n", port);
                                printk("expected 0x%08x, arrival 0x%08x\n", 
                                       dev_mx->rl_rcv.nxt, seq);
                                dumpout_packet(buffer + i);
                        }
                        dev_mx->rl_rcv.nxt = (seq + 1) & RLMX_SEQMASK;
                        
                        /*Copy packets to each port buffer.*/
                        rbuf = dev_mx->rl_rbuf[dev_mx->rl_rcv.tpos];
                        spin_lock(&dev_mx->rl_rcv.lock);
                        for (j = 0; j < DATA_PACKET_WSIZE; j++) {
                                *rbuf = readl(buffer + j + i);
                                rbuf ++;
                        }
                        dev_mx->rl_rcv.tpos =
                                (dev_mx->rl_rcv.tpos + 1) % dev_mx->rl_rcv.buf_size;
                        atomic_inc(&dev_mx->rl_rcv.cnt);
                        dev_mx->rl_rcv.pnum ++;
                        spin_unlock(&dev_mx->rl_rcv.lock);
                        
                        wake_up_interruptible(&dev_mx->rl_rcv.rwait);
                        
                        num_nonack++;
                        break;
#endif

                default: /*non-defined packet type.*/
                        dev_mx->err_cnt.pkt_type ++;
                        printk("rl_mx port[%d] : Illegal packet type! (%d)\n", 
                               pkttyp >> RLMX_SEQBIT);
                        dumpout_packet(buffer + i);
                }
        }

	return num_nonack;
}

int fifo_puts_wio2k(struct dpm_fifo *fifo,
		    unsigned char *buffer, unsigned int len)
{
	unsigned int l;
	unsigned int j;
	unsigned int *pnt;

	if (len & 3)
		return -1;

	len = min(len, fifo->size - fifo->in + fifo->out);
	l = min(len, fifo->size - (fifo->in & (fifo->size - 1)));

	for(j = 0;j < l;j += sizeof(unsigned long)){
		*((unsigned long *)(fifo->buffer 
                                    + (fifo->in & (fifo->size - 1)) + j)) = readl(buffer + j);
        }
        
        for(j = 0;j < (len - l);j += sizeof(unsigned long)){
                *((unsigned long *)(fifo->buffer + j)) =  readl(buffer + j + l);
        }

#ifdef CONFIG_SMP
        smp_wmb();
#else
        wmb();
#endif

	fifo->in += len;
        
	return len;

}

int fifo_puts_k2k(struct dpm_fifo *fifo,
	      unsigned char *buffer, unsigned int len)
{
	unsigned int l;
        
	len = min(len, fifo->size - fifo->in + fifo->out);
	l = min(len, fifo->size - (fifo->in & (fifo->size - 1)));

        memcpy(fifo->buffer + (fifo->in & (fifo->size - 1)), buffer, l);
	memcpy(fifo->buffer, buffer + l, len - l);

#ifdef CONFIG_SMP
        smp_wmb();
#else
        wmb();
#endif
	fifo->in += len;

	return len;
}

int fifo_puts(struct dpm_fifo *fifo,
	      unsigned char *buffer, unsigned int len)
{
	unsigned int l;
        
	len = min(len, fifo->size - fifo->in + fifo->out);
	l = min(len, fifo->size - (fifo->in & (fifo->size - 1)));
        
        if (copy_from_user(fifo->buffer + (fifo->in & (fifo->size - 1)), buffer, l)) {
                return -1;	
        }
        
	if (copy_from_user(fifo->buffer, buffer + l, len - l)) {
		return -1;
	}

#ifdef CONFIG_SMP
        smp_wmb();
#else
        wmb();
#endif

	fifo->in += len;

	return len;
}



unsigned int fifo_outpackets(struct dpm_fifo *fifo, int packetsnum)
{
	unsigned int l, j;
	unsigned int len;
	unsigned long *tmp;
        
        
	len = packetsnum * DATA_PACKET_SIZE;
        
	/*len = min(len, fifo->in - fifo->out);*/
        
	l = min(len, fifo->size - (fifo->out & (fifo->size - 1)));
        
	for (j = 0; j < l; j += sizeof(unsigned long)) {
		writel(*((unsigned long *) (fifo->buffer + (fifo->out & (fifo->size - 1)) + j)),
                       DATA_DPM_OUT_BASE + j);
	}
        
	for (j = 0; j < (len - l); j += sizeof(unsigned long)) {
		writel(*((unsigned long *) (fifo->buffer + j)),
		       DATA_DPM_OUT_BASE + j + l);
	}
        
#ifdef CONFIG_SMP
        smp_rmb();
#else
        rmb();
#endif
        
	fifo->out += len;
        
	return len;
}

unsigned int fifo_gets_k2u(struct dpm_fifo *fifo, 
                           unsigned char *buffer,
			   int len)
{
	unsigned int l, j;
	unsigned long *tmp;
        
	len = min(len, fifo->in - fifo->out);
        
	l = min(len, fifo->size - (fifo->out & (fifo->size - 1)));
        
	if (copy_to_user(buffer, 
                         fifo->buffer + (fifo->out & (fifo->size - 1)), l)) {
		return -1;
	}
        
	if (copy_to_user(buffer + l, fifo->buffer, len - l)) {
		return -1;
	}
        
#ifdef CONFIG_SMP
        smp_rmb();
#else
        rmb()
#endif

	fifo->out += len;	
        
	return len;
}

void fifo_free(struct dpm_fifo *fifo)
{
	kfree(fifo->buffer);
	kfree(fifo);
}

static int resize_fifo(struct dpm_fifo *tag_fifo, int sz_pow)
{
	int tmp_size;
	char *tmp_buf;


	tmp_size = sizeof(char) * (1 << sz_pow);

	tmp_buf = kmalloc(tmp_size, GFP_KERNEL);
	if (!tmp_buf) {
		return -1;
	}

	kfree(tag_fifo->buffer);
	tag_fifo->size = tmp_size;
	tag_fifo->buffer = tmp_buf;
	tag_fifo->in = tag_fifo->out = 0;

	return 0;
}

/*
 * File operation methods
 */
int build_packet(unsigned int *target_buffer, 
                 unsigned char *source_buffer,
                 unsigned int sendsize,
                 unsigned int pkttyp,
                 struct rlmx_dev_info *dev)
{
        unsigned int tmp;

        tmp = (unsigned long)dev->rl_priority & RLDATA_PRIOHLFMASK;
        tmp |= (((unsigned long)dev->rl_priority >> RLDATA_PRIOHLFBIT)
               & RLDATA_PRIOHLFMASK) << RLDATA_ADRBIT;
        tmp = tmp << RLDATA_ADRSIGBIT;
        tmp |= (unsigned long)dev->rl_peerhost & RLDATA_ADRSIGMASK;
        *(target_buffer + RLMX_PKTIX_ADDR) 
                = tmp | (unsigned long)(dev->root_device->myhostid 
                                        & RLDATA_ADRSIGMASK) << RLDATA_ADRBIT;
        tmp = ((unsigned long)dev->rl_peerport & RLMX_PORTMASK);
        *(target_buffer + RLMX_PKTIX_PORT)  =
                tmp | ((unsigned long)dev->rl_myport << RLMX_PORTBIT); 
        tmp = (dev->rl_snd.seq & RLMX_SEQMASK); 
        *(target_buffer + RLMX_PKTIX_TYPSEQ) =
                tmp | (pkttyp << RLMX_SEQBIT);
        tmp = (1 << RL_CSBIT_START) | (1 << RL_CSBIT_END); 
        /*tmp &= ~(RL_CSBIT_SERMASK); */
        tmp |= ((sendsize + 8) & RL_CSBIT_LENMASK)  << RL_CSBIT_DATALEN;
        *(target_buffer + RLMX_PKTIX_CTLSTAT) = 
                tmp | (1 << RL_CSBIT_FULL) * (sendsize == RLMX_MTU);

#ifdef RLMX_CLSCONSEQ_CHECK
        if (pkttyp == RLMX_TYPCTL) {
                *(target_buffer 
                  + RLMX_PKTIX_CONSEQ) = dev->rl_conseq;
        }
#endif
        if (!sendsize) {
                return 0;
        }

        if (copy_from_user((unsigned char *)(target_buffer 
                                                 + RLMX_PKTIX_PAYLOAD), 
                                    source_buffer, sendsize)) {
                return 1;
        }
        return 0;
}


ssize_t rlmx_read(struct file *filp, char __user * buf, size_t count,
		    loff_t * f_pos)
{
	struct rlmx_dev_info *dev = filp->private_data;
	ssize_t retval = 0;
        int i;
	unsigned int tmp_len, tmp, tmp2, tmp3;
        unsigned int *pnt, *ack;

        /*Check this port is active?*/
        tmp = atomic_read(&dev->active);
        if ((tmp != RLMX_CONACTIVE)
            && !atomic_read(&dev->rl_rcv.cnt)) {
                return 0;
        }

        /*If count is zero, read() returns zero and has no other results*/
	if (!count) {
		retval = 0;
		goto out;
	}

       	if (down_interruptible(&dev->sem_rcv))
		return -ERESTARTSYS;

	while (!atomic_read(&dev->rl_rcv.cnt)) {
		up(&dev->sem_rcv);
		if (filp->f_flags & O_NONBLOCK) {
			return -EAGAIN;
		}
		if (wait_event_interruptible(dev->rl_rcv.rwait,
					     atomic_read(&dev->rl_rcv.cnt))) {
			return -ERESTARTSYS;
		}

                if ((atomic_read(&dev->active) != RLMX_CONACTIVE)
                    && !atomic_read(&dev->rl_rcv.cnt)) {
                        return 0;
                }

                if (down_interruptible(&dev->sem_rcv)) {
                        return -ERESTARTSYS;
                }
        }
        
        pnt = dev->rl_rbuf[dev->rl_rcv.spos]; 
        tmp_len = ((*(pnt + RLMX_PKTIX_CTLSTAT) >> RL_CSBIT_DATALEN) /*Payload size */
                    & RL_CSBIT_LENMASK) - RLMX_SZCTRL_IN_PLOAD; 
        tmp_len = min(count, tmp_len);

	if (copy_to_user(buf, (unsigned char *)(pnt + RLMX_PKTIX_PAYLOAD), tmp_len)) {
                retval = -EFAULT;
                goto out;
        }
        dev->rl_rcv.spos =
                (dev->rl_rcv.spos + 1) % dev->rl_rcv.buf_size;
        atomic_dec(&dev->rl_rcv.cnt);
	retval = tmp_len;

#ifdef RLMX_NOACK_ANYPORT
        /* When read packet is not data packet, ACK-PKT is not returned.*/
        tmp = *(pnt + RLMX_PKTIX_TYPSEQ) & RLMX_TYPMASK;
        if (tmp != RLMX_TYPDAT << RLMX_SEQBIT) {
                goto out;
        }
#endif                
        /*Build ACK packet*/
        ack = (unsigned int *)dev->ack_work;
        for (i = 0; i < DATA_PACKET_WSIZE; i++) {
                /*printk("READ_PKT[%d]: 0x%08x\n", i, tmp);*/
                switch (i) {
                case RLMX_PKTIX_ADDR:
                        /*swap dest. for src. */
                        tmp  = *pnt;
                        tmp3 = tmp & RLDATA_PRIOMASK;
                        tmp2 = ((tmp & RLDATA_ADRSIGMASK) << RLDATA_ADRBIT);
                        tmp  = (tmp >> RLDATA_ADRBIT) & RLDATA_ADRSIGMASK;
                        tmp  = tmp | tmp2 | tmp3;
                        *ack = tmp;
                        break;                                

                case RLMX_PKTIX_PORT:
                        tmp  = *pnt;
                        tmp2 = ((tmp & RLMX_PORTSIGMASK) << RLMX_PORTBIT);
                        tmp  = (tmp >> RLMX_PORTBIT) & RLMX_PORTSIGMASK;
                        tmp  = tmp | tmp2;
                        *ack = tmp;
                        break;

                case RLMX_PKTIX_TYPSEQ:
                        tmp  = *pnt;
                        tmp &= ~RLMX_TYPMASK;
                        tmp |= (RLMX_TYPACK << RLMX_SEQBIT);
                        *ack = tmp;
                        break;

#ifdef RLMX_ACKCONSEQ_CHECK
                case RLMX_PKTIX_CONSEQ:
                        tmp =  dev->rl_conseq;
                        /*printk("peerseq = %d\n", dev->rl_peerseq);*/
                        *ack = tmp;
                        break;
#endif

                case RLMX_PKTIX_CTLSTAT: 
                        /*clear Full bit and Data Length.*/
                        tmp  = *pnt;
                        tmp &= ~((1 << RL_CSBIT_FULL) | 
                                 (RL_CSBIT_LENMASK << RL_CSBIT_DATALEN)); 
                        /*tmp &= (1 << RL_CSBIT_DATALEN);*/
                        /*data length = 12 byte*/
                        tmp |= (12 << RL_CSBIT_DATALEN);
                        *ack = tmp;
                        break;
                        
                default:
                        break;
                }
                
                pnt ++;
                ack ++;
        }

        /*Send ACK back*/
        tmp_len = 0;
        while (tmp_len != DATA_PACKET_SIZE) {
                tmp = rlmx_root_write(dev->root_device, 
                                      (unsigned char *)dev->ack_work, DATA_PACKET_SIZE);
                if (tmp == -ERESTARTSYS) {
                        continue;
                }
                tmp_len += tmp;
        }
        
 out:
	up(&dev->sem_rcv);
	return retval;
}

ssize_t rlmx_root_write(struct rldata_dev *dev, 
                        const char __user * buf, size_t count)
{
	ssize_t retval = -ENOMEM;
	int tmp_len;

	if (down_interruptible(&dev->sem_out))
		return -ERESTARTSYS;

        while (!(dev->fifo_dpmout->size 
                 - fifo_datasize(dev->fifo_dpmout))) {
		up(&dev->sem_out);
		if (wait_event_interruptible(dev->write_waitq,
                                             (dev->fifo_dpmout->size 
                                               - fifo_datasize(dev->fifo_dpmout)))) {
                        return -ERESTARTSYS;
                }
                if (down_interruptible(&dev->sem_out)) {
                        return -ERESTARTSYS;
                }
	}
        
	tmp_len = fifo_puts_k2k(dev->fifo_dpmout, buf, count);
        
	count = (size_t) tmp_len;
        
	retval = count;
 out:
	up(&dev->sem_out);
	return retval;
}


ssize_t rlmx_write(struct file * filp, const char __user * buf,
		     size_t count, loff_t * f_pos)
{
	struct rlmx_dev_info *dev = filp->private_data;
 	ssize_t retval = -ENOMEM;
        unsigned int sendsize;
        unsigned long *tmp_sbuf;
        unsigned long tmp;
        int j;

        /*Check this port is active?*/
        tmp = atomic_read(&dev->active);
        if (tmp != RLMX_CONACTIVE) {
                return -EPIPE; 
        }

        if (!count) {
                goto out; 
        }

	if (down_interruptible(&dev->sem_snd))
		return -ERESTARTSYS;

        sendsize = min((unsigned int)count, RLMX_MTU); 
        
        while (atomic_read(&dev->rl_snd.winsize) == 0) {
		up(&dev->sem_snd);
                
                if (filp->f_flags & O_NONBLOCK) {
                    return -EAGAIN;
                }
                if (wait_event_interruptible(dev->rl_snd.enqwait,
                                             (atomic_read(&dev->rl_snd.winsize) != 0))) {
                        return -ERESTARTSYS;
                }
                if (atomic_read(&dev->active) != RLMX_CONACTIVE) {
                        return -EPIPE;
                }
                
                if (down_interruptible(&dev->sem_snd))
                        return -ERESTARTSYS;
                
        }

        tmp_sbuf =
                (unsigned long *)&dev->rl_sbuf[dev->rl_snd.pos];
#ifdef RLMX_CONNECT_PKT
        if (dev->any_port) {
                tmp = RLMX_TYPDAT;
        } else {
                tmp = RLMX_TYPDAT;
        }        
        if (build_packet(tmp_sbuf, buf, 
                         sendsize, tmp, dev)) {
                retval = -EFAULT;
                goto out;
        }
#else
        if (build_packet(tmp_sbuf, buf, 
                         sendsize, RLMX_TYPDAT, dev)) {
                retval = -EFAULT;
                goto out;
        }
#endif   

        retval = 0;
        while (retval != DATA_PACKET_SIZE) {
                tmp = rlmx_root_write(dev->root_device, 
                                    (unsigned char *)tmp_sbuf, 
                                      DATA_PACKET_SIZE);
                if (tmp == -ERESTARTSYS) {
                        continue;
                }
                retval += tmp;
        }
        dev->rl_snd.seq ++;
        dev->rl_snd.pnum ++;
        dev->rl_snd.pos = (dev->rl_snd.pos + 1) % dev->rl_snd.buf_size;

#ifdef RLMX_NOACK_ANYPORT
        if (!dev->any_port) {
                atomic_dec(&dev->rl_snd.winsize);

        }
#else
        atomic_dec(&dev->rl_snd.winsize);
#endif
        RLMX_ASSERT(atomic_read(&dev_mx->rl_snd.winsize) < 0,
                    "Invalid windowsie.(< 0)");

	*f_pos = 0;
 out:
	up(&dev->sem_snd);
	return sendsize;
}


unsigned int rlmx_poll(struct file *filp, poll_table * wait)
{
	struct rlmx_dev_info *dev = filp->private_data;
	unsigned int mask, tmp;

	poll_wait(filp, &dev->rl_snd.enqwait, wait);
	poll_wait(filp, &dev->rl_rcv.rwait, wait);

	mask = 0;

        tmp = atomic_read(&dev->active);
        if (tmp != RLMX_CONACTIVE
            && !atomic_read(&dev->rl_rcv.cnt)) {
                mask |= POLLHUP;
        }

	if (atomic_read(&dev->rl_snd.winsize) > 0) {
		mask |= POLLOUT | POLLWRNORM;
	}
	if (atomic_read(&dev->rl_rcv.cnt)) {
		mask |= POLLIN | POLLRDNORM;
	}
	return mask;
}


int rldata_open(struct inode *inode, struct file *filp)
{
	struct rldata_dev *dev;	
        
	dev = container_of(inode->i_cdev, struct rldata_dev, cdev);
	filp->private_data = dev;

	return 0;
}


int rldata_release(struct inode *inode, struct file *filp)
{
	return 0;
}


int rlmx_open(struct inode *inode, struct file *filp)
{
	struct rlmx_dev_info *dev;	
        
	dev = container_of(inode->i_cdev, struct rlmx_dev_info, cdev);

        if (! atomic_dec_and_test(&dev->available)) {
                atomic_inc(&dev->available);
                return -EBUSY;
        }

	filp->private_data = dev;
        
        dev->rl_snd.seq = 0;
        dev->rl_snd.pos = 0;
        dev->rl_snd.nxt_ack = 0;
        atomic_set(&dev->rl_snd.winsize,
                   dev->rl_snd.max_winsize);

        dev->rl_rcv.nxt = 0;
        dev->rl_rcv.spos = 0;
        dev->rl_rcv.tpos = 0;
        atomic_set(&dev->rl_rcv.cnt, 0);
        
        dev->any_port = 1; 
        atomic_set(&dev->active, RLMX_CONACTIVE);
        
        /*Default destination adress/port is itself.*/
        dev->rl_peerhost = dev->root_device->myhostid;
        dev->rl_peerport = dev - rlmx_dev_info;
        dev->rl_peerseq = 0;

	return 0;
}



int rlmx_release(struct inode *inode, struct file *filp)
{
	struct rlmx_dev_info *dev;	
        unsigned long *tmp_sbuf, tmp, snd_size;

	dev = container_of(inode->i_cdev, struct rlmx_dev_info, cdev);
        
        if (!dev->any_port 
            && atomic_read(&dev->active) == RLMX_CONACTIVE)  {
                /*Send the control packet(close)*/
                if (down_interruptible(&dev->sem_snd))
                        return -ERESTARTSYS;
                tmp_sbuf =
                        (unsigned long *)&dev->rl_sbuf[dev->rl_snd.pos];
                tmp = build_packet(tmp_sbuf, 
                                   NULL, 0, RLMX_TYPCTL, dev);                
                
                snd_size = 0;
                while (snd_size != DATA_PACKET_SIZE) {
                        tmp = rlmx_root_write(dev->root_device, 
                                              (unsigned char *)tmp_sbuf, DATA_PACKET_SIZE);
                        if (tmp == -ERESTARTSYS) {
                                continue;
                        }
                        snd_size += tmp;
                }
                dev->rl_snd.seq ++;
                dev->rl_snd.pos = (dev->rl_snd.pos + 1) % dev->rl_snd.buf_size;
                /*printk("Close Pakcet:\n");*/
                /*dumpout_packet(tmp_sbuf);*/
                /*atomic_dec(&dev->rl_snd.winsize);*/
                up(&dev->sem_snd);
        }

        atomic_set(&dev->active, RLMX_CONDISABLE);
        atomic_set(&dev->rl_snd.winsize,
                   dev->rl_snd.max_winsize);
        atomic_inc(&dev->available);

        wake_up_interruptible(&dev->rl_snd.enqwait);
        wake_up_interruptible(&dev->rl_rcv.rwait);

	return 0;
}


static int rldata_ioctl(struct inode *inode, struct file *filp,
			unsigned int cmd, unsigned long arg)
{
	struct rldata_dev *dev = filp->private_data;
	int retval;
        int err = 0;

        if (_IOC_TYPE(cmd) != RLDATA_IOC_MAGIC) return -ENOTTY;
        if (_IOC_NR(cmd) > RLDATA_IOC_MAXNR) return -ENOTTY;

        if(_IOC_DIR(cmd) & _IOC_READ){
                err = !access_ok(VERYFY_WRITE, 
                                 (void __user *)arg, _IOC_SIZE(cmd));
        }
        if (err) return -EFAULT;

	retval = 0;
	switch (cmd) {
	case RLDATA_PUTOSIZE:
		if (arg < RLDATA_FIFO_SZMIN
                    || arg > RLDATA_FIFO_SZMAX) { 
			return -EINVAL;
		}

		if (down_interruptible(&dev->sem_out))
			return -ERESTARTSYS;
                
		if (fifo_datasize(dev->fifo_dpmout) == 0) {
			if (resize_fifo(dev->fifo_dpmout, arg)) {
				retval = -ENOMEM;
			}else{
                                dev->sz_fifoout = arg;
                        }
		} else {
			retval = -EBUSY;
		}

		up(&dev->sem_out);
		break;
	case RLDATA_PUTISIZE:
		if (arg < RLDATA_FIFO_SZMIN
                    || arg > RLDATA_FIFO_SZMAX) { 
			return -EINVAL;
		}

		if (down_interruptible(&dev->sem_in))
			return -ERESTARTSYS;

		spin_lock(&dev->fifo_dpmin->lock);

		if (fifo_datasize(dev->fifo_dpmin) == 0) {
			if (resize_fifo(dev->fifo_dpmin, arg)) {
				retval = -ENOMEM;
			}else{
                                dev->sz_fifoin = arg;
                        }
		} else {
			retval = -EBUSY;
		}
		spin_unlock(&dev->fifo_dpmin->lock);
                
		up(&dev->sem_in);
		break;
	case RLDATA_GETOSIZE:
		if (down_interruptible(&dev->sem_out))
			return -ERESTARTSYS;

                retval = __put_user(dev->sz_fifoout, (int __user *)arg);
		up(&dev->sem_out);
		break;
	case RLDATA_GETISIZE:
		if (down_interruptible(&dev->sem_in))
			return -ERESTARTSYS;
                retval = __put_user(dev->sz_fifoin, (int __user *)arg);
		up(&dev->sem_in);
		break;

	default:
		return -ENOSYS;
	}

	return retval;
}


static int resize_rlmxbuf(rl_pkt_t **tag_buf, u_short size)
{
	int tmp_size;
	rl_pkt_t *tmp_buf;

 	tmp_buf = kmalloc(sizeof(rl_pkt_t) * size, GFP_KERNEL); 
	if (!tmp_buf) {
		return -1;
	}

        kfree(*tag_buf); 
        *tag_buf = tmp_buf;
	return 0;
}


static int rlmx_ioctl(struct inode *inode, struct file *filp,
			unsigned int cmd, unsigned long arg)
{
	struct rlmx_dev_info *dev = filp->private_data;
        struct rlmx_ioctlport *ioctl_port;
	int retval;
        int err = 0;
        unsigned int *pnt, *pnt2;
        unsigned int val, val2, val3;
        unsigned int i, pktnum, tmp;

        if (_IOC_TYPE(cmd) != RLMX_IOC_MAGIC) return -ENOTTY;
        if (_IOC_NR(cmd) > RLMX_IOC_MAXNR) return -ENOTTY;

        if(_IOC_DIR(cmd) & _IOC_READ){
                err = !access_ok(VERYFY_WRITE, 
                                 (void __user *)arg, _IOC_SIZE(cmd));
        } else if (_IOC_DIR(cmd) & _IOC_WRITE) {
                err = !access_ok(VERYFY_READ, 
                                 (void __user *)arg, _IOC_SIZE(cmd));
        }
        
        if (err) return -EFAULT;

	retval = 0;
	switch (cmd) {
	case RLMX_IOCTL_SETDST:
                ioctl_port = (struct rlmx_ioctlport *)arg;

                if (get_user(val, (int __user *)&ioctl_port->port)) {
                        return -EFAULT;
                }
                if (get_user(val2, (int __user *)&ioctl_port->hostid)) {
                        return -EFAULT;
                }
                
                if (val >= RLMX_PORT_MAX) {
                        return -EINVAL;
                }
                if (val2 > RLDATA_ADRSIGMASK) {
                        return -EINVAL;
                }
                
		if (down_interruptible(&dev->sem_snd))
			return -ERESTARTSYS;
                /*                 while (atomic_read(&dev->rl_snd.winsize)  */
                /*                        < dev->rl_snd.max_winsize) { */
                /*                         up(&dev->sem_snd); */
                /*                         if (filp->f_flags & O_NONBLOCK) { */
                /*                                 return -EAGAIN; */
                /*                         } */
                /*                         if (wait_event_interruptible(dev->rl_snd.enqwait, */
                /*                                                      (atomic_read(&dev->rl_snd.winsize)  */
                /*                                                       >= dev->rl_snd.max_winsize))) { */
                /*                                 return -ERESTARTSYS; */
                /*                         } */
                
                /*                         if (atomic_read(&dev->active) != RLMX_CONACTIVE) { */
                /*                                 return -EPIPE; */
                /*                         } */
                
                /*                         if (down_interruptible(&dev->sem_snd)) */
                /*                                 return -ERESTARTSYS;                         */
                /*                 } */
                

                dev->rl_peerport = (u_short)(val & RLMX_PORTSIGMASK);
                dev->rl_peerhost = (u_short)(val2 & RLDATA_ADRSIGMASK);
                
                dev->rl_snd.seq = 0; 
                dev->rl_snd.nxt_ack = 0;
                /*                 dev->rl_rcv.nxt = 0; */
		up(&dev->sem_snd);
		break;

	case RLMX_IOCTL_GETDST:
		if (down_interruptible(&dev->sem_snd))
			return -ERESTARTSYS;
                ioctl_port = (struct rlmx_ioctlport *)arg;
                if (put_user((unsigned int)dev->rl_peerhost, 
                             (int __user *)&ioctl_port->hostid)) {
                        retval = -EFAULT;
                        up(&dev->sem_snd);
                        break;
                }
                if (put_user((unsigned int)dev->rl_peerport, 
                             (int __user *)&ioctl_port->port)) {
                        retval = -EFAULT;
                        up(&dev->sem_snd);
                        break;
                }
		up(&dev->sem_snd);
		break;

        case RLMX_IOCTL_GETMYPORT:
                if (put_user((unsigned int)dev->rl_myport, (int __user *)arg)) {
                        retval = -EFAULT;
                }
                break;

        case RLMX_IOCTL_SETMYHOST:
                if (get_user(val, (int __user *)arg)) {
                        return -EFAULT;
                }
                if (val > RLDATA_ADRSIGMASK) {
                        return -EINVAL;
                }

		if (down_interruptible(&dev->sem_snd))
			return -ERESTARTSYS;
                
                dev->root_device->myhostid = 
                        (u_short)(val & RLDATA_ADRSIGMASK);

                up(&dev->sem_snd);
                break;

        case RLMX_IOCTL_GETMYHOST:
                if (down_interruptible(&dev->sem_snd))
                        return -ERESTARTSYS;
                if (put_user((unsigned int)dev->root_device->myhostid,
                             (int __user *)arg)) {
                        retval = -EFAULT;
                }
                up(&dev->sem_snd);
                break;

        case RLMX_IOCTL_SETWINSIZE:
                if (get_user(val, (int __user *)arg)) {
                        return -EFAULT;
                }
                if (dev->rl_snd.buf_size < 
                    (unsigned int)(val) || !(val)) {
                        return -EINVAL;
                }
                val = (u_short)(val & RLMX_WINSIZEMASK);

                if (down_interruptible(&dev->sem_snd))
                        return -ERESTARTSYS;
                if (atomic_read(&dev->rl_snd.winsize) 
                    >= dev->rl_snd.max_winsize) {
                        dev->rl_snd.max_winsize = val;
                        atomic_set(&dev->rl_snd.winsize, val);
                } else {
                        retval = -EBUSY;
                }
                up(&dev->sem_snd);

                break;

        case RLMX_IOCTL_GETWINSIZE:
                if (down_interruptible(&dev->sem_snd))
                        return -ERESTARTSYS;
                if (put_user((int)dev->rl_snd.max_winsize,
                             (int __user *)arg)) {
                        retval = -EFAULT;
                }
                up(&dev->sem_snd);

                break;

        case RLMX_IOCTL_SETSNDSIZE:
                if (get_user(val, (int __user *)arg)) {
                        return  -EFAULT;
                }
                if (val > RLMX_SBUFSIZEMASK || !(val)) {
                        return -EINVAL;
                }
                if (val < dev->rl_snd.max_winsize) {
                        return -EINVAL;
                }
                
                if (down_interruptible(&dev->sem_snd))
                        return -ERESTARTSYS;
                if (atomic_read(&dev->rl_snd.winsize) 
                    >= dev->rl_snd.max_winsize) {
                        val = (u_short)(val & RLMX_SBUFSIZEMASK);
                        if (resize_rlmxbuf(&dev->rl_sbuf, val)) {
                                retval = -ENOMEM;
                        }else{
                                dev->rl_snd.buf_size = val;
                                dev->rl_snd.pos = 0;
                        }
                } else {
                        retval = -EBUSY;
                }
                up(&dev->sem_snd);

                break;

        case RLMX_IOCTL_GETSNDSIZE:
                if (down_interruptible(&dev->sem_snd))
                        return -ERESTARTSYS;
                if (put_user((unsigned int)dev->rl_snd.buf_size,
                             (int __user *)arg)) {
                        retval = -EFAULT;
                }
                up(&dev->sem_snd);

                break;

        case RLMX_IOCTL_SETRCVSIZE:
                if (get_user(val, (int __user *)arg)) {
                        return -EFAULT;
                }
                if (val > RLMX_RBUFSIZEMASK || !(val)) {
                        return -EINVAL;
                }
                val = (u_short)(val & RLMX_RBUFSIZEMASK);

                if (down_interruptible(&dev->sem_rcv))
                        return -ERESTARTSYS;
                spin_lock(&dev->rl_rcv.lock);
                if (atomic_read(&dev->rl_rcv.cnt) == 0) {
                        if (resize_rlmxbuf(&dev->rl_rbuf, val)) {
                                retval = -ENOMEM;
                        }else{
                                dev->rl_rcv.spos = 0;
                                dev->rl_rcv.tpos = 0;
                                dev->rl_rcv.buf_size = val;
                        }
                } else {
                        retval = -EBUSY;
                }
                spin_unlock(&dev->rl_rcv.lock);
                up(&dev->sem_rcv);

                break;

        case RLMX_IOCTL_GETRCVSIZE:
                if (down_interruptible(&dev->sem_rcv))
                        return -ERESTARTSYS;
                if (put_user((int)dev->rl_rcv.buf_size,
                             (int __user *)arg)) {
                        retval = -EFAULT;
                }
                up(&dev->sem_rcv);
                break;

        case RLMX_IOCTL_SETANYFLAG:
                if (get_user(val, (int __user *)arg)) {
                        retval = -EFAULT;
                        break;
                }

                if (down_interruptible(&dev->sem_rcv))
                        return -ERESTARTSYS;

                /* CASE: (ANY -> Not Any)*/
                tmp = dev->any_port;
                dev->any_port = (u_short)(val != 0);                
                if (!tmp || val) { 
                        up(&dev->sem_rcv);                        
                        break;                        
                }               

                /*Clean up packets whose soruce addresses are unexpected  */
                /* and connection-control packets */
                spin_lock(&dev->rl_rcv.lock);
                pnt = dev->rl_rbuf[(dev->rl_rcv.spos) 
                                   % dev->rl_rcv.buf_size];
                pktnum = 0;
                for (i = 0; i < atomic_read(&dev->rl_rcv.cnt); i++) {
                        pnt2 = dev->rl_rbuf[(dev->rl_rcv.spos + i) 
                                            % dev->rl_rcv.buf_size];
                        val  = *(pnt2 + RLMX_PKTIX_ADDR);
                        val2 = *(pnt2 + RLMX_PKTIX_PORT);
                        val  = (val  >> RLDATA_ADRBIT) & RLDATA_ADRSIGMASK; 
                        val2 = (val2 >> RLMX_PORTBIT) & RLMX_PORTMASK;

#ifdef RLMX_CONNECT_PKT
                        val3 =  (*(pnt2 + RLMX_PKTIX_TYPSEQ) & RLMX_TYPMASK) >> RLMX_SEQBIT;
                        if (val == dev->rl_peerhost 
                            && val2 == dev->rl_peerport
                            && val3 != RLMX_TYPCNT) {
                                memcpy(pnt, pnt2, DATA_PACKET_SIZE);
                                pktnum ++;
                                pnt = dev->rl_rbuf[(dev->rl_rcv.spos + pktnum)
                                                   % dev->rl_rcv.buf_size];
                        }
#else
                        if (val == dev->rl_peerhost 
                            && val2 == dev->rl_peerport) {
                                memcpy(pnt, pnt2, DATA_PACKET_SIZE);
                                pktnum ++;
                                pnt = dev->rl_rbuf[(dev->rl_rcv.spos + pktnum)
                                                   % dev->rl_rcv.buf_size];
                        }
#endif                
                }
                atomic_set(&dev->rl_rcv.cnt, pktnum);
                spin_unlock(&dev->rl_rcv.lock);

                up(&dev->sem_rcv);
                break;

        case RLMX_IOCTL_GETANYFLAG:
                if (put_user((int)dev->any_port,
                             (int __user *)arg)) {
                        retval = -EFAULT;
                }
                break;
                
        case RLMX_IOCTL_SETROOTBUFSIZE:
                if (get_user(val, (unsigned int __user *)arg)) {
                        return  -EFAULT;
                }
		if (val < RLDATA_FIFO_SZMIN
                    || val > RLDATA_FIFO_SZMAX) { 
			return -EINVAL;
		}
                
		if (down_interruptible(&dev->root_device->sem_out))
			return -ERESTARTSYS;
                if (fifo_datasize(dev->root_device->fifo_dpmout) == 0) {
			if (resize_fifo(dev->root_device->fifo_dpmout, val)) {
				retval = -ENOMEM;
			}else{
                                dev->root_device->sz_fifoout = val;
                        }
		} else {
			retval = -EBUSY;
		}
		up(&dev->root_device->sem_out);

		break;
                
        case RLMX_IOCTL_GETROOTBUFSIZE:
                if (down_interruptible(&dev->root_device->sem_out))
                        return -ERESTARTSYS;
                if (put_user((unsigned int)dev->root_device->sz_fifoout,
                             (unsigned int __user *)arg)) {
                        retval = -EFAULT;
                }
                up(&dev->root_device->sem_out);
                
                break;
                
        case RLMX_IOCTL_SETPRIO:
                if (get_user(val, (unsigned int __user *)arg)) {
                        return  -EFAULT;
                }
                if (val >= (1 << RLDATA_PRIOBIT) || val < 0) {
                        return -EINVAL;
                }
                
                if (down_interruptible(&dev->sem_snd))
                        return -ERESTARTSYS;
                dev->rl_priority = val & ((1 << RLDATA_PRIOBIT) - 1);
                up(&dev->sem_snd);

                break;

        case RLMX_IOCTL_GETPRIO:
                if (down_interruptible(&dev->sem_snd))
                        return -ERESTARTSYS;
                if (put_user((unsigned int)dev->rl_priority,
                             (unsigned int __user *)arg)) {
                        retval = -EFAULT;
                }
                up(&dev->sem_snd);

                break;

        case RLMX_IOCTL_GETCONSEQ:
                spin_lock(&dev->rl_lock_conseq);
                dev->rl_conseq ++;
                if (put_user((unsigned int)dev->rl_conseq,
                             (unsigned int __user *)arg)) {
                        retval = -EFAULT;
                }
                spin_unlock(&dev->rl_lock_conseq);
                break;

#ifdef RLMX_ACKCONSEQ_CHECK
        case RLMX_IOCTL_SETDSTSEQ:
                if (get_user(val, (unsigned int __user *)arg)) {
                        return  -EFAULT;
                }
                if (down_interruptible(&dev->sem_rcv))
                        return -ERESTARTSYS;
                dev->rl_peerseq = val;
                up(&dev->sem_rcv);                
                break;

        case RLMX_IOCTL_GETDSTSEQ:
                if (down_interruptible(&dev->sem_rcv))
                        return -ERESTARTSYS;
                if (put_user(dev->rl_peerseq, 
                             (unsigned int __user *)arg)) {
                        retval = -EFAULT;
                }
                up(&dev->sem_rcv);                
                break;
#endif

        default:
                return -ENOSYS;
        }

	return retval;
}


struct file_operations rldata_fops = {
	.owner = THIS_MODULE,
	.ioctl = rldata_ioctl,
	.read = NULL,
	.write = NULL, 
	.open = rldata_open,
	.release = rldata_release,
	.poll = NULL 
};


struct file_operations rlmx_fops = {
	.owner = THIS_MODULE,
	.ioctl = rlmx_ioctl,
	.read = rlmx_read,
	.write = rlmx_write,
	.open = rlmx_open,
	.release = rlmx_release,
	.poll = rlmx_poll
};


static void add_tv(struct timeval *tv, long sec, long usec)
{
	int carry;

	if (usec < 0 || sec < 0) {
		printk("add_tv: bad add value (sec=%d usec=%06d)\n",
		       sec, usec);
		return;
	}

	usec += tv->tv_usec;
	sec += usec / 1000000;
	usec = usec % 1000000;

	tv->tv_sec += sec;
	tv->tv_usec = usec;
}


/*#define MON_NEXT_AVERAGE*/
static int rldata_rtthd_poldpm(void *unused)
{
	int curpacknum_read;
	int curpacknum_write;
	int nxtpacknum_read;
	int nxtpacknum_write;
	int snd_endpoint;
	int rval;
        int num_recv, size_recv;
	int tmp_len;
        unsigned int *dpm_start;
        unsigned int *ptrl;
	struct timeval start, end;
	unsigned period;
#ifdef MON_NEXT_AVERAGE
	int counter = 0;
	int next_rval = 0;
	int average_next_rval = 0;
	int max_next_rval = 0;
#endif
	int tmp , tmp2, i, j;


	/*printk("Entered to rldata_rtthd_poldpm.\n", rval);*/
	curpacknum_read = readl(DATA_DPM_IN_CTRL + DPM_CTLREG_PCKTNUM);
	curpacknum_write = readl(DATA_DPM_OUT_CTRL + DPM_CTLREG_PCKTNUM);
	snd_endpoint = curpacknum_write;

 restart:
	do_gettimeofday(&start);
	end = start;
        
	add_tv(&start, 0, 0);	
	add_tv(&end, 86400, 5);	/*1Day*/
	period = 10000; 
	rval = do_rtthread_start_periodic_thread(&start, &end, period);
        /*printk("rldata: real time thread is waken up. rval= %d\n", rval);
          printk("&nxtpacknum_write = 0x%08x\n", &nxtpacknum_write);
          printk("&snd_endpoint = 0x%08x\n", &snd_endpoint); */

#ifdef MON_NEXT_AVERAGE
        printk("&counter = 0x%08x\n", &counter);
        printk("&average_next_rval = 0x%08x\n", &average_next_rval);
        printk("&max_next_rval = 0x%08x\n", &max_next_rval);        
#endif
	do {
		nxtpacknum_read =
                        readl(DATA_DPM_IN_CTRL + DPM_CTLREG_PCKTNUM);
                
		if (nxtpacknum_read != curpacknum_read) {
			num_recv = nxtpacknum_read - curpacknum_read;
                        
			if (num_recv < 0) {
                                num_recv = DPM_DATA_PACKET_NUM + num_recv;
			}
                        size_recv = num_recv * DATA_PACKET_SIZE;
                        
                        dpm_start = (unsigned int *)(DATA_DPM_IN_BASE
                                                     + curpacknum_read * DATA_PACKET_SIZE);
                        rval = copydpm_and_countdata(rldata_devices, dpm_start, size_recv);
                        
			curpacknum_read = nxtpacknum_read;
		}
                
		nxtpacknum_write = readl(DATA_DPM_OUT_CTRL + DPM_CTLREG_PCKTNUM);
                
		tmp_len = (signed int) fifo_datasize(rldata_devices->fifo_dpmout);

		if (tmp_len >= DATA_PACKET_SIZE) {
			tmp_len = tmp_len / DATA_PACKET_SIZE;
			if (nxtpacknum_write == snd_endpoint) {
                                /* tmp_len <= nxtpacknum_write) { */
                                tmp_len = min(tmp_len, DPM_DATA_PACKET_NUM / 2 - 1);
				writel(tmp_len * 0x10, DATA_DPM_OUT_CTRL);
				fifo_outpackets(rldata_devices->fifo_dpmout, tmp_len);
				snd_endpoint = tmp_len;
				wake_up_interruptible(&rldata_devices->write_waitq);
			}
		}
                
		curpacknum_write = nxtpacknum_write;

#ifdef MON_NEXT_AVERAGE
		tmp = do_rtthread_next_period();	
		if (tmp < 0)
			goto restart;
		next_rval += tmp;
		if (tmp > max_next_rval) {
			max_next_rval = next_rval;
		}
		if (!(counter++ & 63)) {
			average_next_rval = next_rval >> 5;
			next_rval = 0;
		}

#else
		if (do_rtthread_next_period() < 0)
			goto restart;
#endif
	} while (1);
}




#ifdef RLMX_PROC 
/*
 * For now, the seq_file implementation will exist in parallel.  The
 * older read_procmem function should maybe go away, though.
 */

/*
o * Here are our sequence iteration methods.  Our "position" is
 * simply the device number.
 */
static void *rlmx_seq_start(struct seq_file *s, loff_t *pos)
{
	if (*pos >= RLMX_PORT_MAX)
		return NULL;   /* No more to read */
	return rlmx_dev_info + *pos;
}

static void *rlmx_seq_next(struct seq_file *s, void *v, loff_t *pos)
{
	(*pos)++;
	if (*pos >= RLMX_PORT_MAX)
		return NULL;
	return rlmx_dev_info + *pos;
}

static void rlmx_seq_stop(struct seq_file *s, void *v)
{
	/* Actually, there's nothing to do here */
}

static int rlmx_seq_show(struct seq_file *s, void *v)
{
	struct rlmx_dev_info *dev = (struct rlmx_dev *) v;
	struct rlmx_qset *d;
	int i, dev_ix, tmp;

        dev_ix = dev - rlmx_dev_info;
        if (!dev_ix) {
                /* Punch out the root device information. */
                seq_printf(s, "hostid = 0x%03x\n", 
                           rldata_devices->myhostid);
                seq_printf(s, "FIFO size for DPM_OUT : %dbyte", 
                           rldata_devices->fifo_dpmout->size);
                seq_printf(s, "(usage: %dbyte)\n", 
                           fifo_datasize(rldata_devices->fifo_dpmout));
                seq_printf(s, "Number of packet has illegal destination port : %8d\n", 
                           rldata_devices->err_cnt.port_num); 
                seq_printf(s, "Number of packet has fatal error : %8d\n", 
                           rldata_devices->err_cnt.invlpkt_cnt); 
                seq_printf(s, "PORT:O A "
                           "PHOST  PPORT "
                           "SNDBUF WINSIZ MAXWIN SEND_PCKT NXTSNDSEQ "
                           "RXVBUF LEFTPK RECV_PCKT NXTRCVSEQ "
                           "UNEXPECT  BUF_FULL  "
                           "PKT_ORD   ACK_ORD   "
                           "PKT_TYPE \n");
        } 
        seq_printf(s, "%4d:%1d %1d ",
                   dev_ix, (atomic_read(&dev->available) < 1),  
                   (atomic_read(&dev->active) >0));
        seq_printf(s,"0x%04x  %04d ",
                   dev->rl_peerhost, dev->rl_peerport);
        seq_printf(s,"%6d %6d %6d %9d %9d ",
                   dev->rl_snd.buf_size, dev->rl_snd.winsize,
                   dev->rl_snd.max_winsize, dev->rl_snd.pnum,
                   dev->rl_snd.seq);
        seq_printf(s,"%6d %6d %9d %9d ",
                   dev->rl_rcv.buf_size, atomic_read(&dev->rl_rcv.cnt),
                   dev->rl_rcv.pnum, dev->rl_rcv.nxt);
        seq_printf(s, "%9d",
                   dev->err_cnt.unexpect);
        seq_printf(s, "%9d %9d ",
                   dev->err_cnt.buf_full, dev->err_cnt.pkt_ord);
        seq_printf(s, "%9d %9d\n",
                   dev->err_cnt.ack_ord, dev->err_cnt.pkt_type);
        
        
	return 0;
}
	
/*
 * Tie the sequence operators up.
 */
static struct seq_operations rlmx_seq_ops = {
	.start = rlmx_seq_start,
	.next  = rlmx_seq_next,
	.stop  = rlmx_seq_stop,
	.show  = rlmx_seq_show
};

/*
 * Now to implement the /proc file we need only make an open
 * method which sets up the sequence operators.
 */
static int rlmx_proc_open(struct inode *inode, struct file *file)
{
	return seq_open(file, &rlmx_seq_ops);
}

/*
 * Create a set of file operations for our proc file.
 */
static struct file_operations rlmx_proc_ops = {
	.owner   = THIS_MODULE,
	.open    = rlmx_proc_open,
	.read    = seq_read,
	.llseek  = seq_lseek,
	.release = seq_release
};
	

/*
 * Actually create (and remove) the /proc file(s).
 */

static void rlmx_create_proc(void)
{
	struct proc_dir_entry *entry;
	entry = create_proc_entry("rlmx_stat", 0, NULL);
	if (entry)
		entry->proc_fops = &rlmx_proc_ops;
}

static void rlmx_remove_proc(void)
{
	/* no problem if it was not registered */
	remove_proc_entry("rlmxseq", NULL);
}


#endif  /*RLMX_PROC*/

void rldata_cleanup_module(void)
{
	int i;
	dev_t devno = MKDEV(rldata_major, rldata_minor);

	if (rldata_devices) {
                /*free FIFO( DPM_IN/_OUT)*/
                fifo_free(&rldata_devices->fifo_dpmout);
                fifo_free(&rldata_devices->fifo_dpmin);
                cdev_del(&rldata_devices->cdev);
                kfree(rldata_devices);
	}

	unregister_chrdev_region(devno, rldata_nr_devs);
}


/*
 * Set up the char_dev structure for this device.
 */
static int rldata_setup_cdev(struct rldata_dev *dev, int index)
{
	int err, devno = MKDEV(rldata_major, rldata_minor + index);
	struct dpm_fifo *tmp_dpm_fifo;

	cdev_init(&dev->cdev, &rldata_fops);
	dev->cdev.owner = THIS_MODULE;
	dev->cdev.ops = &rldata_fops;
        dev->myhostid = 0x0001;

        /* initialize FIFO(for DPM_OUT)*/
	dev->sz_fifoout = RLDATA_FIFO_SZPOW;
	dev->fifo_dpmout = 
                kmalloc(sizeof(struct dpm_fifo), GFP_KERNEL);
	if (!dev->fifo_dpmout) {
		err = -ENOMEM;
		goto fail;
	}
	dev->fifo_dpmout->size =
                sizeof(char) * (1 << dev->sz_fifoout);
	dev->fifo_dpmout->buffer =
                kmalloc(dev->fifo_dpmout->size, GFP_KERNEL);
	if (!dev->fifo_dpmout->buffer) {
		err = -ENOMEM;
		goto fail;
	}
	dev->fifo_dpmout->in = dev->fifo_dpmout->out = 0;
	spin_lock_init(&dev->fifo_dpmout->lock);

        /* initialize FIFO(for DPM_IN) */
	dev->sz_fifoin = RLDATA_FIFO_SZPOW;
	dev->fifo_dpmin = 
                kmalloc(sizeof(struct dpm_fifo), GFP_KERNEL);
	if (!dev->fifo_dpmin) {
		err = -ENOMEM;
		goto fail;
	}
	dev->fifo_dpmin->size =
                sizeof(char) * (1 << dev->sz_fifoin);
	dev->fifo_dpmin->buffer =
                kmalloc(dev->fifo_dpmin->size, GFP_KERNEL);
	if (!dev->fifo_dpmin->buffer) {
		err = -ENOMEM;
		goto fail;
	}
	dev->fifo_dpmin->in = dev->fifo_dpmin->out = 0;
	spin_lock_init(&dev->fifo_dpmin->lock);

	err = cdev_add(&dev->cdev, devno, 1);

fail:
	if (err) printk(KERN_NOTICE "Error %d adding rldata%d", err, index);

        return err;
}


static int rldata_setup_port_cdev(struct rlmx_dev_info *dev, 
                                  struct rldata_dev *root_dev, int index)
{
	int err, devno = MKDEV(rldata_major, rldata_minor + index);
	struct dpm_fifo *tmp_dpm_fifo;
	cdev_init(&dev->cdev, &rlmx_fops);
	dev->cdev.owner = THIS_MODULE;
	dev->cdev.ops = &rlmx_fops;

	init_MUTEX(&dev->sem_snd);
	init_MUTEX(&dev->sem_rcv);
	init_waitqueue_head(&dev->rl_snd.enqwait);
	init_waitqueue_head(&dev->rl_rcv.rwait);

        atomic_set(&dev->available, 1);
        atomic_set(&dev->active, 0);
        dev->any_port = RLMX_DEF_ANY; 
        dev->root_device = root_dev;
        /*dev->rl_status = 0;*/
        dev->rl_peerhost = root_dev->myhostid;
        dev->rl_myport = index - 1;
        dev->rl_peerport = index -1;
        dev->rl_priority = RLMX_DEF_PRIO;
	spin_lock_init(&dev->rl_lock_conseq);
        dev->rl_conseq = 0;
#ifdef RLMX_ACKCONSEQ_CHECK
        dev->rl_peerseq = 0;
#endif
        /*Snd Buffer*/
        dev->rl_snd.max_winsize = RLMX_DFL_WINSIZE;
        atomic_set(&dev->rl_snd.winsize, RLMX_DFL_WINSIZE);
        dev->rl_snd.buf_size = RLMX_DEF_SBUFSIZE;
        atomic_set(&dev->rl_snd.winsize, 
                   dev->rl_snd.max_winsize);
        dev->rl_snd.nxt = 0;
        dev->rl_snd.pnum = 0;
        dev->rl_sbuf = kmalloc(sizeof(rl_pkt_t) 
                               * dev->rl_snd.buf_size, GFP_KERNEL);

        /*Rcv Buffer*/
        dev->rl_rcv.buf_size = RLMX_DEF_RBUFSIZE;
        dev->rl_rcv.nxt = 0;
        dev->rl_rcv.spos = 0;
        dev->rl_rcv.tpos = 0;
        dev->rl_rcv.pnum = 0;
        atomic_set(&dev->rl_rcv.cnt, 0);
        dev->rl_rbuf = kmalloc(sizeof(rl_pkt_t) 
                               * dev->rl_rcv.buf_size, GFP_KERNEL);
	spin_lock_init(&dev->rl_rcv.lock);

        /*Error count*/
        dev->err_cnt.unexpect = 0;
        dev->err_cnt.buf_full = 0;
        dev->err_cnt.pkt_ord = 0;
        dev->err_cnt.ack_ord = 0;
        dev->err_cnt.pkt_type = 0;

	err = cdev_add(&dev->cdev, devno, 1);
        if (err) printk(KERN_NOTICE "Error %d adding rlmx/%d", err, index);
        return err;
}


int rldata_init_module(void)
{
	int result, i;
	int pid;
	dev_t dev = 0;

        /*
         * Get a range of minor numbers to work with, asking for a dynamic
         * major unless directed otherwise at load time.
         */
	if (rldata_major) {
		dev = MKDEV(rldata_major, rldata_minor);
		result = register_chrdev_region(dev, rldata_nr_devs, "rldata");
	} else {
		result = alloc_chrdev_region(&dev, rldata_minor, 
                                             rldata_nr_devs, "rldata");
		rldata_major = MAJOR(dev);
	}

	if (result < 0) {
		printk(KERN_WARNING "rldata: can't get major %d\n",
		       rldata_major);
		return result;
	}

	/* 
         * for core device
	 */
	rldata_devices =
                kmalloc(sizeof(struct rldata_dev), GFP_KERNEL);
	if (!rldata_devices) {
		result = -ENOMEM;
		goto fail;
	}
	memset(rldata_devices, 0, sizeof(struct rldata_dev));

        /*
         * for each ports  
         */
	rlmx_dev_info =
                kmalloc(rlmx_num_port * sizeof(struct rlmx_dev_info), 
                        GFP_KERNEL);
	if (!rlmx_dev_info) {
                printk("No enough memory for rlmx poirt buffer\n");
		result = -ENOMEM;
		goto fail;
	}
	memset(rlmx_dev_info, 0,
	       rlmx_num_port * sizeof(struct rlmx_dev_info));


        /*
         * initialize Responsive Link registers.
         */
	axe_rl_init();
        
        /*
         * for core device.
         */ 
	init_waitqueue_head(&rldata_devices->write_waitq);
	init_waitqueue_head(&rldata_devices->read_waitq);

	init_MUTEX(&rldata_devices->sem_in);
	init_MUTEX(&rldata_devices->sem_out);

	result = rldata_setup_cdev(rldata_devices, 0);
        if(result){
                goto fail;
        }

        /*
         * for each ports
         */
        for (i = 0; i < rlmx_num_port; i++) {
                result = rldata_setup_port_cdev(&rlmx_dev_info[i],
                                                rldata_devices, i + 1);
                if(result){
                        goto fail;
                }
        }


        /* start RT-Thread */
	pid = kernel_thread(rldata_rtthd_poldpm, NULL, CLONE_KERNEL);
	if (!pid) {
		printk(KERN_WARNING
		       "rldata: Failed to start rtthd_poldpm\n");
                result = pid;
		goto fail;
	}
        
        rlmx_create_proc();
        
	return 0;
        
 fail:
	rldata_cleanup_module();
	return result;
}


module_init(rldata_init_module);
module_exit(rldata_cleanup_module);



