/*
 * Switch a MMU context.
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file "COPYING" in the main directory of this archive
 * for more details.
 *
 * Copyright (C) 1996, 1997, 1998, 1999 by Ralf Baechle
 * Copyright (C) 1999 Silicon Graphics, Inc.
 */
#ifndef _ASM_MMU_CONTEXT_H
#define _ASM_MMU_CONTEXT_H

#include <linux/config.h>
#include <linux/errno.h>
#include <linux/sched.h>
#include <linux/slab.h>
#include <asm/cacheflush.h>
#include <asm/tlbflush.h>
#include <asm/tlb.h>
#include <asm/thread_control.h>

/*
 * For the fast tlb miss handlers, we currently keep a per cpu array
 * of pointers to the current pgd for each processor. Also, the proc.
 * id is stuffed into the context register. This should be changed to
 * use the processor id via current->processor, where current is stored
 * in watchhi/lo. The context register should be used to contiguously
 * map the page tables.
 */
#define TLBMISS_HANDLER_SETUP_PGD(pgd) \
	pgd_current[smp_processor_id()] = (unsigned long)(pgd)
#define TLBMISS_HANDLER_SETUP() \
	TLBMISS_HANDLER_SETUP_PGD(swapper_pg_dir)
extern unsigned long pgd_current[];

#if defined(CONFIG_CPU_R3000) || defined(CONFIG_CPU_TX39XX)

#define ASID_INC	0x40
#define ASID_MASK	0xfc0

#else
#ifdef CONFIG_RMT
#define ASID_INC	0x2
#define ASID_MASK	0x3e
#else /* FIXME: not correct for R6000, R8000 */
#define ASID_INC	0x1
#define ASID_MASK	0xff
#endif

#endif

#ifdef CONFIG_RMT
/* TLB Entry 1 */
#define ENTRY1_SHR	0x000000ff
#define ENTRY1_PRO	0x00000700
#define ENTRY1_LCK	0x00000800
#define ENTRY1_VPN	0xfffff000

#define	PRO_ALL_RW	0x7
#define	PRO_ALL_R	0x6
#define	PRO_USR_RW	0x5
#define	PRO_USR_R	0x4
#define	PRO_SV_RW	0x3
#define	PRO_SV_R	0x2
#define	PRO_KER_RW	0x1
#define	PRO_KER_R	0x0

/* TLB Entry 2 */
#define ENTRY2_BRT	0x00000003
#define	ENTRY2_UNC	0x00000004
#define ENTRY2_CLC	0x00000008
#define ENTRY2_GRP	0x000003f0
#define	ENTRY2_PSZ	0x00000c00
#define	ENTRY2_PPN	0xfffff000

#define	BRT_LEN0	0x3	/*  32byte */
#define	BRT_LEN2	0x2	/*  64byte */
#define	BRT_LEN4	0x1	/* 128byte */
#define	BRT_LEN8	0x0	/* 256byte */

#define	TLB_PSZ_4KB	0x00000000
#define	TLB_PSZ_64KB	0x00000400
#define	TLB_PSZ_1MB	0x00000800
#define	TLB_PSZ_16MB	0x00000c00
#endif

#if 0
#define cpu_context(cpu, mm)	((mm)->context.cpus[cpu])
#else
#define cpu_context(cpu, mm)	((mm)->context[cpu])
#endif
#define cpu_asid(cpu, mm)	(cpu_context((cpu), (mm)) & ASID_MASK)
#ifdef CONFIG_RMT
#define asid_cache(cpu)		(cpu_data[0].asid_cache)
#else
#define asid_cache(cpu)		(cpu_data[cpu].asid_cache)
#endif

#ifdef CONFIG_RMT_TLB_GROUP_FIX_BIND
#define cpu_fix_context(cpu)	(TLB_GRP_USER + (cpu))
#endif

static inline void enter_lazy_tlb(struct mm_struct *mm, struct task_struct *tsk)
{
}

/*
 *  All unused by hardware upper bits will be considered
 *  as a software asid extension.
 */
#define ASID_VERSION_MASK  ((unsigned long)~(ASID_MASK|(ASID_MASK-1)))
#define ASID_FIRST_VERSION ((unsigned long)(~ASID_VERSION_MASK) + 1)

static inline void
get_new_mmu_context(struct mm_struct *mm, unsigned long cpu)
{
#ifdef CONFIG_RMT_TLB_GROUP_FIX_BIND
	local_flush_tlb_mm(mm);
	cpu_context(cpu, mm) = asid_cache(cpu) = cpu_fix_context(cpu);
#else
#error '!CONFIG_RMT_TLB_GROUP_FIX_BIND no support'
	unsigned long asid = asid_cache(cpu);

	if (! ((asid += ASID_INC) & ASID_MASK) ) {
#ifdef CONFIG_VTAG_ICACHE
		flush_icache_all();
#endif
		local_flush_tlb_all();	/* start new asid cycle */
		if (!asid)		/* fix version if needed */
			asid = ASID_FIRST_VERSION;
	}
	cpu_context(cpu, mm) = asid_cache(cpu) = asid;
#endif
}

/*
 * Initialize the context related info for a new mm_struct
 * instance.
 */
static inline int
init_new_context(struct task_struct *tsk, struct mm_struct *mm)
{
	int i;

	rmt_str_printf("init_new_context(): #1\n");
	rmt_str_printf("num_online_cpus() = 0x");
	rmt_hex_printf(num_online_cpus());
	rmt_str_printf("mm = 0x");
	rmt_hex_printf(mm);
	rmt_str_printf("current = 0x");
	rmt_hex_printf(current);
#ifndef CONFIG_RMT
	for (i = 0; i < num_online_cpus(); i++)
		cpu_context(i, mm) = 0;
#endif

	return 0;
}

static inline void switch_mm(struct mm_struct *prev, struct mm_struct *next,
                             struct task_struct *tsk)
{
	unsigned int cpu = smp_processor_id();
	unsigned long flags;
	int	i;

	local_irq_save(flags);

#ifdef CONFIG_RMT
#if 1
	for(i=0; i<8; i++)
		write_mmu_register(MMU_SPR_GRP_FLUSH,
			TLB_GRP_USER + i);
#else
	invalid_tlb_grpid(prev->context.grpid);
	atomic_dec(&prev->context.counter);

	valid_tlb_grpid(next->context.grpid);
	atomic_dec(&next->context.counter);
#endif
#endif

#ifndef CONFIG_RMT
	/* Check if our ASID is of an older version and thus invalid */
	if ((cpu_context(cpu, next) ^ asid_cache(cpu)) & ASID_VERSION_MASK)
		get_new_mmu_context(next, cpu);

	write_c0_entryhi(cpu_context(cpu, next));
#endif
	TLBMISS_HANDLER_SETUP_PGD(next->pgd);

	/*
	 * Mark current->active_mm as not "active" anymore.
	 * We don't want to mislead possible IPI tlb flush routines.
	 */
	clear_bit(cpu, &prev->cpu_vm_mask);
	set_bit(cpu, &next->cpu_vm_mask);

	local_irq_restore(flags);
}

/*
 * Destroy context related info for an mm_struct that is about
 * to be put to rest.
 */
static inline void destroy_context(struct mm_struct *mm)
{
}

#define deactivate_mm(tsk,mm)	do { } while (0)

/*
 * After we have set current->mm to a new value, this activates
 * the context for the new mm so we see the new mappings.
 */
static inline void
activate_mm(struct mm_struct *prev, struct mm_struct *next)
{
	unsigned long flags;
	int cpu = smp_processor_id();

	local_irq_save(flags);

#if 0 //def CONFIG_RMT
	invalid_tlb_grpid(prev->context.grpid);
	valid_tlb_grpid(next->context.grpid);
#else
	/* Unconditionally get a new ASID.  */
	get_new_mmu_context(next, cpu);

	//write_c0_entryhi(cpu_context(cpu, next));
#endif
	TLBMISS_HANDLER_SETUP_PGD(next->pgd);

	/* mark mmu ownership change */
	clear_bit(cpu, &prev->cpu_vm_mask);
	set_bit(cpu, &next->cpu_vm_mask);

	local_irq_restore(flags);
}

/*
 * If mm is currently active_mm, we can't really drop it.  Instead,
 * we will get a new one for it.
 */
static inline void
drop_mmu_context(struct mm_struct *mm, unsigned cpu)
{
	unsigned long flags;

	local_irq_save(flags);

	if (test_bit(cpu, &mm->cpu_vm_mask))  {
		get_new_mmu_context(mm, cpu);
#ifndef CONFIG_RMT
		write_mmu_register(MMU_SPR_ENTRY1, cpu_asid(cpu, mm));
#endif
	} else {
		/* will get a new context next time */
#ifndef CONFIG_RMT
		cpu_context(cpu, mm) = 0;
#endif
	}

	local_irq_restore(flags);
}

#endif /* _ASM_MMU_CONTEXT_H */
