/*
 * rmtproc.c:	PCI device driver module for RMTP PCI Eva. Board
 *
 * Editors:
 *	Shinpei Kato <shinpei@ny.ics.keio.ac.jp>
 *
 * This module works only on the linux kernel 2.6.
 */

#include <linux/init.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/pci.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/interrupt.h>
#include <asm/dma.h>
#include <asm/uaccess.h>

#include "rmtproc.h"

MODULE_LICENSE("Dual BSD/GPL");
MODULE_DESCRIPTION("PCI Driver for Responsive Multithreaded Processor");
MODULE_AUTHOR("Shinpei Kato <shinpei@ny.ics.keio.ac.jp>");

/* 
 * this indicates the base-two logarithm of the number of pages
 * you are requesting or freeing (i.e. log2N)
 * you need this variable when you get or free pages for DMA buffer.
 */
int dma_page_order = 0;

/* device numbers */
int rmtproc_major = RMTPROC_MAJOR;
int rmtproc_minor = RMTPROC_MINOR;
int rmtproc_nr_devs = 1;

/* for spin lock */
spinlock_t rmtproc_lock = SPIN_LOCK_UNLOCKED;

LIST_HEAD(rmtproc_drivers);
/* for semaphore */
DECLARE_WAIT_QUEUE_HEAD(rmtproc_queue);
/* page size in transfering by DMA */
#define	dma_page_size	(PAGE_SIZE << dma_page_order)

/* write to the local address */
int rmtproc_write_local_ad(Rmtproc_Dev *dev, unsigned long addr)
{
	unsigned long addr_base = dev->addr_base & PCI_BASE_ADDRESS_IO_MASK;

	/* write in the pci bus access port */
	iowrite32(addr, (void*)(addr_base + OFFSET_LOCAL_AD));
	// dont do this! the kernel would be hung up because of memory?
//	printk(KERN_WARNING PFX "local address: 0x%08lx.\n", addr);
			
	return 0; /* succeeded */
}

/* write to the bus access port */
int rmtproc_write_bap(Rmtproc_Dev *dev, unsigned long data)
{
	unsigned long addr_base = dev->addr_base & PCI_BASE_ADDRESS_IO_MASK;

	/* write in the pci bus access port */
	iowrite32(data, (void*)(addr_base + OFFSET_LOCAL_BAP));
	// dont do this! the kernel would be hung up because of memory?
//	printk(KERN_WARNING PFX "written data 0x%08lx.\n", data);
			
	return 0; /* succeeded */
}

/* read from to the bus access port */
int rmtproc_read_bap(Rmtproc_Dev *dev)
{
	unsigned long addr_base = dev->addr_base & PCI_BASE_ADDRESS_IO_MASK;

	/* write in the pci bus access port */
	return ioread32((void*)(addr_base + OFFSET_LOCAL_BAP));
	// dont do this! the kernel would be hung up because of memory?
//	printk(KERN_WARNING PFX "written data 0x%08lx.\n", data);
			
	return 0; /* succeeded */
}

/*
 * this function sends the data @x to Mail Box A
 */
int rmtproc_send_mail(Rmtproc_Dev *dev, unsigned long x, unsigned long y)
{
	unsigned long addr_base;
	int timer = TIMER_COUNT;

	/* drop off the lower bits */
	addr_base = dev->addr_base & PCI_BASE_ADDRESS_IO_MASK;

	/* wait for LIS down */
	while (timer > 0) {
//		if ((ioread8((void*)(addr_base + OFFSET_ICR)) << 24 & ICR_LS) == 0)
		if ((ioread8((void*)(addr_base + OFFSET_ICR)) & ICR_LS) == 0)
			break;
		timer--;
	}

	/* write in mailboxA */
	iowrite32(x, (void*)(addr_base + OFFSET_MAILBOX_A_L));
	iowrite32(y, (void*)(addr_base + OFFSET_MAILBOX_A_H));
			
	return 0; /* succeeded */
}

/*
 * this function receives data from Mail Box B
 */
int rmtproc_recv_mail(Rmtproc_Dev *dev, unsigned long *x)
{
	unsigned short tail;

	/* sleep. wait for data received */
	printk(KERN_WARNING PFX "Waiting for the event...\n");
	if (wait_event_interruptible(rmtproc_queue,
								 dev->mail_head != dev->mail_tail))
		return -ERESTARTSYS;

	/* get the data from the tail of the ring buffer */
	tail = dev->mail_tail;
	*x = dev->mail[tail];
	/* make sure to load the data */
	barrier();
	/* next buffer */
	dev->mail_tail = (tail == (MAIL_NUM - 1)) ? 0: tail + 1;

//	printk(KERN_WARNING PFX "received 0x%08lx.\n", *x);

	return 0; /* succeeded */
}

/*
 * interrupt handler.
 * this function is usuall called when data arrive at the mailboxB.
 */
irqreturn_t rmtproc_intr(int irq, void *dev_id, struct pt_regs *regs)
{
	int handled = 1;
	unsigned long io_addr;
	unsigned long icr;
	unsigned int flags;
	unsigned short head;
	Rmtproc_Dev *dev = (Rmtproc_Dev *) dev_id;

	printk(KERN_WARNING PFX "rmtproc_intr was invoked.\n");

	/* drop off the lower bits */
	io_addr = dev->addr_base & PCI_BASE_ADDRESS_IO_MASK;
	/* interruption control regiseter */
	icr = ioread32((void*)(io_addr + OFFSET_ICR));

	/* if PS(PCI Bus Interrupt Satus) is set, get data from the mailboxB 
	   and clear PS. */
	if (icr & ICR_PS) {
		spin_lock_irqsave(&rmtproc_lock, flags);

		head = dev->mail_head;
		/* load the data to the head of the buffer */
		dev->mail[head] = ioread32((void*)(dev->addr_base + OFFSET_MAILBOX_B));
		/* reset the interruption for mailboxB */
		iowrite32(icr | ICR_PS, (void*)(io_addr + OFFSET_ICR));
		/* this is ring buffer */
		dev->mail_head = (head == MAIL_NUM - 1)? 0 : head + 1;

		spin_unlock_irqrestore(&rmtproc_lock, flags);
		/* wake up the procedure */
		printk(KERN_WARNING PFX "waking up the task...\n");
		wake_up_interruptible(&rmtproc_queue);
	}
	else
		handled = 0;

	return IRQ_RETVAL(handled);
}

/*
 * prepare dma.
 */
int rmtproc_dma_prepare(int channel, int mode, unsigned int buf, unsigned int count)
{
	unsigned long flags;

	flags = claim_dma_lock();
	disable_dma(channel);
	clear_dma_ff(channel);
	set_dma_mode(channel, mode);
	set_dma_addr(channel, virt_to_bus(buf));
	set_dma_count(channel, count);
	enable_dma(channel);
	release_dma_lock(flags);

	return 0;
}

/*
 * check whether dma is done.
 */
int rmtproc_dma_isdone(int channel)
{
	int residue;

	unsigned long flags = claim_dma_lock();
	residue = get_dma_residue(channel);
	release_dma_lock(flags);

	return residue == 0;
}

/*
 * read from dma
 */
int rmtproc_read_dma(Rmtproc_Dev *dev, size_t count)
{
	printk(KERN_WARNING PFX "rmtproc_read_dma()\n");

	unsigned long addr_base = dev->addr_base & PCI_BASE_ADDRESS_IO_MASK;
	unsigned long addr = virt_to_bus(dev->Rdma_data);

	rmtproc_dma_prepare(dev->dma_ch, DMA_MODE_READ, dev->Rdma_data, count);

	iowrite32(addr & DMA_PCI_MASK, (void*)(addr_base + OFFSET_DMA_ADDRESS));
	iowrite32(count & DMA_PCI_MASK, (void*)(addr_base + OFFSET_DMA_COUNT));
	iowrite8(RMT_PCI_DMR_MEMREAD, (void*)(addr_base + OFFSET_DMA_DMR));
	iowrite8(0, (void*)(addr_base + OFFSET_DMA_DMCR));
	iowrite8(0, (void*)(addr_base + OFFSET_DMA_FMCR));

	while (!rmtproc_dma_isdone(dev->dma_ch))
		;

	return 0;
}

#ifndef RMTP_DRV
static inline Rmtproc_Dev *get_rmtproc_driver(dev_t device)
{
	Rmtproc_Dev *p;

	list_for_each_entry(p, &rmtproc_drivers, list) {
		if (p->cdev->dev == device)
			return p;
	}

	return NULL;
}
#endif

/*
 * this function is called when the device is opened.
 * the open method should perform the following tasks:
 *	1. check for device-specific errors (such as device-not-ready)
 *	2. initialize the device if it is being opened for the first time
 */
int rmtproc_open(struct inode *inode, struct file *filp)
{
	struct Rmtproc_Dev *dev; /* device information */
	unsigned int irq;
	unsigned long addr_base;
	unsigned int icr;
	int err;
	int i;

#ifdef RMTP_DRV
	dev = container_of(inode->i_cdev, struct Rmtproc_Dev, cdev);
#else
	dev = get_rmtproc_driver(inode->i_rdev);
#endif
	filp->private_data = dev; /* for other methods */

	/* need to register the irq when first opened */
	if (!dev->count) {
		irq = dev->irq;
		/* drop off the lower bits */
		addr_base = dev->addr_base & PCI_BASE_ADDRESS_IO_MASK;
		/* interruption control regiseter */
		icr = ioread32((void*)(addr_base + OFFSET_ICR));
		icr &= ~ICR_PE;
		icr |= ICR_PS;
		/* reset (clear) the statuses */
		iowrite32(icr, (void*)(addr_base + OFFSET_ICR));

		/* request the irq */
		if ((err = request_irq(irq, rmtproc_intr, SA_SHIRQ,
							   THIS_MODULE_NAME, dev))) {
			printk(KERN_WARNING PFX "Cannot get IRQ.\n");
			return err;
		}

		/* request the dma channel */
		for (i = 0; i < MAX_DMA_CHANNELS; i++) {
			if (!(err = request_dma(dev->dma_ch = i, THIS_MODULE_NAME)))
				break;
		}

		if (err) {
			printk(KERN_WARNING PFX "Cannot get dma channel.\n");
			free_irq(irq, dev);
			return err;
		}

		/* initialize the buffer indexes */
		dev->mail_head = 0;
		dev->mail_tail = 0;

		/* enable pci bus interruption */
		iowrite32(icr | ICR_PE, (void*)(addr_base + OFFSET_ICR));
		printk(KERN_WARNING PFX "Enabled PCI Bus Intr.\n");
	}

	dev->count++;

	return 0; /* open succeeded */
}

/*
 * this function closes the device.
 */
int rmtproc_release(struct inode *inode, struct file *filp)
{
	Rmtproc_Dev *dev;

	dev = filp->private_data;

	/* free irq when closed for the last time */
	spin_lock_irq(&rmtproc_lock);
	dev->count--;
	spin_unlock_irq(&rmtproc_lock);
	if (!dev->count) {
		free_dma(dev->dma_ch);
		free_irq(dev->irq, dev);
	}

	return 0;
}

/*
 * this function transfer the data to RMT processor based on the protocol.
 * the function in RMT processor should know the protocol.
 *
 * the procedure of the protocol:
 *		1. send command
 *		2. send address
 *		3. send size
 *		4. recv ack
 *		5. 
 */
int rmtproc_transfer(Rmtproc_Dev *dev, void *dma_data, int dir, size_t count)
{
	unsigned long tmp;
	int res;
	dma_addr_t bus_addr;

	bus_addr = pci_map_single(dev->pci_dev, dma_data, PAGE_SIZE, dir);

	/* command, address, size */
	res = rmtproc_send_mail(dev, dir, 0);
	if (res < 0) {
		count = res;
		goto error;
	}
	res = rmtproc_send_mail(dev, bus_addr, 0);
	if (res < 0) {
		count = res;
		goto error;
	}
	res = rmtproc_send_mail(dev, ((count + 3) & ~0x3), 0);
	if (res < 0) {
		count = res;
		goto error;
	}

	/* wait for ack: notification of completion */
	rmtproc_recv_mail(dev, &tmp);

error:
	pci_unmap_single(dev->pci_dev, bus_addr, PAGE_SIZE, dir);

	/* stabilizer (though bugus :) */
	*(volatile unsigned long *)dma_data = tmp;

	return count;
}

	/*

ssize_t rmtproc_llseek(unsigned int fd, unsigned long offset_high,
											 unsigned long offset_low, loff_t *result,
											 unsigned int whence)
{
	printk("rmtproc_llseek\n");
	switch (whence) {
	case SEEK_SET:
		break;
	}
}
	*/



ssize_t rmtproc_read_from_bap(Rmtproc_Dev *dev, unsigned long addr, char *buf, size_t count)
{
	int i, j;
	size_t retval = 0;
	unsigned long len, data;
	printk(KERN_WARNING PFX "rmtproc_read_from_bap():\n");

	printk(KERN_WARNING PFX "addr = 0x%x buf = 0x%x count = 0x%x\n",
				 addr, buf, count);

	/* read data from local bus access port */
	for (i = 0; i < count; i += sizeof(unsigned long)) {
		//	printk(KERN_WARNING PFX "i = 0x%x\n", i);
		/* write to the local address */
		rmtproc_write_local_ad(dev, addr + i);
		/* read from the local address port */
		data = rmtproc_read_bap(dev);
		//printk(KERN_WARNING PFX "data = 0x%x\n", data);

		//printk(KERN_WARNING PFX "i + sizeof(unsigned long) = 0x%x count = %x\n",
		//i + sizeof(unsigned long), count);
		if (i + sizeof(unsigned long) > count) {
			len = count - i;
			for (j = 0; j < len; j++) {
				buf[i+j] = (char)((data >> j * sizeof(char)) & 0x000000ff);
			}
		}
		else {
			*(unsigned long*)(buf + i) = data;
			len = sizeof(unsigned long);
		}
		retval += len;
	}

	printk(KERN_WARNING PFX "rmtproc_read_from_bap() end:\n");
	return retval;
}

ssize_t rmtproc_read_from_dma(Rmtproc_Dev *dev, char *buf, size_t count)
{
	printk(KERN_WARNING PFX "rmtproc_read_from_dma:\n");

	int i, j;
	size_t retval = 0;
	unsigned long *addr = (unsigned long*) dev->Rdma_data;
	unsigned long len, data;


	rmtproc_read_dma(dev, count);

	printk(KERN_WARNING PFX "rmtproc_read_from_dma: count = %d\n", count);

	for (i = 0; i < count; i += sizeof(unsigned long)) {
		/* read from the local address port */
		data = addr[i];
		if (i + sizeof(unsigned long) > count) {
			len = count - i;
			for (j = 0; j < len; j++) {
				buf[i+j] = (char)((data >> j * sizeof(char)) & 0x000000ff);
			}
		}
		else {
			*(unsigned long*)(buf + i) = data;
			len = sizeof(unsigned long);
		}
		retval += len;
	}
	// return 0;
	return retval;
	
}


ssize_t rmtproc_read(struct file *filp, char *buf, size_t count, loff_t *f_pos)
{
	Rmtproc_Dev *dev;
	size_t retval = 0;
	unsigned long cmd, addr, size;

	printk(KERN_WARNING PFX "rmtproc_read entered\n");
	
	if (count < 0)
		return 0;

	dev = filp->private_data;

	if (down_interruptible(&dev->sem))
		return (-ERESTARTSYS);

	/* receive a mail */
	rmtproc_send_mail(dev, PCI_CMD_READ, 0);
	rmtproc_recv_mail(dev, &cmd);
	cmd = word_endian(cmd);
	printk(KERN_WARNING PFX "rmtproc_read cmd: %08x\n", (int)cmd);

	rmtproc_send_mail(dev, PCI_CMD_READ, 0);
	rmtproc_recv_mail(dev, &addr);
	addr = word_endian(addr);
	printk(KERN_WARNING PFX "rmtproc_read addr: %08x\n", (int)addr);

	rmtproc_send_mail(dev, PCI_CMD_READ, 0);
	rmtproc_recv_mail(dev, &size);
	size = word_endian(size);
	printk(KERN_WARNING PFX "rmtproc_read size: %d\n", (int)size);

	if (size > count)
		size = count;

	switch (cmd)
	{
	case PCI_CMD_READ_BAP:
		retval = rmtproc_read_from_bap(dev, addr, buf, size);
		break;
	case PCI_CMD_READ_DMA:
		retval = rmtproc_read_from_dma(dev, buf, size);
		break;
	default:
		printk(KERN_WARNING PFX "rmtproc_read invalid cmd\n");
	}

	printk(KERN_WARNING PFX "rmtproc_read done\n");
	up(&dev->sem);

	return retval;
}

/*
 * this function writes @buf whose size is @count to PCI bus.
 */
ssize_t rmtproc_write(struct file *filp, const char *buf, size_t count,
					  loff_t *f_pos)
{
	Rmtproc_Dev *dev;
	int res = 0;
	unsigned long len, data, mb_l, mb_h;
	unsigned long addr, size;
	int i, j;
	size_t retval = 0;
	char *program = NULL;

	printk(KERN_WARNING PFX "Write start\n");
	printk(KERN_WARNING PFX "Data size: %d\n", count);

	/* check if the value of the size is valid */
	if (count < 0)
		return -EFAULT;

	/* PCI device */
	dev = filp->private_data;
		
	/* get the semaphore */
	if (down_interruptible(&dev->sem)) {
		return (-ERESTARTSYS);
	}

	/* send a mail */
	if (count == 0) {
		mb_l = ((unsigned long*)buf)[0];
		mb_h = ((unsigned long*)buf)[1];
		printk(KERN_WARNING PFX "Send mail: 0x%lx, 0x%lx\n",
			   mb_l, mb_h);
		res = rmtproc_send_mail(dev, mb_l, mb_h);
		retval = 1;
	}
	/* write to the bus access port */
	else {
		addr = ((unsigned long*)buf)[0];
//		printk(KERN_WARNING PFX "Addr: 0x%lx\n", addr);
		size = ((unsigned long*)buf)[1];
//		printk(KERN_WARNING PFX "Size: %lu\n", size);

		program = (char*)(buf + sizeof(unsigned long) * 2);

		/* write to the local address */
//		rmtproc_write_local_ad(dev, addr);
		/* write data to local bus access port */
		retval = 0;
		for (i = 0; i < count; i += sizeof(unsigned long)) {
			if (i + sizeof(unsigned long) > count) {
				data = 0;
				len = count - i;
				for (j = 0; j < len; j++) {
					data |= (unsigned long)
						((char)program[i+j] << j * sizeof(char));
				}
			}
			else {
				data = *(unsigned long*)(program + i);
				len = sizeof(unsigned long);
			}
			retval += len;

			/* write to the local address */
			rmtproc_write_local_ad(dev, addr + i);
			/* write to the bus access port */
			res = rmtproc_write_bap(dev, data);
		}
		printk(KERN_WARNING PFX "data transfered\n");
	}
	printk(KERN_WARNING PFX "rmtproc_write done\n");
	
	up(&dev->sem);
	return retval;
}

/*
 * the structure storing the rmtprco file operations
 */
struct file_operations rmtproc_fops = {
	.owner  	= THIS_MODULE,
	.llseek  	= NULL,//rmtproc_llseek,
	.read   	= rmtproc_read,
	.write  	= rmtproc_write,
	.ioctl  	= NULL,//rmtproc_ioctl,
	.open   	= rmtproc_open,
	.release	= rmtproc_release,
	.poll   	= NULL,//rmtproc_poll,
};

/*
 * This structure is used to define a list of the different types
 * of PCI devices that a driver support.
 *
 * In this case, it creates a list of IDs with an empty structure
 * set to all zerosas the last value in the list, which means
 * only the vender id and the device id are set.
 */
static struct pci_device_id rmtproc_ids[] = {
	{ PCI_DEVICE(PCI_VENDER_ID_RMT, PCI_DEVICE_ID_RMT) },
	{ 0, }
};


/* 
 * allocate a page for the dma transfer buffers.
 * this function is called when the Rmtproc_Dev strucure of drvdata
 * is initialized.
 */
static int rmtproc_init_buf(struct Rmtproc_Dev *rmtproc)
{
	rmtproc->Rdma_data = __get_free_pages(GFP_KERNEL | GFP_DMA,
										  dma_page_order);
	if (!(rmtproc->Rdma_data)) {
		goto err_out_ralloc_fail;
	}

	rmtproc->Wdma_data = __get_free_pages(GFP_KERNEL | GFP_DMA,
										  dma_page_order);
	if (!(rmtproc->Wdma_data)) {
		goto err_out_walloc_fail;
	}

	return 0;

err_out_walloc_fail:
	free_pages(rmtproc->Rdma_data, dma_page_order);

err_out_ralloc_fail:
	printk(KERN_INFO PFX "NO more memory for dma buffer");

	return -ENOMEM;
}

static void __devinit rmtproc_init_board(struct pci_dev *pdev)
{
#if 0
	pci_write_config_dword(pdev, PCI_COMMAND,
			       PCI_COMMAND_SERR | PCI_COMMAND_PARITY
			       | PCI_COMMAND_IO);
	//pci_write_config_byte(pdev, PCI_LATENCY_TIMER, 0x40);
#endif

	/* clear master abort status */
	pci_write_config_word(pdev, PCI_STATUS, PCI_STATUS_REC_MASTER_ABORT);
}

#ifdef RMTP_DRV
/*
 * This funtion initializes and registers the kernel internal char
 * device which is embedded within struct rmtproc_dev.
 */
static int rmtproc_setup_cdev(struct Rmtproc_Dev *dev, int index)
{
	int err;
	int devno = MKDEV(rmtproc_major, rmtproc_minor + index);

	/* initialize the structure*/
	cdev_init(&dev->cdev, &rmtproc_fops);
	dev->cdev.owner = THIS_MODULE;
	dev->cdev.ops = &rmtproc_fops;
	/* register the char device */
	err = cdev_add(&dev->cdev, devno, 1);

	return err;
}

/*
 * The probe function for the PCI driver.
 * This function is called by the PCI core when it has a struct pci_dev
 * that it thinks this driver wants to control.
 */
static int __devinit 
rmtproc_probe(struct pci_dev *dev, const struct pci_device_id *id)
{
	int err; /* error code */
	Rmtproc_Dev *rmtproc; /* device information */
	dev_t devno = 0; /* 12 bits for major no. and 20 for minor no. */
	static int index = 0; /* minor number index */

	/* the driver must call this function before it can access any 
	   device resource of the PCI device.
	   It wakes up the device and also assigns its interrupt line
	   and I/O regions in some cases */
	err = pci_enable_device(dev);
	if (err) {
		printk(KERN_ERR PFX "Failed to enable the RMT PCI device.\n");
		return err;
	}

	/* check whether I/O memory is available */
	
	/* mark all the PCI regions associated with the struct pci_dev 
	   as being reserved by owner THIS_MODULE_NAME */
	err = pci_request_regions(dev, THIS_MODULE_NAME);
	if (err) {
		printk(KERN_ERR PFX "Cannot obtain PCI resources.\n");
		goto err_disable_dev;
	}

#if 1
	/* just to make it sure... */
	pci_set_master(dev);

	/* check which DMA attribute is enable */
	err = pci_set_dma_mask(dev, 0xffffffffffffffffULL);
	if (!err) {
		printk(KERN_ALERT PFX "64-bit DMA is available.\n");
	} else {
		err = pci_set_dma_mask(dev, 0xffffffffULL);
		if (err) {
			printk(KERN_ERR PFX "No usable DMA configuration\n");
			goto err_free_res;
		}
		printk(KERN_ALERT PFX "32-bit DMA is available.\n");
	}
#endif

	/* init work ? */
	rmtproc_init_board(dev);

	/* device information */
	rmtproc = kmalloc(sizeof(struct Rmtproc_Dev), GFP_KERNEL);
	if (!rmtproc) {
		printk (KERN_ERR PFX "Cannot allocate device struct.\n");
		goto err_free_res;
	}
	memset(rmtproc, 0, sizeof(struct Rmtproc_Dev));
	sema_init (&rmtproc->sem, 1);
	rmtproc_init_buf(rmtproc);
	pci_set_drvdata(dev, rmtproc);

	/* obtain one or more device numbers to work with */
	if (rmtproc_major) {
		devno = MKDEV(rmtproc_major, rmtproc_minor);
		err = register_chrdev_region(devno, rmtproc_nr_devs, THIS_MODULE_NAME);
	}
	else {
		/* Here we usually dont know which major numbers our device
		   will use. in this case, we can happily use the following
		   function. */
		err = alloc_chrdev_region(&devno, 0, 1, THIS_MODULE_NAME);
		rmtproc_major = MAJOR(devno);
		rmtproc_minor = MINOR(devno);
	}
	if (err) {
		printk (KERN_ERR PFX "Cannot allocate a char device.\n");
		goto err_free_rmtproc;
	}
	
	/* the base address associated with PCI I/O region #1 */
	rmtproc->addr_base = pci_resource_start(dev, 0);
	/* the length of the region */
	rmtproc->addr_len = pci_resource_len(dev, 0);
	/* remap the address cuz this is I/O memory */
	rmtproc->addr_base = (unsigned long) ioremap(rmtproc->addr_base,
												 rmtproc->addr_len);

	/* the last address associated with PCI I/O region #1 
	   note that this is the last usable address, not the first
	   address after the region. */
//	rmtproc->addr_end = pci_resource_end(dev, 0);
	/* the irq of this device */
	rmtproc->irq = dev->irq;
	printk(KERN_WARNING PFX "IRQ:%d\n", dev->irq);
	/* setup the kernel internal char device */
	err = rmtproc_setup_cdev(rmtproc, index);
	if (err) {
		printk (KERN_ERR PFX "Cannot setup an internal char device.\n");
		goto err_cdev;
	}
	index++;

#if 0	
	dword = ioread32((void*)rmtproc->addr_base);
	printk(KERN_WARNING PFX "Base Address: 0x%08x\n", dword);

	/* check PCI -> Local communication */
	rmtproc_write_local_ad(rmtproc, PCI_TEST_ADDR);
	rmtproc_write_bap(rmtproc, PCI_TEST_DATA1);
	rmtproc_send_mail(rmtproc, PCI_TEST_DATA1, PCI_TEST_DATA2);
	printk(KERN_WARNING PFX "RMT-PCI Check Done\n");
#endif

	return 0;

err_cdev:
	cdev_del(&rmtproc->cdev);

err_free_rmtproc:
	kfree(rmtproc);

err_free_res:
	pci_release_regions(dev);

err_disable_dev:
	pci_disable_device(dev);
	pci_set_drvdata(dev, NULL);
	
	/* the procedure failed */
	return err;

}

/*
 * This function is called by the PCI core when the struct pci_dev is
 * being removed from the syste, or when the PCI driver is being
 * unloaded from the kernel.
 */
static void __devinit rmtproc_remove(struct pci_dev *dev)
{
	struct Rmtproc_Dev *rmtproc = pci_get_drvdata(dev);
	
	if (rmtproc) {
		if (rmtproc->Rdma_data)
			free_page(rmtproc->Rdma_data);

		if (rmtproc->Wdma_data)
			free_page(rmtproc->Wdma_data);

		iounmap((void *)rmtproc->addr_base);
		cdev_del(&(rmtproc->cdev));
		pci_release_regions(dev);
		pci_disable_device(dev);
		pci_set_drvdata(dev, NULL);
		if (rmtproc)
			kfree(rmtproc);
	}
}

#else

static int __devinit
rmtproc_probe(struct pci_dev *pdev, const struct pci_device_id *id)
{
	struct Rmtproc_Dev *rmtproc;
	int err;
	static dev_t dev = 0;
	struct cdev *cdev;
	static int devnum = 0;

	err = pci_enable_device(pdev);
	if (err) {
		printk(KERN_ERR PFX "failed to enable PCI device.\n");
		return err;
	}

	/* check whether IO port is available */
	if (!(pci_resource_flags(pdev, 0) & IORESOURCE_MEM)) {
		printk (KERN_ERR PFX "No proper device with iomem found.\n");
		err = -ENODEV;
		goto err_disable_dev;
	}

	err = pci_request_regions(pdev, THIS_MODULE_NAME);
	if (err) {
		printk(KERN_ERR PFX "Cannot obtain PCI resources.\n");
		goto err_disable_dev;
	}

	pci_set_master(pdev);

	/* Configure DMA attributes */
	err = pci_set_dma_mask(pdev, 0xffffffffffffffffULL);
	if (!err) {
		PDEBUG("64 bit DMA can be used for " THIS_MODULE_NAME "\n");
	} else {
		err = pci_set_dma_mask(pdev, 0xffffffffULL);
		if (err) {
			printk(KERN_ERR PFX "No usable DMA configuration\n");
			goto err_free_res;
		}
		PDEBUG("32 bit DMA can be used for " THIS_MODULE_NAME "\n");
	}

	rmtproc_init_board(pdev);
	rmtproc = kmalloc( sizeof (struct Rmtproc_Dev), GFP_KERNEL);
	if (!rmtproc) {
		printk (KERN_ERR PFX "Cannot allocate device struct.\n");
		goto err_free_res;
	}
	memset(rmtproc, 0, sizeof (*rmtproc));
	sema_init (&rmtproc->sem, 1);
	rmtproc_init_buf(rmtproc);
	pci_set_drvdata(pdev, rmtproc);


	/* Char device */
	if (dev == 0) {
		err = alloc_chrdev_region(&dev, 0, 1, THIS_MODULE_NAME);
		if (err) {
			printk (KERN_ERR PFX "Cannot allocate chrdev.\n");
			goto err_free_rmtproc;
		}
	} else {
		dev = MKDEV(MAJOR(dev), MINOR(dev)+1);
#if 0
		err = register_chrdev_region(dev, 1, THIS_MODULE_NAME);
		if (err) {
			printk (KERN_ERR PFX "Cannot register chrdev.\n");
			goto err_free_rmtproc;
		}
#endif
	}

	cdev = cdev_alloc();
	if (!cdev) {
		printk (KERN_ERR PFX "Cannot allocate cdev.\n");
		err = -ENOMEM;
		goto err_free_rmtproc;
	}

	rmtproc->cdev = cdev;
	rmtproc->addr_base = pci_resource_start(pdev, 0);
	rmtproc->addr_len = pci_resource_len(pdev, 0);
	rmtproc->addr_base =
		(unsigned long)ioremap(rmtproc->addr_base, rmtproc->addr_len);
	printk(KERN_WARNING PFX "RMTP PCI-base: 0x%lx\n", rmtproc->addr_base);
	rmtproc->irq = pdev->irq;
	kobject_set_name(&cdev->kobj, "rmtproc%d", devnum);
	devnum++;
	cdev->owner = THIS_MODULE;
	cdev->ops = &rmtproc_fops;

	err = cdev_add(cdev, dev, 1);
	if (err) {
		goto err_cdev;
	}

	list_add(&rmtproc->list, &rmtproc_drivers);

	return 0;

err_cdev:
	kobject_del(&cdev->kobj);

err_free_rmtproc:
	kfree(rmtproc);

err_free_res:
	pci_release_regions(pdev);

err_disable_dev:
	pci_disable_device(pdev);
	pci_set_drvdata(pdev, NULL);

	return err;
}

static void __devinit rmtproc_remove(struct pci_dev *pdev)
{
	struct Rmtproc_Dev *rmtproc = pci_get_drvdata(pdev);

	if (rmtproc) {
		if (rmtproc->cdev)
			unregister_chrdev_region(rmtproc->cdev->dev, 1);

		if (rmtproc->Rdma_data)
			free_page(rmtproc->Rdma_data);

		if (rmtproc->Wdma_data)
			free_page(rmtproc->Wdma_data);

		iounmap((void *)rmtproc->addr_base);

		cdev_del(rmtproc->cdev);
		pci_release_regions(pdev);
		pci_disable_device(pdev);
		pci_set_drvdata(pdev, NULL);

		list_del(&rmtproc->list);
		kfree(rmtproc);
	}
}

#endif

/*
 * This structure consists of a number of function callbacks 
 * and variables that describe the PCI driver to the PCI core.
 *
 * const char *name;
 *		The name of the driver. It must be unique.
 *		It will be in /sys/bus/pci/drivers when it is installed.
 * const struct pci_device_id *id_table;
 * int (*probe) (struct pci_dev *dev, const struct pci_device_id *id);
 * void (*remove) (struct pci_dev *dev);
 * int (*suspend) (struct pci_dev *dev, u32 state);
 *		This function is optional; we dont have to provide it.
 * int (*resume) (struct pci_dev *dev);
 *		This function is optional; we dont have to provide it.
 */
static struct pci_driver pci_driver = {
	.name = THIS_MODULE_NAME,
	.id_table = rmtproc_ids,
	.probe = rmtproc_probe,
	.remove = rmtproc_remove
};

/*
 * Entry point of the module
 */
static int __init rmtproc_init(void)
{
	/* greeting message will be in /var/log/messages */
	printk(KERN_ALERT PFX "Connecting to RMTP via PCI...\n");
	/* register the pci driver */
	return pci_register_driver(&pci_driver);
}

/*
 * Leaving point of the module
 */
static void __exit rmtproc_exit(void)
{
	/* leaving message will be in /var/log/messges */
	printk(KERN_ALERT PFX "Goodbye, RMTP!\n");
	/* unregister the pci driver */
	pci_unregister_driver(&pci_driver);
}



module_init(rmtproc_init);
module_exit(rmtproc_exit);

