/*
 * FILE
 *	flash.c
 *
 *	Copyright(c) 1994, FMI, Fujitsu Microelectronics, Inc.
 *	All rights reserved.
 *
 *	This software (including any documentation) is untested, has not been
 *	fully tested for viruses and has been provided to you without charge.
 *	ACCORDINGLY, IT IS DELIVERED "AS IS" WITH NO WARRANTIES EXPRESS OR
 *	IMPLIED, INCLUDING WITHOUT LIMITATION WARRANTIES OF MERCHANTABILITY,
 *	FITNESS FOR A PARTICULAR PURPOSE OR NONINFRINGEMENT.  You bear all 
 *	risk of nonperformance, loss of data and other problems and Fujitsu
 *	Microelectronics, Inc. and Fujitsu Limited will not be liable under any
 *	contract, negligence, strict liability or other theory for any damages
 *	including, without limitation, direct, consequential or incidental nor
 *	be required to provide substitute goods, services or technology.
 *
 * DESCRIPTION
 *	AMD AM29F010 FLASH RAM diagnostic
 *
 * HISTORY
 *	November 9, 1994	DCL	inititial version
 *	November 17, 1994	DCL	clean-up
 */
 
#define FLASH_SIZE	(128*1024)
#define FLASHRAM_START	0x3000000

#define FLASHC(addr) (*(unsigned char *)(FLASHRAM_START + addr))
#define FLASHL(addr) (*(unsigned long *)(FLASHRAM_START + addr))

/*
* SETUP_RD()
*
*	write the read command sequence.
*/
#define SETUP_RD() \
	(FLASHC(0x5555) = 0xaa, \
	FLASHC(0x2aaa) = 0x55, \
	FLASHC(0x5555) = 0xf0)

/*
* SETUP_WR()
*
*	write the program command sequence.
*/
#define SETUP_WR() \
	(FLASHC(0x5555) = 0xaa, \
	FLASHC(0x2aaa) = 0x55, \
	FLASHC(0x5555) = 0xa0)

/*
* FLASH_WRCHAR(addr, data)
*
*	write a byte to flashram
* 
* 1. write program command sequence.
* 2. write data to location addr.
* 3. poll for valid data.
*
*	return:  flash_poll return value
*/
#define FLASH_WRCHAR(addr, data) \
	(SETUP_WR(), \
	FLASHC(addr) = data, \
	flash_poll(addr, data))

/*
* FLASH_WRLONG(addr, data)
*
*	write a long to flashram
* 
*       Since each byte requires a program command sequence
*       and poll, each byte must be written separately.
*
* For bytes 3-0:
*
* 1. call FLASH_WRCHAR.
*
*	return:  FLASH_WRCHAR() logical-or
*/
#define FLASH_WRLONG(addr, data) \
	(FLASH_WRCHAR(addr, ((data) >> 24)) | \
	FLASH_WRCHAR(addr+1, ((data) >> 16)) | \
	FLASH_WRCHAR(addr+2, ((data) >> 8)) | \
	FLASH_WRCHAR(addr+3, (data)))

/*
* polling masks
*/
#define D5_MASK 0x20
#define D6_MASK 0x40
#define D7_MASK 0x80

/*
* polling macros
*/
#define D5(data) (data & D5_MASK)
#define D6(data) (data & D6_MASK)
#define D7(data) (data & D7_MASK)

/*
* flash_test()
*
*	flash memory test
*
* 1. address test.
* 2. pattern test (if EXTENDED_FLASHTEST is defined).
* 3. erase flashram.
*
*	return: -1 if any error.
*	return: 0 if no errors.
*/
flash_test()
{
	register unsigned long addr;
	register unsigned long range;

	addr = 0L;
	range = FLASH_SIZE;

	if (address_test(addr, range, 0xaaaaaaaa) == -1)
		return -1;

#ifdef EXTENDED_FLASHTEST 
	if (address_test(addr, range, 0x55555555) == -1)
		return -1;

	if (pat_test(addr, range) == -1)
		return -1;
#endif

	if (flash_erase() == -1)
		return -1;

	return 0;
}

/*
* address_test(addr, range, mask)
*
*	flash address test
*
* Write 32-bit address into each 32-bit location, with
* an optional mask to find adjacent data-bit shorts.
*
*	return: -1 if any erase/read/write error.
*	return: 0 if no errors.
*/
address_test(addr, range, mask)
register unsigned long addr;
register unsigned long range;
register unsigned long mask ;
{
	unsigned long start, end;

	start = addr;
	end = start + range;

	if (flash_erase() == -1)
		return -1;

	while (addr < end) {
		if (FLASH_WRLONG(addr, addr ^ mask) == -1)
			return -1;

		addr += sizeof(long);
	}

	SETUP_RD();

	addr = start;
	while (addr < end) {
		if (FLASHL(addr) != (addr ^ mask))
			return -1;

		addr += sizeof(long);
	}

	return 0;
}

/*
* pat_test(addr, range)	flash pattern test
*
* 1. Erase flashram (0xff into all locations).
* 2. Write 8-bit pattern 0xaa into all locations,
*    checking location+1 for last pattern (0xff).
* 3. Write 8-bit pattern 0x00 into all locations,
*    checking location+1 for last pattern (0xaa).
*    
*	return: -1 if any erase, read, or write failure.
*	return: 0 if no errors.
*/
pat_test(addr, range)
register unsigned long addr;
register unsigned long range;
{
	register unsigned char pat;

	pat = 0xaa;

	if (flash_erase() == -1)
		return -1;

	/*
	* write pat, and check for 0xff at loc+1
	*/

	if (pat_sub(addr, addr+range, pat, 0xff) == -1)
		return -1;

	/*
	* write 0x0, and check for pat at loc+1
	*/
	if (pat_sub(addr, addr+range, 0, pat) == -1)
		return -1;

	return 0;
}

int
pat_sub(addr, end, pat, lpat)
register unsigned long addr;
register unsigned long end;
register unsigned char pat;
register unsigned char lpat;
{
	/* lpat --> pat */
	while (addr < end-1) {
		if (FLASH_WRCHAR(addr, pat) == -1)
			return -1;

		/*
		* change to read mode
		*/
		SETUP_RD();

		/* 
		* verify what we wrote
		*/
		if (FLASHC(addr) != pat)
			return -1;

		/* 
		* check the next location for last pattern
		*/
		if (FLASHC(++addr) != lpat)
			return -1;
	}

	/* write the last location */
	if (FLASH_WRCHAR(addr, pat) == -1)
		return -1;

	return 0;
}

/*
* flash_erase()
*
*	erase the entire part.
*
* 1. write the chip erase command sequence.
* 2. poll for completion.
*
*	return: return from flash_poll().
*/
int
flash_erase()
{
	FLASHC(0x5555) = 0xaa;
	FLASHC(0x2aaa) = 0x55;
	FLASHC(0x5555) = 0x80;
	FLASHC(0x5555) = 0xaa;
	FLASHC(0x2aaa) = 0x55;
	FLASHC(0x5555) = 0x10;

	return flash_poll(0x1234, 0x80);
} 

/*
* flash_poll(addr, data)  AMD Am29F010 flashram polling routine
*
* 1. Read byte at addr.
* 2. If bit 7 matches bit 7 of data, return 0.
* 3. If bit 5 is 1, goto 1.
* 4. Read byte at addr, 
* 5. If bit 7 matches bit 7 of data, return 0.
* 6. Return -1.
*
*	return: -1 if error.
*	return: 0 if no errors.
*/
int
flash_poll(addr, data)
register unsigned long addr;
register unsigned char data;
{
	unsigned char d7, tdata, dq5, dq7;
	int i;
	
	d7 = D7(data);

	for (i=0; i<10000000; ++i) {

		tdata = FLASHC(addr);

		dq7 = D7(tdata);
		dq5 = D5(tdata);

		if (dq7 == d7)
			return 0;

		if (dq5) {
			if (D7(FLASHC(addr)) == d7)
				return 0;
			else
				return -1;
		}
	}

	/*
	* timeout
	*/
	return -1;
}
