/*
 * FILE
 *      ne.c
 * 
 *	Copyright(c) 1993, FMI, Fujitsu Microelectronics, Inc.
 *	All rights reserved.
 *
 *	This software (including any documentation) is untested, has not been
 *	fully tested for viruses and has been provided to you without charge.
 *	ACCORDINGLY, IT IS DELIVERED "AS IS" WITH NO WARRANTIES EXPRESS OR
 *	IMPLIED, INCLUDING WITHOUT LIMITATION WARRANTIES OF MERCHANTABILITY,
 *	FITNESS FOR A PARTICULAR PURPOSE OR NONINFRINGEMENT.  You bear all 
 *	risk of nonperformance, loss of data and other problems and Fujitsu
 *	Microelectronics, Inc. and Fujitsu Limited will not be liable under any
 *	contract, negligence, strict liability or other theory for any damages
 *	including, without limitation, direct, consequential or incidental nor
 *	be required to provide substitute goods, services or technology.
 *
 * DESCRIPTION
 *	This file contains source code for UDP/IP implementation on 
 *	SPARClite evaluation board.
 *
 * HISTORY
 *      April 1993      Clark Li
 */

#include "splite.h"
#include "net.h"

#define	RECV_NORMAL	0
#define	RECV_ALL	1
#define	MYPORT		7000

#define	ETHER_HLEN	14
#define	IP_HLEN		20
#define	UDP_HLEN	8

#define CMD_GET_STATUS      0x00
#define CMD_LOAD            0x01
#define CMD_RUN_SR          0x02
#define CMD_EXEC            0x03
#define CMD_MOVE_TRAP       0x04
#define CMD_REP_VER         0x05

#define	LD_START	0x00
#define	LD_ADDR		0x01
#define	LD_DATA		0x02

#define NE_CLR_RXPKT()	wrasi4b(NE_DLCR1, rdasi4b(NE_DLCR1) | 0x80)	
#define	NE_NORM_RX()	wrasi4b(NE_DLCR5, 0x02)

#ifdef MASTER_CK_14
#define	SHOW_DIAG_TIME	0x1d8000
#endif
#ifdef MASTER_CK_20
#define	SHOW_DIAG_TIME	0x210000
#endif
#ifdef MASTER_CK_25
#define	SHOW_DIAG_TIME	0x294000
#endif
#ifdef MASTER_CK_30
#define SHOW_DIAG_TIME	0x318000
#endif
#ifdef MASTER_CK_40
#define	SHOW_DIAG_TIME	0x80000
#endif
#ifdef MASTER_CK_50
#define SHOW_DIAG_TIME	0xb0000
#endif
#ifdef MASTER_CK_60
#define SHOW_DIAG_TIME	0xe0000
#endif

uchar	myip[4];
uchar   myether[6] = {0x0,0x0,0xE,0x31,0x0,0x0};
uchar	hostip[4];
uchar	hostether[6];
uchar	etherbroad[6] = {0xFF,0xFF,0xFF,0xFF,0xFF,0xFF};
uchar	com_ready = 0;

uchar	netibuf[1600], netobuf[1600];
ushort	*inbufp, *outbufp;
ushort	inpkt_len, outpkt_len;

UDPPKT	udpbuf;
char	*p_udpd = (char *)&udpbuf.data[0];
char	led_save = 0x0;				/* save LED value */

ushort	ipchecksum();
ulong	atol();

ushort	inpkseq;

extern uchar	version;

/*-----------------------------------------------------------------------------
 * NAME      : main
 *
 * PURPOSE   : Test NICE loopback and Ethernet connection. If any test fails,
 *	       return to the caller; otherwise, enter Ethernet monitor.
 *
 * NOTES     : called by SPARClite EB diagnostic program
 *
 * ARGUMENTS : -
 *
 * RETURNS   : 1	lookback test failed
 *	       2	Ethernet is not connected
 *---------------------------------------------------------------------------*/
_main(memsize)
{
	int	i, trytimes;

#ifdef DEBUG
	printf("*************************************\r\n");
#endif

	/* show diagnostic results on LEDs for about two seconds */
	for (i=0; i<SHOW_DIAG_TIME; i++);

	/* turn off LEDs for about one second */
	*(uchar *)0x2000003 = 0xff;
	for (i=0; i<SHOW_DIAG_TIME; i++);

	/* show amount of memory on LEDs for about two seconds */
	*(uchar *)0x2000003 = (~(memsize >> 20));
	for (i=0; i<SHOW_DIAG_TIME*2; i++);

	/* turn off LEDs for about one second */
	*(uchar *)0x2000003 = 0xff;
	for (i=0; i<SHOW_DIAG_TIME; i++);

	/* complement the value read from the DIP switch and mark out
	   the upper four bits. */
	myether[5] = 0x0F & (~(*(uchar *)0x2000003)); 

	ne_init();

        *(uchar *)0x2000003 = 0xFF;     /* before test, clear LEDs */

        if (ne_loopback_test())
                return(1);              /* loopback failed */
        else
                *(uchar *)0x2000003 = ~(led_save |= 0x20);
                                        /* loopback passed, light LED bit 5 */

        if (ne_check_ether())
                return(2);              /* Ethernet connection failed */
        else
                *(uchar *)0x2000003 = ~(led_save |= 0x40);
					/* connection passed, light LED bit 6*/
	ne_send_rarp();		
	ne_recv();
}


/*-----------------------------------------------------------------------------
 * NAME      : ne_init
 * 
 * PURPOSE   : Initialize NICE chip
 * 
 * NOTES     : -
 * 
 * ARGUMENTS : -
 * 
 * RETURNS   : -
 *---------------------------------------------------------------------------*/
ne_init()
{
	/* Write node id '222FMI' to IDR */
	wrasi4b(NE_DLCR6, 0x80); 	/* disable data link to access ID regs*/
	wrasi4b(NE_DLCR7, 0x20); 	/* set addr to IDR */

        wrasi4b(NE_IDR8,  myether[0]);	/* my ethernet address */
        wrasi4b(NE_IDR9,  myether[1]);
        wrasi4b(NE_IDR10, myether[2]);
        wrasi4b(NE_IDR11, myether[3]);
        wrasi4b(NE_IDR12, myether[4]);
        wrasi4b(NE_IDR13, myether[5]);

	/* set configuration */
	wrasi4b(NE_DLCR6, 0x45);	/* config */
	wrasi4b(NE_DLCR7, 0x28);	/* config */
	wrasi4b(NE_DLCR4, 0x06);	/* TX mode */
	wrasi4b(NE_DLCR5, 0x02);	/* RX mode */
}


/*-----------------------------------------------------------------------------
 * NAME      : ne_init_proto
 * 
 * PURPOSE   : Initialize UDP packet data.  Leave the unknown entries open.
 * 
 * NOTES     : -
 * 
 * ARGUMENTS : -
 * 
 * RETURNS   : -
 *---------------------------------------------------------------------------*/
ne_init_proto()
{
	mem_move(hostether, udpbuf.d.dst, 6);		
	mem_move(myether, udpbuf.d.src, 6);
	udpbuf.d.type = EIP;

	udpbuf.i.versionandhdrlen = 0x45;
	udpbuf.i.service = 0x00;
/*	udpbuf.i.tlen = 0x????;			?: IP total len */
	udpbuf.i.ident = 0x0000;
	udpbuf.i.frags = 0x0000;
	udpbuf.i.ttl = 0xFF;
	udpbuf.i.protocol = PROTUDP;
/*	udpbuf.i.check = 0x0000;		?: reset every time */
	mem_move(myip, udpbuf.i.ipsrc, 4);		
	mem_move(hostip, udpbuf.i.ipdst, 4);	
/*	udpbuf.i.check = ipchecksum(udpbuf.i, 20);	?: IP checksum */

	udpbuf.u.source = MYPORT;
/*	udpbuf.u.dest = ?????;			?: host UDP port */
/*	udpbuf.u.length = 8 + 18;		?: UDP total len */
	udpbuf.u.check = 0x0000;
}


/*-----------------------------------------------------------------------------
 * NAME      : ne_send
 * 
 * PURPOSE   : Write data to NICE buffer
 * 
 * NOTES     : -
 * 
 * ARGUMENTS : buf	buffer address
 *	       size	length of buffer
 * 
 * RETURNS   : -
 *---------------------------------------------------------------------------*/
ne_send(buf, size)
ushort	*buf;
uint	size;
{
	uint	i;

	while (rdasi4b(NE_DLCR0) & 0x40);	/* net is busy */

	wrasi4h(NE_BMR8, ne_swap(size<60 ? 60 : size));	
						/* NICE header, len >= 60 */
	for (i = 0; i < size/2; i++)
		wrasi4h(NE_BMR8, *buf++);
	if (size & 0x1)				/* odd number */
		wrasi4h(NE_BMR8, *buf++);
	size += size & 0x1;
	if (size < 60) 
		for (i = 0; i < (60-size)/2; i++)
			wrasi4h(NE_BMR8, 0x00);

	wrasi4b(NE_BMR10, 0x81);		/* start transmit */
	while (!(rdasi4b(NE_DLCR0) & 0x80));	/* waiting for TX done */
}


/*-----------------------------------------------------------------------------
 * NAME      : ne_send_rarp
 * 
 * PURPOSE   : Send a RARP packet
 * 
 * NOTES     : -
 * 
 * ARGUMENTS : -
 * 
 * RETURNS   : -
 *---------------------------------------------------------------------------*/
ne_send_rarp()
{
	ARPPKT	*arp = (ARPPKT *)netobuf;

	/* init Ether header */
	mem_move(etherbroad, &arp->d.dst[0], 6);/* ether target hw addr */
	mem_move(myether, &arp->d.src[0], 6);	/* ether source hw addr */
	arp->d.type = ERARP;

	/* init ARP packet */
	arp->hrd = HTYPE;			/* Ether = 1 */
	arp->pro = ARPPRO;			/* IP protocol = 0x0800 */
	arp->hln = 6;				/* Ether h/w addr length */
	arp->pln = 4;				/* IP length = 4 */
	arp->op = RARPQ;			/* ARP request */
	mem_move(myether, &arp->sha[0], 6);	/* source ether addr */
	mem_zero(&arp->spa[0], 4);		/* source ip addr */
	mem_move(myether, &arp->tha[0], 6);	/* init target h/w addr */
	mem_zero(&arp->tpa[0], 4);		/* init target ip addr */
	mem_zero(&arp->pad[0], 18);

	ne_send(arp, sizeof(ARPPKT));
#ifdef DEBUG
	printf("Connect to server\r\n");
#endif
}


/*-----------------------------------------------------------------------------
 * NAME      : ne_recv
 * 
 * PURPOSE   : Receive packets from NICE
 * 
 * NOTES     : called by main()
 *	       In an infinite loop.  Check if there is a packet in the
 *	       NICE receiving buffer.
 * 
 * ARGUMENTS : -
 * 
 * RETURNS   : -
 *---------------------------------------------------------------------------*/
ne_recv()
{
	ushort	sh, len;
	DLAYER	*etherhead;
	uint	i;

#ifdef DEBUG
	printf("Receiving packets ...\r\n");
#endif

	while (1) {
		while (rdasi4b(NE_DLCR5) & 0x40);	/* RX buf empty? */

		sh  = rdasi4h(NE_BMR8);			/* filter 2 bytes */
		inpkt_len = ne_swap(rdasi4h(NE_BMR8));	/* length of packet */

		for (i = 0, inbufp = (ushort *)netibuf; i < inpkt_len/2; i++)
			*inbufp++ = rdasi4h(NE_BMR8);	/* read packet */
		if (inpkt_len & 0x1) 		      /* odd number */
			*inbufp++ = rdasi4h(NE_BMR8); /* clear buf */

		etherhead = (DLAYER *)netibuf;

		switch (etherhead->type) {
		case EARP:
		case ERARP:
			arp_interpret(etherhead);
			break;
		case EIP:
			ip_interpret(etherhead);
			break;
		default:
#ifdef DEBUG
			printf("Received a packet, len = %d\r\n", inpkt_len);
#endif
			break;
		}
	}
}


/*-----------------------------------------------------------------------------
 * NAME      : arp_interpret
 * 
 * PURPOSE   : Decode ARP packet
 * 
 * NOTES     : 
 * 
 * ARGUMENTS :
 * 
 * RETURNS   : 
 *---------------------------------------------------------------------------*/
arp_interpret(p)
ARPPKT	*p;
{
	uint	i;

	/* check for ARP request */
	if (p->op == ARPREQ && mem_comp(&p->tpa[0], myip, 4)) {
#ifdef DEBUG
		printf("Received a ARP request, len = %d\r\n", inpkt_len);
#endif
		replyarp(&p->sha[0], &p->spa[0]);
#ifdef DEBUG
		printf("Sent a ARP reply\r\n");
#endif
		return(0);
	} else

	/* check for RARP reply */
	if (p->op == RARPR && mem_comp(&p->tha[0], myether, 6) && !com_ready) {
		mem_move(&p->tpa[0], myip, 4);		/* get my ip addr */
		mem_move(&p->spa[0], hostip, 4);		/* get host ip addr */
		mem_move(&p->sha[0], hostether, 6);	/* get host ehter addr*/

#ifdef DEBUG
		printf("Received a RARP reply\r\n");
		printf("Server IP = ");
		for (i = 0; i < 4; i++) {
			printf("%d", hostip[i]);
			if (i < 3)
				printf(".");
		}
		printf("  Server Ether = ");
		for (i = 0; i < 6; i++) {
			printf("%x", hostether[i]);
			if (i < 5)
				printf(":");
		}
		printf("\r\n");
#endif

		ne_init_proto();
		com_ready = 1;
		*(uchar *)0x2000003 = ~(led_save |= 0x80);
					/* Ethernet ready, light LED bit 7 */
		return(0);
	}

#ifdef DEBUG
	printf("Received an ARP broadcast\r\n");
#endif

	return(1);
}


/*-----------------------------------------------------------------------------
 * NAME      : replyarp
 * 
 * PURPOSE   : Send an ARP reply packet
 * 
 * NOTES     : 
 * 
 * ARGUMENTS : tha	target hardware address
 *	       tpa	terget IP address
 * 
 * RETURNS   : 
 *---------------------------------------------------------------------------*/
replyarp(tha, tpa)
uchar	*tha,		/* target h/w addr */ 
	*tpa;		/* target ip addr */	
{
	ARPPKT	*arp = (ARPPKT *)netobuf;

	/* init Ether header */
	mem_move(tha, &arp->d.dst[0], 6);	/* ether target hw addr */
	mem_move(myether, &arp->d.src[0], 6);	/* ether source hw addr */
	arp->d.type = EARP;

	/* init ARP packet */
	arp->hrd = HTYPE;			/* Ether = 1 */
	arp->pro = ARPPRO;			/* IP protocol = 0x0800 */
	arp->hln = 6;				/* Ether h/w addr length */
	arp->pln = 4;				/* IP length = 4 */
	arp->op = ARPREP;			/* ARP reply */
	mem_move(myether, &arp->sha[0], 6);	/* source ether addr */
	mem_move(myip, &arp->spa[0], 4);	/* source ip addr */
	mem_move(tha, &arp->tha[0], 6);		/* init target h/w addr */
	mem_move(tpa, &arp->tpa[0], 4);		/* init target ip addr */

	ne_send(arp, sizeof(ARPPKT));
}


/*-----------------------------------------------------------------------------
 * NAME      : ip_interpret
 * 
 * PURPOSE   : Decode IP packet
 * 
 * NOTES     : 
 * 
 * ARGUMENTS : p	pointer to IP packet
 * 
 * RETURNS   : 
 *---------------------------------------------------------------------------*/
ip_interpret(p)
IPPKT	*p;
{
	ushort	iplen, i, inseq;
	UDPPKT	*pudp;

	if (!mem_comp(myip, &p->i.ipdst[0], 4))	/* it is not my ip */
		return(1);			/* discard it */

	iplen = p->i.tlen;
	i = (p->i.versionandhdrlen & 0xF)<<2;

	switch(p->i.protocol) {
	case PROTUDP:
		pudp = (UDPPKT *)p;
		if (pudp->u.dest == MYPORT)
			udp_interpret(pudp);
		break;
	case PROTICMP:
#ifdef DEBUG
		printf("Received a ICMP packet, len = %d\r\n",inpkt_len);
#endif
		return(icmp_interpret(p, iplen - i));
		break;
	case PROTTCP:
#ifdef DEBUG
		printf("Received a TCP packet, len = %d\r\n",inpkt_len);
#endif
		break;
	default:
#ifdef DEBUG
		printf("Received an IP packet\r\n");
#endif
		return(1);
	}
}

ulong	loadaddr;
ushort	dpknum;

/*-----------------------------------------------------------------------------
 * NAME      : udp_interpret
 * 
 * PURPOSE   : Decode UDP packet
 * 
 * NOTES     : 
 * 
 * ARGUMENTS : 
 * 
 * RETURNS   : p	pointer to UDP packet
 *---------------------------------------------------------------------------*/
udp_interpret(p)
UDPPKT	*p;
{
	register ushort	spknum, scksum, dcksum;
	int	dlen = p->u.length - 8;	/* data length, sub 8b UDP header */
	uchar	*pd = (uchar *)&p->data[0];
	void	(*func)();

	udpbuf.u.dest = p->u.source;		/* save sender's UDP port */
	mem_move(p->d.src, udpbuf.d.dst, 6);    /* save sender's ether addr */
        mem_move(p->i.ipsrc, udpbuf.i.ipdst, 4);/* save sender's ip addr */

	switch (*pd) {			
	case CMD_GET_STATUS:
		ne_send_udpack(0xaa);
		break;
	case CMD_LOAD:
		switch (*(pd+1)) {
		case LD_START:
			ne_send_udpack(0x5a);
			break;
		case LD_ADDR:
			ne_send_udpack(comp_cksum_b(pd, dlen));
			loadaddr = atol(pd+2);
			dpknum = 0;
			break;
		case LD_DATA:
			spknum = ((*(pd+2)<<8) | *(pd+3));
			scksum = ((*(pd+4)<<8) | *(pd+5));
			dcksum = comp_cksum_h(pd+6, dlen-6);
			if (spknum < dpknum) 
				ne_send_udpack(0xFF);
			else
			if (spknum != dpknum || scksum != dcksum)
				ne_send_udpack(0x00);
			else {
				mem_copy(pd+6, loadaddr, dlen-6);
				loadaddr  += dlen-6;
				dpknum++;
				ne_send_udpack(0xFF);
			}
			break;
		}	
	case CMD_RUN_SR:
		break;
	case CMD_EXEC:
		ne_send_udpack(0x55);
		(ulong)func = atol(pd+2);
		(*func)();
		break;
	case CMD_MOVE_TRAP:
		ne_send_udpack(0xab); 
		set_tbr(SRAM_START); 
		break;
	case CMD_REP_VER:
		*p_udpd     = 0xac;
		*(p_udpd+1) = (uchar)version;
		*(p_udpd+2) = *(uchar *)(&version + 1);
		ne_send_udpdata(3);
		break;
	}
}



ulong atol(s)
uchar	*s;
{
	return((*(s)<<24) | (*(s+1)<<16) | (*(s+2)<<8) | *(s+3));
}


mem_copy(src, dst, n)
register uchar	*src, *dst;
register int	n;
{
	while (n--)
		*dst++ = *src++;
}


comp_cksum_h(p, n)
register uchar	*p;
ulong	n;      /* number of bytes */
{
        register ushort 	cksum = 0;
        register ulong  	m = n/2;

        while (m--) {
                cksum += (*p<<8) | *(p+1);
		p += 2;
	}
 
        if (n & 0x1)            /* n is an odd number */
                cksum += (*(uchar *)p << 8);
 
        return(cksum);
}
 
 
comp_cksum_b(p, n)
uchar	*p;
int	n;
{
	uchar	cksum = 0;

	while (n--)
		cksum += *p++;
	return(cksum);
}


ne_send_udpdata(n)
int	n;
{
	ushort	datalen, i;

	if (!com_ready)
		return(1);

	datalen = n;

	udpbuf.i.tlen = IP_HLEN + UDP_HLEN + datalen;
	udpbuf.i.check = 0x0000;
	udpbuf.i.check = ipchecksum(udpbuf.i, IP_HLEN);

	udpbuf.u.length = UDP_HLEN + datalen;

	ne_send(udpbuf, ETHER_HLEN + IP_HLEN + UDP_HLEN + datalen);
#ifdef DEBUG
	printf("Sent data, size = %d\r\n", n);
#endif
}


ne_send_udpack(ack)
uchar	ack;
{
	ushort	datalen;

	if (!com_ready)
		return(1);

	datalen = 1;

	udpbuf.i.tlen = IP_HLEN + UDP_HLEN + datalen;
	udpbuf.i.check = 0x0000;
	udpbuf.i.check = ipchecksum(udpbuf.i, IP_HLEN);

	udpbuf.u.length = UDP_HLEN + datalen;

	udpbuf.data[0] = ack; 
	ne_send(udpbuf, ETHER_HLEN + IP_HLEN + UDP_HLEN + datalen);
#ifdef DEBUG
	printf("Sent ACK %02x\r\n", ack);
#endif
}


icmp_interpret(p, icmplen)
ICMPPKT	*p;
ushort	icmplen;
{
	ushort	i = p->c.type;

	switch(i) {
	case 8:				/* ping request */
		p->c.type = 0;		/* echo reply type */
		replyicmp(p, icmplen);	/* send back */
		break;
	default:
#ifdef DEBUG
		printf("Received a ICMP packet, len = %d\r\n",inpkt_len);
#endif
		break;
	}
}


replyicmp(p, icmplen)
ICMPPKT	*p;
ushort	icmplen;
{
	mem_move(p->d.src, p->d.dst, 6);	/* reverse Ether addr */
	mem_move(myether, p->d.src, 6);

	mem_move(p->i.ipsrc, p->i.ipdst, 4);	/* reverse ip addr */
	mem_move(myip, p->i.ipsrc, 4);

	p->c.check = 0;				/* ICMP checksum */
	p->c.check = ipchecksum(&p->c, icmplen);

	p->i.ident = 0;				/* IP identification */
	p->i.check = 0;				/* IP checksum */
	p->i.check = ipchecksum(&p->i, 20);

	ne_send(p, sizeof(DLAYER)+sizeof(IPLAYER)+icmplen);
#ifdef DEBUG
	printf("Sent a ICMP reply, len = %d\r\n", 
		sizeof(DLAYER)+sizeof(IPLAYER)+icmplen);
#endif
}


ushort
ipchecksum(p, len)
ushort	*p;
ushort	len;
{
	ushort	i; 
	ulong	sum = 0;

	len /= 2;			/* byte number to short number */
	for (i = 0; i < len; i++) 
		sum += *p++;
	sum = ((sum & 0xFFFF0000) >> 16) + (sum & 0xFFFF);
	sum = ((sum & 0xFFFF0000) >> 16) + (sum & 0xFFFF);
	return(~sum);
}


/* 
 * Compage two memory blocks
 * Output
 *	1 = equal
 *	0 = not equal
 */
mem_comp(s, d, n)
uchar	*s, *d;
uint	n;
{
	uint	i;

	for (i = 0; i < n; i++)
		if (*s++ != *d++)
			return(0);
	return(1);
}

mem_move(s, d, n)
uchar	*s, *d;
uint	n;
{
	uint	i;

	for (i = 0; i < n; i++)
		*d++ = *s++;
}

uchar *memcpy(dst, src, n)
register uchar	*src, *dst;
register int	n;
{
	while (n--)
		*dst++ = *src++;
	return(dst);
}

mem_zero(s, n)
uchar	*s;
uint	n;
{
	uint	i;

	for (i = 0; i < n; i++)
		*s++ = 0;
}


#ifdef DEBUG
ne_print_buf(p, len)
uchar	*p;
uint	len;
{
	uint	i;

	for (i = 0; i < len; i++) {
		printf("%02x ", *p++);
		if (!((i+1) % 16))
			printf("\r\n");
	}
	if (i % 16)
		printf("\r\n");
	printf("\r\n");
}
#endif


/*-----------------------------------------------------------------------------
 * NAME      : ne_loopback_test
 * 
 * PURPOSE   : Test NICE internal loopback
 * 
 * NOTES     : 
 * 
 * ARGUMENTS : 
 * 
 * RETURNS   : 0        OK
 *             1        Failed
 *---------------------------------------------------------------------------*/
ne_loopback_test()
{
	uint	i, len;
	ushort	sh0, sh1;

	for (i = 0, outbufp = (ushort *)netobuf; i < 40; i++)
		*outbufp++ = ((i<<1)<<8) | (i<<1)+1;
	for (i = 0, inbufp = (ushort *)netibuf; i < 40; i++)
		*inbufp++ = 0;

	/* set DLCR4 bit 1 = 0, enable loopback */
	wrasi4b(NE_DLCR4, rdasi4b(NE_DLCR4) & ~0x2);	
	NE_CLR_RXPKT();

/*
 * Transmit
 */
	wrasi4h(NE_BMR8, ne_swap(74));		/* NICE header: len = 74 */

        wrasi4h(NE_BMR8, (myether[0]<<8)|myether[1]);   /* dst Ether addr */
        wrasi4h(NE_BMR8, (myether[2]<<8)|myether[3]);
        wrasi4h(NE_BMR8, (myether[4]<<8)|myether[5]);

        wrasi4h(NE_BMR8, (myether[0]<<8)|myether[1]);   /* src Ether addr */
        wrasi4h(NE_BMR8, (myether[2]<<8)|myether[3]);
        wrasi4h(NE_BMR8, (myether[4]<<8)|myether[5]);

	wrasi4h(NE_BMR8, 0);			/* type */

	/* data */
	for (i = 0, outbufp = (ushort *)netobuf; i < 30; i++)	
		wrasi4h(NE_BMR8, *outbufp++);

	wrasi4b(NE_BMR10, 0x81);		/* start transmit */
	for (i = 100; i && !(rdasi4b(NE_DLCR0)&0x80); /* waiting for TX done */
		i--, ne_delay(1));
	if (i == 0)				/* time out */
		goto ne_lb_err;

/*
 * Receive
 */
	for (i = 100; i && !(rdasi4b(NE_DLCR1) & 0x80); /* waiting for RX */
		i--, ne_delay(1));
	if (i == 0)				/* time out */
		goto ne_lb_err;
	NE_CLR_RXPKT();				/* clear DLCR1 RX_PKT bit */
	rdasi4h(NE_BMR8);			/* filter 2 bytes */
	len = ne_swap(rdasi4h(NE_BMR8));	/* length of packet */

	for (i = 0, inbufp = (ushort *)netibuf; i < len/2; i++)
		*inbufp++ = rdasi4h(NE_BMR8);
/*
 * Compare
 */
	if (len != 74) 
		goto ne_lb_err;
#ifdef DEBUG
	for (i = 0, inbufp = (ushort *)netibuf; i < 74/2; i++) {
		printf("%04x ", *inbufp++);
		if (!((i+1) % 8))
			printf("\r\n");
	}
	printf("\r\n");
#endif
	for (i = 0, inbufp=(ushort *)(netibuf+14), outbufp=(ushort *)netobuf; 
		i < 30; i++) 
		if (*inbufp++ != *outbufp++)
			goto ne_lb_err;

#ifdef DEBUG
	printf("Loopback test passed\r\n");
#endif
	wrasi4b(NE_DLCR4, rdasi4b(NE_DLCR4) | 0x2); /* restore DLCR4 */
	return(0);

ne_lb_err:
#ifdef DEBUG
	printf("Loopback test failed\r\n");
#endif
	wrasi4b(NE_DLCR4, rdasi4b(NE_DLCR4) | 0x2); /* restore DLCR4 */
	return(1);
}


/*-----------------------------------------------------------------------------
 * NAME      : ne_check_ether
 * 
 * PURPOSE   : Check if Ethernet cable is connected
 * 
 * NOTES     : 
 * 
 * ARGUMENTS : 
 * 
 * RETURNS   : 0        Connected
 *             1        Not connected
 *---------------------------------------------------------------------------*/
ne_check_ether()
{
	uint	len, i;

	wrasi4b(NE_DLCR5, 0x0A);		/* receive short packet */
	NE_CLR_RXPKT();				/* clear DLCR1 RX_PKT bit */

	wrasi4h(NE_BMR8, ne_swap(6));		/* NICE header: len = 6 */

        wrasi4h(NE_BMR8, (myether[0]<<8)|myether[1]);   /* dst Ether addr */
        wrasi4h(NE_BMR8, (myether[2]<<8)|myether[3]);
        wrasi4h(NE_BMR8, (myether[4]<<8)|myether[5]);

	wrasi4b(NE_BMR10, 0x81);		/* start transmit */

	for (i = 100; i && !(rdasi4b(NE_DLCR1) & 0x80); /* waiting for RX */
		i--, ne_delay(1));
	if (i == 0) {				/* time out */
#ifdef DEBUG
		printf("Ethernet is not connected\r\n");
#endif
		NE_NORM_RX();
		return(1);
	}

#ifdef DEBUG
	printf("Ethernet is connected\r\n");
#endif
	rdasi4h(NE_BMR8);			/* skip 2 bytes */
	len = ne_swap(rdasi4h(NE_BMR8));

	for (i = 0; i < len/2; i++)		/* clear RX buffer */
		rdasi4h(NE_BMR8);
	NE_NORM_RX();
	return(0);
}


#ifdef DEBUG
ne_rd_dlcr()
{
printf("DLCR0 DLCR1 DLCR2 DLCR3 DLCR4 DLCR5 DLCR6 DLCR7\r\n");
printf("%02x    %02x    %02x    %02x    %02x    %02x    %02x    %02x\r\n",
	rdasi4b(NE_DLCR0),
	rdasi4b(NE_DLCR1),
	rdasi4b(NE_DLCR2),
	rdasi4b(NE_DLCR3),
	rdasi4b(NE_DLCR4),
	rdasi4b(NE_DLCR5),
	rdasi4b(NE_DLCR6),
	rdasi4b(NE_DLCR7));
}
#endif

ne_swap(sh)
ushort	sh;
{
	return(((sh & 0xFF)<<8) | ((sh & 0xFF00)>>8));
}

ne_delay(n)
uint	n;
{
	n *= 0x8000;
	while (n--);
}
