/*
 * FILE
 *	uieth.c
 *
 *	Copyright(c) 1993, FMI, Fujitsu Microelectronics, Inc.
 *	All rights reserved.
 *
 *	This software (including any documentation) is untested, has not been
 *	fully tested for viruses and has been provided to you without charge.
 *	ACCORDINGLY, IT IS DELIVERED "AS IS" WITH NO WARRANTIES EXPRESS OR
 *	IMPLIED, INCLUDING WITHOUT LIMITATION WARRANTIES OF MERCHANTABILITY,
 *	FITNESS FOR A PARTICULAR PURPOSE OR NONINFRINGEMENT.  You bear all 
 *	risk of nonperformance, loss of data and other problems and Fujitsu
 *	Microelectronics, Inc. and Fujitsu Limited will not be liable under any
 *	contract, negligence, strict liability or other theory for any damages
 *	including, without limitation, direct, consequential or incidental nor
 *	be required to provide substitute goods, services or technology.
 *
 *
 * DESCRIPTION
 * 	This program provides the user interface between the SUN4 workstation
 *	and the Fujitsu SPARClite Eval Board.  Communication is done through
 *	the Ethernet.
 *
 * HISTORY
 *	April 1993	Clark Li
 *
 */

#include	<stdio.h>
#include	<sys/types.h>
#include	<sys/socket.h>
#include	<netinet/in.h>
#include	<arpa/inet.h>
#include	<netdb.h>
#include	<sys/ioctl.h>
#include	<sys/filio.h>
#include	<sys/time.h>

#define CMD_GET_STATUS      0x00
#define CMD_LOAD            0x01
#define CMD_RUN_SR          0x02
#define CMD_EXEC            0x03
#define CMD_MOVE_TRAP       0x04
#define CMD_REP_VER         0x05

#define LD_START        0x00
#define LD_ADDR         0x01
#define LD_DATA  	0x02

#define		SERV_UDP_PORT	7000
#define		BUFMAX		1500

char			servname[] = "sleb0";
int			sockfd;
struct sockaddr_in	cli_addr, serv_addr;
int			nonblock = 1;
u_char			slebbuf[2048], inbuf[80];
u_char			pkbuf[0x10000];

char			htob();
u_short			comp_cksum_h();
u_long			slebip;

main(argc, argv)
int	argc;
char	**argv;
{
	int	trytimes;

	if (argc > 2) {
		printf("Too many arguments\n");
		exit(1);
	}

	if (argc == 2) 
		strcpy(servname, argv[1]);

	printf("SPARClite Evaluation Board Ethernet User Interface\nCopyright (c) 1993, Fujitsu Microelectronics, Inc. All rights reserved.\n\n");

	init_socket();
	for (trytimes = 5; trytimes > 0; trytimes--) {
		slebbuf[0] = CMD_REP_VER;
		if (send_data_ack(2) == 0xac)
			break;
	}

	if (trytimes <= 0) {			/* test timeout */
		printf("Cannot connect to Eval Board\n");
		exit(1);
	} else {
		printf("Ethernet connection to Eval Board established\n");
		printf("SPARClite EPROM Version %d.%d\n", 
			slebbuf[1], slebbuf[2]);
	}
		

	while (1) {
		print_cmd_list();
		gets(inbuf);
		switch(inbuf[0]) {
		case 'q': exit(0);
		case 's': getstatus();
			  break;
		case 'l': loadsr();
			  break;
		case 'a': loadaout();
			  break;
		case 'r':
		case 'b':
			  break;
		case 'e': execprog();
			  break;
		case 'h': trap2sram();
			  break;
		default:  printf("???\n");
		}
	}
}


/*
 * Output
 *	-1	Fail
 *	0	OK
 */
getstatus()
{
	slebbuf[0] = CMD_GET_STATUS;
	if (send_data_ack(2) != 0xaa) {
		printf("Cannot get SPARClite status\n");
		return(-1);
	} else {
		printf("SPARClite is ALIVE!!!\n");
		return(0);
	}
}


loadsr()
{

	FILE	*sfile, *tmpfile;
	char	filename[40];
	char	sfilebuf[256], *p;
	int	byten, rectype;
	u_long	addr;

	/* Open files */
	printf("Name of Srecord file to load: ");
	gets(filename);
	if ((sfile = fopen(filename, "r")) == NULL) {
		printf("Cannot open file '%s'\n", filename);
		return(1);
	}
	if ((tmpfile = fopen("srecord.tmp", "w+"))==NULL) {
		printf("Cannot create temporary file\n");
		fclose(sfile);
		return(1);
	}

	/* convert Srecord to binary */
	while (fgets(sfilebuf, 256, sfile) != NULL) {
		rectype = sfilebuf[1];
		if (sfilebuf[0] != 'S' || rectype < '1' || rectype > '5')
			continue;
		byten = htob(&sfilebuf[2]) - (2 + (rectype - '0'));
		p = &sfilebuf[6 + (rectype - '0')*2];

		while (byten-- > 0) {
			putc(htob(p), tmpfile);
			p += 2;
		}
	}
	fclose(sfile);

	rewind(tmpfile);

	slebbuf[0] = CMD_LOAD;
	slebbuf[1] = LD_START;
	if (send_data_ack(2) != 0x5a)  
		goto load_err;

	printf("<LOAD> Start address: ");
	gets(inbuf);
	sscanf(inbuf, "%lx", &addr);
	slebbuf[0] = CMD_LOAD;
	slebbuf[1] = LD_ADDR;
	ltoa(&slebbuf[2], addr);
	if (send_data_ack(6) < 0)	
		goto load_err;

	if (send_file(tmpfile) < 0)
		goto load_err;
	else {
		printf("Srecord file successfully loaded.\n");
		fclose(tmpfile);
		unlink("srecord.tmp");
		return(0);
	}

load_err:
	printf("Loading program failed\n");
	fclose(tmpfile);
	unlink("srecord.tmp");
}


loadaout()
{

	FILE	*afile, *tmpfile;
	char	filename[40];
	char	afilebuf[256], *p;
	int	i, byten, rectype;
	u_long	addr;

	/* Open files */
	printf("Name of a.out file to load: ");
	gets(filename);
	if ((afile = fopen(filename, "r")) == NULL) {
		printf("Cannot open file '%s'\n", filename);
		return(1);
	}
	if ((tmpfile = fopen("aoutfile.tmp", "w+"))==NULL) {
		printf("Cannot create temporary file\n");
		fclose(afile);
		return(1);
	}

	/* Create temporary binary file */
	fseek(afile, 32, 0);	/* skip a.out header */
	while ((byten = fread(afilebuf, 1, 256, afile)) > 0)
		fwrite(afilebuf, 1, byten, tmpfile);
	fclose(afile);

	rewind(tmpfile);

	slebbuf[0] = CMD_LOAD;
	slebbuf[1] = LD_START;
	if (send_data_ack(2) != 0x5a)  
		goto load_err;

	printf("<LOAD> Start address: ");
	gets(inbuf);
	sscanf(inbuf, "%lx", &addr);
	slebbuf[0] = CMD_LOAD;
	slebbuf[1] = LD_ADDR;
	ltoa(&slebbuf[2], addr);
	if (send_data_ack(6) < 0)	
		goto load_err;

	if (send_file(tmpfile) < 0)
		goto load_err;
	else {
		printf("a.out file successfully loaded.\n");
		fclose(tmpfile);
		unlink("aoutfile.tmp");
		return(0);
	}

load_err:
	printf("Loading program failed\n");
	fclose(tmpfile);
	unlink("aoutfile.tmp");
}


/*
 * Output
 *	-1	Fail
 *	0	OK
 */
execprog()
{
	u_long	addr;

	printf("<EXEC> Start address: ");
	gets(inbuf);
	sscanf(inbuf, "%lx", &addr);

	slebbuf[0] = CMD_EXEC;
	ltoa(&slebbuf[2], addr);
	if (send_data_ack(6) == 0x55) {
		printf("SPARClite reports execution started.\n");
		return(0);
	} else {
		printf("No response from Eval Board\n");
		return(-1);
	}
}


#define	MAXTRY	10

/*
 * Output
 *	-1	Fail
 *	0	OK
 */
send_file(fd)
FILE	*fd;
{
	int	trytimes, pklen;
	u_short	cksum, pknum;
	int	n, i;

	printf("Transmission started...\n");
	rewind(fd);
	pknum = 0;

	while ((pklen = fread(pkbuf, 1, 1440, fd)) > 0) {
		for (trytimes = 1; trytimes <= MAXTRY; trytimes++) {
			slebbuf[0] = CMD_LOAD;
			slebbuf[1] = LD_DATA;
			slebbuf[2] = (pknum & 0xFF00) >> 8;
			slebbuf[3] = (pknum & 0xFF);
			cksum = comp_cksum_h(pkbuf, pklen);
			slebbuf[4] = (cksum & 0xFF00) >> 8;
			slebbuf[5] = (cksum & 0xFF);
			mem_copy(pkbuf, &slebbuf[6], pklen);
			send_sleb(6+pklen);

			for (i = 0; i < trytimes*2*1000; i++)
				if (read_sleb() > 0)
					break;
			if (i > 0 && slebbuf[0] == 0xFF) 
				break;		/* successfully sent a packet*/
#ifdef DEBUG
			else
				printf("Retransmit...\n");
#endif

		}
		if (trytimes > MAXTRY) 
			return(-1);		/* transmit failed */
		else
			pknum++;		/* successfully sent a packet*/
	} /* end of while */

	return(0);
}


u_short
comp_cksum_h(p, n)
u_short	*p;
u_long	n;	/* number of bytes */
{
	u_short	cksum = 0;
	u_long	m = n/2;

	while (m--)
		cksum += *p++;

	if (n & 0x1)		/* n is an odd number */
		cksum += (*(u_char *)p << 8);

	return(cksum);
}


comp_cksum_b(p, n)
u_char	*p;
int	n;
{
	u_char	cksum = 0;

	while (n--)
		cksum += *p++;

	return(cksum);
}


/*
 * Output
 *	-1	Error
 *	num	Acknowledge
 */
send_cmd(cmd)
u_char	cmd;
{
	int	i;

	slebbuf[0] = cmd;
	send_sleb(1);
	for (i = 40; i > 0; i--) {
		delay(1);
		if (read_sleb() > 0)	
			break;
		send_sleb(1);
	}
	if (i <= 0) {
		printf("Sending command time out\n");
		return(-1);
	} else 
		return(slebbuf[0]);
}


/*
 * Output
 *	-1	Fail
 *	ack	Acknowledge from Eval Board
 */
send_data_ack(size)
u_long	size;
{
	int	i;

	send_sleb(size);
	for (i = 40; i > 0; i--) {
		delay(1);
		if (read_sleb() > 0)	
			break;
		send_sleb(size);
	}
	if (i <= 0) {
		printf("Sending data time out\n");
		return(-1);
	} else 
		return(slebbuf[0]);
}


/*
 * Output
 *	-1 	Fail
 *	0	OK
 */
recv_data()
{
	int	i;

	for (i = 40; i > 0; i--) {
		delay(1);
		if (read_sleb() > 0)	
			break;
	}
	if (i <= 0) {
		printf("Waiting for data time out\n");
		return(-1);
	} else 
		return(0);
}


char htob(s)
char	*s;
{
	int	i;
	char	*p = s;
	char	retb = 0;

	for (i = 0; i < 2; i++, p++) {
		retb <<= 4;
		if (*p >= 'a' && *p <= 'f')
			retb += *p - 'a' + 10;
		else
		if (*p >= 'A' && *p <= 'F')
			retb += *p - 'A' + 10;
		else
			retb += *p - '0';
	}
	return(retb);
}


send_sleb(n)
int	n;
{
	sendto(sockfd, slebbuf, n, 0, (struct sockaddr *)&serv_addr, 
		sizeof(serv_addr));
}


/*
 * Output
 *	-1	Error
 *	num	Actual number of bytes received
 */
read_sleb()
{
	int	servlen;	/* actual value = sizeof(*pserv_addr) */

	return(recvfrom(sockfd, slebbuf, BUFMAX, 0, 
		(struct sockaddr *)&serv_addr, &servlen));
}


print_cmd_list()
{
	printf("\nValid command list -\n");
	printf("<q> = quit to OS\n");
	printf("<s> = get evaluation board status\n");
	printf("<l> = load Srecord file\n");
	printf("<a> = load a.out file\n");	
/*	printf("<r> = run a program\n");	*/
/*	printf("<b> = run a benchmark\n");	*/
	printf("<e> = execute a program without waiting for SPARClite to finish\n");
	printf("<h> = load traphandler to SRAM\n");	
	printf("Enter a command: \n");
}


init_socket()
{
	get_slebip();

	/*
	 * Open a UDP socket (an Internet datagram socket).
	 */
	if ((sockfd = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
		err_sys("client: can't open datagram socket");

	/*
	 * Set the socket as a non-blocking process
	 */
	ioctl(sockfd, FIONBIO, &nonblock);

	/*
	 * Fill in the structure "serv_addr" with the address of the server
	 * that we want to sent to.
	 */
	bzero((char *)&serv_addr, sizeof(serv_addr));
	serv_addr.sin_family	  = AF_INET;
	serv_addr.sin_addr.s_addr = slebip;
	serv_addr.sin_port	  = htons(SERV_UDP_PORT);

	/*
	 * Bind any local address for us.
	 */
	bzero((char *)&cli_addr, sizeof(cli_addr));
	cli_addr.sin_family	 = AF_INET;
	cli_addr.sin_addr.s_addr = htonl(INADDR_ANY);
	cli_addr.sin_port	 = htons(0);
	if (bind(sockfd, (struct sockaddr *)&cli_addr, sizeof(cli_addr))<0)
		err_sys("client: can't bind local address");

}

get_slebip()
{
        FILE    *hostfd;
        char    prestr[256], curstr[256];

        if ((hostfd = fopen("/etc/hosts", "r")) == NULL) {
                printf("Cannot open '/etc/hosts'\n");
                exit(1);
        }

        while (fscanf(hostfd, "%s", curstr) != EOF)
                if (strcmp(curstr, servname) == 0) {
                        slebip = inet_addr(prestr);
                        break;
                } else
                        strcpy(prestr, curstr);

        if (strcmp(curstr, servname) != 0) {
                printf("Cannot find '%s' in /etc/hosts\n", servname);
		exit(1);
	} 

        close(hostfd);
}


trap2sram()
{
	int 	ack;

	slebbuf[0] = CMD_MOVE_TRAP;
	ack = send_data_ack(2);
        if (ack==0xab)
                printf("SPARClite reports loading started\n");
	else
	if (ack < 0)
		printf("SPARClite communication failed\n");
	else
                printf("Got wrong message from SPARClite\n");
}


unsigned short	seq = 0;

udp_cli(sockfd, pserv_addr, servlen)
int		sockfd;
struct sockaddr *pserv_addr;	/* ptr to appropriate sockaddr_XX structure */
int		servlen;	/* actual sizeof(*pserv_addr) */
{
	int	outlen, inlen, i, n;
	u_char	clibuf[1600], clibufin[1600];
	int	outservlen = servlen;
	FILE	*fd;
	struct timeval		tp;
	struct timezone		tzp;

	gettimeofday(&tp, &tzp);
	fd = fopen("log", "w");
	fprintf(fd, "Log begins\t\t%s", ctime(&tp.tv_sec));
	fclose(fd);

	for (i = 0; i < 1600; i++)
		clibuf[i] = i;

	while (1) {
		outlen = 2;
		*(ushort *)&clibuf[0] = ++seq;

		sendto(sockfd, clibuf, outlen, 0, pserv_addr, outservlen);
		printf("Sent a UDP packet, outseq = %d\n", seq);

		/*
		 * If fail to receive acknonwledge from the server,
		 * retransmit the packet 10 times; afterwards, timeout.
		 */
		for (i = 40; i > 0; i--) {
			delay(1);
			inlen = recvfrom(sockfd, clibufin, BUFMAX,0, pserv_addr,
				&servlen);
			if (inlen >= 0)				/* success */
				break;
			sendto(sockfd, clibuf, outlen, 0,	/* rexmit */ 
				pserv_addr, outservlen);
			printf("Retransmit packet %d\n", seq);
		}

		/* DEBUG */
		if (i != 10) {
			gettimeofday(&tp, &tzp);
			fd = fopen("log", "a");
			if (i > 0 && i < 10) 
				fprintf(fd, "rexmit %d times\t\t%s", 10 - i,
					ctime(&tp.tv_sec));
			else if (i == 0)
				fprintf(fd, "timeout\t\t\t%s", 
					ctime(&tp.tv_sec));
			fclose(fd);
		}

		/* timeout */
		if (i == 0) {
			printf("Sending packet %d timeout\n\n", seq);
			continue;
		}

		/* received an acknowledge */
		if (inlen > 0) {
			printf("Received an ACK for packet %d\n", clibufin[0]);
			/*
			for (i = 0; i < inlen; i++)
				printf("%02x ", clibuf[i]);
			*/
		}
		printf("\n");
		delay(1);
	}
}

mem_copy(s, d, n)
char	*s, *d;
int	n;
{
	while (n--)
		*d++ = *s++;
}


ltoa(s, l)
u_char  *s;
u_long	l;
{
	*s++ = (l & 0xFF000000) >> 24;
	*s++ = (l & 0xFF0000) >> 16;
	*s++ = (l & 0xFF00) >> 8;
	*s   = (l & 0xFF);
}


delay(n)
int	n;
{
	n *= 0x80000;
	while (n--);
}


err_sys(s)
char	*s;
{
	fprintf(stderr, "%s\n", s);
	exit(1);
}



