/*
 * FILE
 *  sun4uih.c
 *
 *  Copyright(c) 1991, FMI, Fujitsu Microelectronics, Inc.
 *  All rights reserved.
 *
 *  This software (including any documentation) is untested, has not been
 *  fully tested for viruses and has been provided to you without charge.
 *  ACCORDINGLY, IT IS DELIVERED "AS IS" WITH NO WARRANTIES EXPRESS OR
 *  IMPLIED, INCLUDING WITHOUT LIMITATION WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE OR NONINFRINGEMENT.  You bear all 
 *  risk of nonperformance, loss of data and other problems and Fujitsu
 *  Microelectronics, Inc. and Fujitsu Limited will not be liable under any
 *  contract, negligence, strict liability or other theory for any damages
 *  including, without limitation, direct, consequential or incidental nor
 *  be required to provide substitute goods, services or technology.
 *
 * DESCRIPTION
 *  This program provides the user interface between SUN4 and 
 *  the Fujitsu SPARClite Evaluation Board.  Communications is done 
 *  through the serial port.
 *
 * HISTORY
 *  Sep. 1991   Roland Chu
 *  Apr. 1993   Clark Li
 *
 * INTERNAL DESCRIPTION
 *      - Initialize serial port.
 *      - Get and perform user command.
 *      - Repeat until user enter quit command.
 */

#define SELFLOAD

/* INCLUDE */
#include <stdio.h>
#ifdef MSDOS
#include <bios.h>
#define _COM_SEND_FAST  _COM_SEND   /* send character */
#else

/* serial port services */
#define _COM_INIT       0           /* init serial port */
#define _COM_SEND       1           /* send character */
#define _COM_RECEIVE    2           /* receive character */
#define _COM_STATUS     3           /* get serial port status */
#define _COM_SEND_FAST  4           /* send character */

/* character size initializers */
#define _COM_CHR8       3           /* 8 bits characters */

/* stop bit values - on or off */
#define _COM_STOP1      0           /* 1 stop bit */

/*  parity initializers */
#define _COM_NOPARITY   0           /* no parity */

/*  baud rate initializers */
#define _COM_9600       224         /* 9600 baud */
#endif
unsigned char *collectinp();

#define SERIAL_PORT     0x00
#define GET_STATUS  0x00
#define LOAD        0x01
#define RUN_SR      0x02
#define EXEC        0x03
#define MOVE_TRAP   0x04
#define REP_VER     0x05

unsigned char   v1=0, v2=0;     /* version numbers */

/*
* FUNCTION
*   int xchar2i(char x)
*
* DESCRIPTION
*   This function converts a hex ascii character into its integer
*    equivalence and returns the value.
*
* INTERNAL DESCRIPTION
*   - Convert x.
*   - Return to caller.
*
* RETURN
*   - Integer equivalence of x.
*/
xchar2i( x) 
char x;
{
    if (x >= '0' && x <= '9')
        return(x - '0');
    else
    if (x >= 'a' && x <= 'f')
        return(x - 'a' + 10);
    else
    if (x >= 'A' && x <= 'F')
        return(x - 'A' + 10);
    else
        return(0);
}
    

/*
* FUNCTION
*   htob()
*
* DESCRIPTION
*   cvt ASCII hex byte to binary
*
* RETURN
*   binary value
*/
htob(p)
  char *p;
/* cvt ASCII hex byte to binary */
{
    return (xchar2i(*p)<<4) | xchar2i(*(p+1));
}

/*
* FUNCTION
*   sdtr_sendfile()
*
* DESCRIPTION
*   download binary file to SPARClite board.
*
* RETURN
*   - 0 = failure.
*   - 1 = success.
*/

sdtr_sendfile(file2send) FILE *file2send; {
  unsigned char checksum=0;
  long fsize, i;
  unsigned data,u;
 
  printf("Transmission started...\n");
  checksum=0;
  fseek(file2send, 0L, 2);
  fsize=ftell(file2send);
  rewind(file2send);
  /* Send size of file, MSB first */
  serial_putword(fsize);        /* Send address to EVIL */
  /* Send file */

  for (i=0; i<fsize; i++) {
    _bios_serialcom(_COM_SEND_FAST,SERIAL_PORT,data=(unsigned)fgetc(file2send));
/*    if ((i%1000)==999) {
      printf("%d\r",i+1);
      fflush(stdout);
    } */
    checksum=checksum+(unsigned char)data;
  }
  checksum&=0xff;
  /* Wait for SPARClite checksum */
  while (((data=_bios_serialcom(_COM_RECEIVE,SERIAL_PORT,data))&0x8000)!=0x0);
  if ((data&0xff)!=checksum) {
    printf("SPARClite failed and returned 0x%x instead of 0x%x on checksum...\n",data,checksum);
    return 0;
  }
  return 1;

}

/*
* FUNCTION
*       serial_putword()
*
* DESCRIPTION
*       This function sends one 32-bit word to the serial port.  The order is MSB
*        first, LSB last.
*
* INTERNAL DESCRIPTION
*       - Send MSB, 2nd MSB, 2nd LSB, LSB.
*       - Return to caller.
*
* RETURN
*       - None.
*/
serial_putword(word)
unsigned long int word;
{

  if(_bios_serialcom(_COM_SEND,SERIAL_PORT,(unsigned)(word>>24)&0xff)&0x8000) {
    printf("PC COM%d transmit error...\n",SERIAL_PORT+1);
    return(0);
  }
  if(_bios_serialcom(_COM_SEND,SERIAL_PORT,(unsigned)(word>>16)&0xff)&0x8000) {
    printf("PC COM%d transmit error...\n",SERIAL_PORT+1);
    return(0);
  }
  if(_bios_serialcom(_COM_SEND,SERIAL_PORT,(unsigned)(word>>8)&0xff)&0x8000) {
    printf("PC COM%d transmit error...\n",SERIAL_PORT+1);
    return(0);
  }
  if(_bios_serialcom(_COM_SEND,SERIAL_PORT,(unsigned)word&0xff)&0x8000) {
    printf("PC COM%d transmit error...\n",SERIAL_PORT+1);
    return(0);
  }
}


/*
* FUNCTION
*   getstatus()
*
* DESCRIPTION
*   This function checks the status of the evaluation board.
*
* INTERNAL DESCRIPTION
*   - Send a test character to the evaluation board.
*   - Receive a reply character from the evaluation board or time out.
*   - Return to caller.
*
* RETURN
*   - The reply character from the evaluation board.
*   - 0 = failure.
*/
getstatus() 
{
    unsigned    u, dummy;
    int     i, ntries=0;

    do {
            /* Send command GET_STATUS */
            if (_bios_serialcom(_COM_SEND,SERIAL_PORT,GET_STATUS)&0x8000) {
                printf("transmit error...\n",SERIAL_PORT+1);
                return(0);
            }

            /* Receive SPARClite reply byte 1 */
            for (i=0; i<10; i++) {  /* try 10 times before giving up */
                if (((u=_bios_serialcom(_COM_RECEIVE,SERIAL_PORT,u))
                &0x8000) ==0x0 && (u&0xff)==0xaa) 
                break;
            delay();
            }
            ntries++;
    } while (ntries < 10 && (u&0xff)!=0xaa); 
                /* try 10 times before giving up */

    if ((u&0x8000)!=0) {
            printf("SPARClite timed out\n");
            return(0);
    } else 
    if ((u&0xff)!=0xaa) {
            printf("SPARClite failed and returned 0x%x instead of 0xaa on get status command 1st reply...\n",u);
            dumpinp();
            return(0);
    }

    /* Send 0x55 */
    delay();
    if (_bios_serialcom(_COM_SEND,SERIAL_PORT,0x55)&0x8000) {
            printf("transmit error...\n");
            return(0);
    }

    /* Receive SPARClite reply byte 2 */
    delay();
    for (i=0; i<10; i++) {
            if (((u=_bios_serialcom(_COM_RECEIVE,SERIAL_PORT,u))&0x8000)
            ==0x0) 
            break;
            printf("status: %x\n",
            _bios_serialcom(_COM_STATUS,SERIAL_PORT,u));
    }
    if ((u&0x8000)!=0) 
            printf("timed out on error\n");
    else 
    if ((u&0xff)!=0x55) {
            printf("SPARClite failed and returned 0x%x instead of 0x55 on get status command 2nd reply...\n",u);
            dumpinp();
            return(0);
    } else 
        printf("SPARClite is ALIVE!!!...\n");

    return(u);
}


/*
* FUNCTION
*   rep_version()
*
* DESCRIPTION
*   print PROM version number
*
* INTERNAL DESCRIPTION
*       - Send a version report character to the evaluation board.
*       - Receive a reply character from the evaluation board or time out.
*   - Receive version number 2 bytes
*
* RETURN
*       0   failure
*       1   old PROM
*   2   success
*/
rep_version() 
{
    unsigned    u, dummy;
    int     i, ntries=0;

    dumpinp();

        /* Send command REP_VER */
        if (_bios_serialcom(_COM_SEND,SERIAL_PORT,REP_VER)&0x8000)
                return(0);

        /* Receive SPARClite reply byte 1 */
    for (i = 0; i < 100; i++) {
        if (!((u=_bios_serialcom(_COM_RECEIVE,SERIAL_PORT,u))&0x8000))
            break;
        delay();
    }
    if (i >= 100)
                return(0);
        else
        if ((u&0xff)!=0xac) {
                 /*
                  * Clear serial line after failure of rep_version();
                  * This is for compatability of new host interface with
                  * Version 1.4 or older PROM.  The old PROM (before v1.5)
                  * has a bug: if sdtr in PROM receives an unknown command,
                  * it runs sending status.
                  */
                  dumpinp();
 
                  /* Send a byte to SPARClite */
                  if (_bios_serialcom(_COM_SEND,SERIAL_PORT,GET_STATUS)&0x8000)
                          return(0);
 
                  /* Receive a byte from SPARClite */
                  while (((u=_bios_serialcom(_COM_RECEIVE,SERIAL_PORT,u)) &
                          0x8000)!=0x0);
                  return(1);
        }
 
    if (!recv_seri_byte(&v1)) {
        v1 = 0;
        return(0);
    }
    if (!recv_seri_byte(&v2)) {
        v2 = 0;
        return(0);
    }

    return(2);
}


/*
* FUNCTION
*   move_trap()
*
* DESCRIPTION
*   This function tells eval board to move trap table and handler
*   from EPROM to SRAM.
*
* INTERNAL DESCRIPTION
*   - Send a moving trap character to the evaluation board.
*
* RETURN
*
*/
move_trap() 
{
    unsigned    u, dummy;
    int     i;

    /* Send command MOVE_TRAP */
    if (_bios_serialcom(_COM_SEND,SERIAL_PORT,MOVE_TRAP)&0x8000) {
        printf("transmit error...\n",SERIAL_PORT+1);
        return(0);
    }

    /* Receive SPARClite reply byte */
    for (i=0; i<10; i++) {  /* try 10 times before giving up */
        if (((u=_bios_serialcom(_COM_RECEIVE,SERIAL_PORT,u))
            &0x8000) ==0x0) 
        break;
        delay();
    }

    if ((u&0x8000)!=0) 
            printf("COM port timed out on error\n");
    else 
    if ((u&0xff)==0xab) 
        printf("SPARClite reports loading started\n");
    else
    if ((u&0xff)==0xad) 
        printf("SRAM test failed. Cannot move trap table to SRAM\n");
    else
            printf("Got wrong message from SPARClite\n");
}


/*
* FUNCTION
*   loadsr()
*
* DESCRIPTION
*   This function loads an Srecord file from disk to the SPARClite
*    Evaluation Board.
*
* INTERNAL DESCRIPTION
*   - Send LOAD command.
*   - Convert S-record file to binary file
*   - Send binary file one byte at a time.
*   - Return to caller.
*
* RETURN
*   - 1 = success.  *staddr = program starting execution address
*   - 0 = failure.
*/
loadsr() 
{
    unsigned    u,stat;
    char        filename[40];
    FILE        *sfile, *tmpfile;
    int         i, j;
    long        ldaddr = -1, srecaddr;
#define SBUFSIZ 256
    char        sfile_buf[SBUFSIZ], *fstat, rectype;
    long        fsize, bytecnt, total_bytes=0;
    char        inbuf[100];

    /* Open files */
    printf("Name of Srecord file to load: ");
    gets(filename);
    if ((sfile=fopen (filename, "r"))==NULL) {
        printf("Cannot open file...\n");
        return(0);
    }

    /* Create temporary binary data file */
#if MSDOS
    if ((tmpfile=fopen("srecord.tmp", "wb+"))==NULL) {
#else
    if ((tmpfile=fopen("srecord.tmp", "w+"))==NULL) {
#endif
        printf("Cannot create temporary file...\n");
        fclose(sfile);
        return(0);
    }

    /* Create temporary binary data file */
    do { /* Find start of real data */
          fstat = fgets(sfile_buf, SBUFSIZ, sfile);
    } while (fstat && sfile_buf[1] < '1');

    rectype = sfile_buf[1];
    while (fstat && sfile_buf[0]=='S' && rectype <= '5') {
        /* num data bytes = total - num bytes for address */
        int bytecnt = htob(&sfile_buf[2]) - (2 + (rectype-'0'));
        char *p = &sfile_buf[6+((rectype-'0')*2)]; 
            /* start of data (ignore addr) */       
        while (bytecnt-- > 0) {
              putc(htob(p), tmpfile);
              p += 2;
        }
        fstat = fgets(sfile_buf, SBUFSIZ, sfile);
        rectype = sfile_buf[1];
    }
    fclose(sfile);

    rewind(tmpfile); /* Go to beginning of file */

    /* Send LOAD command */
    if((stat=_bios_serialcom(_COM_SEND,SERIAL_PORT,LOAD))&0x8000) {
        printf("transmit error (%x)\n",stat);
        return(0);
    }

    /* Receive SPARClite reply byte */
    for (i=0; i<10; i++) {
        delay();
        if (((u=_bios_serialcom(_COM_RECEIVE,SERIAL_PORT,u))&0x8000)
            ==0x0) break;
        printf("status: %x\n",
            _bios_serialcom(_COM_STATUS,SERIAL_PORT,u));
    }
    if ((u&0x8000)!=0) 
        printf("COM port timed out on error\n");
    else if ((u&0xff)!=0x5a) {
        printf("SPARClite failed and returned 0x%x instead of 0x5a on load command reply...\n",u );
        return(0);
    }

    /* Prompt user for start address */
    printf("<LOAD> Start address: ");
    gets(inbuf);
    sscanf(inbuf,"%lx",&ldaddr);
    serial_putword(ldaddr);      /* Send address to EVIL */
 
    /* Send actual file */
    if (sdtr_sendfile(tmpfile)==1) 
        printf("Srecord file successfully loaded.\n");
    fclose(sfile);
    fclose(tmpfile);
    unlink("srecord.tmp");
    return(1); /* Success */

}

/*
* FUNCTION
*       loadaout()
*
* DESCRIPTION
*       This function loads an a.out file from the host to the SPARClite
*       Evaluation Board.  
*
* INTERNAL DESCRIPTION
*       - Send LOAD command.
*       - Send file size.
*       - Send a.out file one byte at a time.
*       - Return to caller.
*
* RETURN
*       - 1 = success.
*       - 0 = failure.
*/
loadaout() {
    unsigned    u,stat;
    char        filename[40];
    FILE        *sfile, *tmpfile;
    int     i, j;
#define SBUFSIZ 256
    char        sfile_buf[SBUFSIZ], *fstat, rectype;
    long        fsize, bytecnt;
    unsigned long   address;
    char        inbuf[100];

    /* Open files */
    printf("Name of a.out file to load: ");
    gets(filename);
    if ((sfile=fopen (filename, "rb+"))==NULL) {
        printf("Cannot open file...\n");
        return(0);
    }
    if ((tmpfile=fopen("srecord.tmp", "wb+"))==NULL) {
        printf("Cannot create temporary file...\n");
        fclose(sfile);
        return(0);
    }

    /* Create temporary file */
    fseek(sfile, 32, 0); /* Skip header */
    while ((i=fgetc(sfile))!=EOF) 
        fputc(i, tmpfile);
    fclose(sfile);
    fclose(tmpfile);
    if ((tmpfile=fopen("srecord.tmp", "rb+"))==NULL) {
        printf("Cannot create temporary file...\n");
        return(0);
    }
    rewind(tmpfile); /* Go to beginning of file */

    /* Send LOAD command */
    if((stat=_bios_serialcom(_COM_SEND,SERIAL_PORT,LOAD))&0x8000) {
        printf("PC COM%d transmit error (%x)\n",SERIAL_PORT+1,stat);
        return(0);
    }

    /* Receive SPARClite reply byte */
    while (((u=_bios_serialcom(_COM_RECEIVE,SERIAL_PORT,u))&0x8000)!=0x0);
#ifdef  timeout
    delay();
    delay();
    for (i=0; i<10; i++) {
        if (((u=_bios_serialcom(_COM_RECEIVE,SERIAL_PORT,u))&0x8000)
            ==0x0) break;
        printf("COM status: %x\n",
            _bios_serialcom(_COM_STATUS,SERIAL_PORT,u));
    }
    if ((u&0x8000)!=0) printf("COM port timed out on error\n");
#endif
    if ((u&0xff)!=0x5a) {
        printf("SPARClite failed and returned 0x%x instead of 0x5a on load command reply...\n",u );
        return(0);
    }

    /* Prompt user for start address */
    printf("<LOAD> Start address: ");
    gets(inbuf);
    sscanf(inbuf,"%lx",&address);
    serial_putword(address);      /* Send address to EVIL */

    /* Send actual file */
    if (sdtr_sendfile(tmpfile)==1)
        printf("a.out file successfully loaded.\n");
    fclose(tmpfile);
    unlink("srecord.tmp");
    return(1); /* Success */
}

/*
* FUNCTION
*       runsr()
*
* DESCRIPTION
*       This function tells the SPARClite Evaluation Board to execute the
*        loaded code.
*
* INTERNAL DESCRIPTION
*       - Send RUN_SR command.
*       - Wait for start reply from SPARClite.
*       - Wait for stop reply from SPARClite.
*       - Return to caller.
*
* RETURN
*       - 0 = failure.
*       - 1 = success.
*/
runsr() {
unsigned u;
unsigned long address;
int i;
char inbuf[100],ch;

  /* Send RUN_SR command to SPARClite */
  if (_bios_serialcom(_COM_SEND,SERIAL_PORT,RUN_SR)&0x8000) {
    printf("PC COM%d transmit error...\n",SERIAL_PORT+1);
    return(0);
  }

  /* Prompt user for start address */
  printf("<RUN> Start address: ");
  gets(inbuf);
  sscanf(inbuf,"%lx",&address);
  serial_putword(address);      /* Send address to EVIL */

  /* Receive SPARClite start reply byte */
  delay();
  delay();
  for (i=0; i<10; i++) {
    if (((u=_bios_serialcom(_COM_RECEIVE,SERIAL_PORT,u))&0x8000)==0x0) break;
    printf("COM status: %x\n",_bios_serialcom(_COM_STATUS,SERIAL_PORT,u));
  }
  if ((u&0x8000)!=0) printf("COM port timed out on error\n");
  if ((u&0xff)!=0x55) {
    printf("SPARClite failed to start execution and returned 0x%x instead of 0x55...\n",u);
    return(0);
  } else printf("SPARClite reports execution started.\n");

  /* Send start byte received reply to SPARClite */
  if (_bios_serialcom(_COM_SEND,SERIAL_PORT,0xaa)&0x8000) {
    printf("PC COM%d transmit error...\n",SERIAL_PORT+1);
    return(0);
  }

  /* Receive SPARClite stop reply byte */
  delay();
  delay();
  delay();
  delay();
  delay();
  for (i=0; i<10; i++) {
    if (((u=_bios_serialcom(_COM_RECEIVE,SERIAL_PORT,u))&0x8000)==0x0) break;
    printf("COM status: %x\n",_bios_serialcom(_COM_STATUS,SERIAL_PORT,u));
  }
  if ((u&0x8000)!=0) printf("COM port timed out on error\n");
  if ((u&0xff)!=0xaa) {
    printf("SPARClite execution failed to end and returned 0x%x instead of 0xaa...\n",u);
    return(0);
  } else printf("SPARClite reports execution completed.\n");

}

/*
* FUNCTION
*       runbm()
*
* DESCRIPTION
*       This function tells the SPARClite Evaluation Board to execute the
*        loaded code.
*
* INTERNAL DESCRIPTION
*       - Send RUN_SR command.
*       - Wait for start reply from SPARClite.
*       - Wait for stop reply from SPARClite.
*       - Return to caller.
*
* RETURN
*       - 0 = failure.
*       - 1 = success.
*/
runbm() {
unsigned i, u;
unsigned long address;
char inbuf[100];

  /* Send RUN_SR command to SPARClite */
  if (_bios_serialcom(_COM_SEND,SERIAL_PORT,RUN_SR)&0x8000) {
    printf("PC COM%d transmit error...\n",SERIAL_PORT+1);
    return(0);
  }

  /* Prompt user for start address */
  printf("<RUN> Start address: ");
  gets(inbuf);
  sscanf(inbuf,"%lx",&address);
  serial_putword(address);      /* Send address to EVIL */

  /* Receive SPARClite start reply byte */
  while (((u=_bios_serialcom(_COM_RECEIVE,SERIAL_PORT,u))&0x8000)!=0x0);
#ifdef  timeout
  delay();
  delay();
  for (i=0; i<10; i++) {
    if (((u=_bios_serialcom(_COM_RECEIVE,SERIAL_PORT,u))&0x8000)==0x0) break;
    printf("COM status: %x\n",_bios_serialcom(_COM_STATUS,SERIAL_PORT,u));
  }
  if ((u&0x8000)!=0) printf("COM port timed out on error\n");
#endif
  if ((u&0xff)!=0x55) {
    printf("SPARClite failed to start execution and returned 0x%x instead of 0x55...\n",u);
    return(0);
  } else printf("SPARClite reports execution started.\n");

  while (((u=_bios_serialcom(_COM_RECEIVE,SERIAL_PORT,u))&0x8000)!=0x0);
  i=(u&0xff)<<24;
  while (((u=_bios_serialcom(_COM_RECEIVE,SERIAL_PORT,u))&0x8000)!=0x0);
  i+=(u&0xff)<<16;
  while (((u=_bios_serialcom(_COM_RECEIVE,SERIAL_PORT,u))&0x8000)!=0x0);
  i+=(u&0xff)<<8;
  while (((u=_bios_serialcom(_COM_RECEIVE,SERIAL_PORT,u))&0x8000)!=0x0);
  i+=u&0xff;
  while (((u=_bios_serialcom(_COM_RECEIVE,SERIAL_PORT,u))&0x8000)!=0x0);
  address=(u&0xff)<<24;
  while (((u=_bios_serialcom(_COM_RECEIVE,SERIAL_PORT,u))&0x8000)!=0x0);
  address+=(u&0xff)<<16;
  while (((u=_bios_serialcom(_COM_RECEIVE,SERIAL_PORT,u))&0x8000)!=0x0);
  address+=(u&0xff)<<8;
  while (((u=_bios_serialcom(_COM_RECEIVE,SERIAL_PORT,u))&0x8000)!=0x0);
  address+=u&0xff;
  i+=address;
  printf("SPARClite reports time 1 = 0x%x\n",i);
  printf("SPARClite reports time 2 = 0x%x\n",address);

  /* Send start byte received reply to SPARClite */
  if (_bios_serialcom(_COM_SEND,SERIAL_PORT,0xaa)&0x8000) {
    printf("PC COM%d transmit error...\n",SERIAL_PORT+1);
    return(0);
  }

  /* Receive SPARClite stop reply byte */
  while (((u=_bios_serialcom(_COM_RECEIVE,SERIAL_PORT,u))&0x8000)!=0x0);
#ifdef  timeout
  delay();
  delay();
  for (i=0; i<10; i++) {
    if (((u=_bios_serialcom(_COM_RECEIVE,SERIAL_PORT,u))&0x8000)==0x0) break;
    printf("COM status: %x\n",_bios_serialcom(_COM_STATUS,SERIAL_PORT,u));
  }
  if ((u&0x8000)!=0) printf("COM port timed out on error\n");
#endif
  if ((u&0xff)!=0xaa) {
    printf("SPARClite execution failed to end and returned 0x%x instead of 0xaa...\n",u);
    return(0);
  } else printf("SPARClite reports execution completed.\n");

}

/*
* FUNCTION
*   load_testcomm()
*
* DESCRIPTION
*   Load the testcomm program, explain how to run it, then quit
*
*
* RETURN
*   - 0 = failure.
*   - program exits if successful
*/
load_testcomm() {
  int i,u;
  long staddr;
  if (loadsr("testcomm.sss",&staddr)==0)
    return 0;
  printf("The testcomm program has been successfully downloaded.\n");
  sleep(2);
  /* Send RUN_SR command to SPARClite */
  if (_bios_serialcom(_COM_SEND,SERIAL_PORT,RUN_SR)&0x8000) {
    printf("PC COM%d transmit error...\n",SERIAL_PORT+1);
    return(0);
  }

  printf("Beginning execution at %lx\n",staddr);
  serial_putword(staddr);      /* Send address to EVIL */
  /* Receive SPARClite start reply byte */
  delay();
  delay();
  for (i=0; i<10; i++) {
    if (((u=_bios_serialcom(_COM_RECEIVE,SERIAL_PORT,u))&0x8000)==0x0) break;
    printf("status: %x\n",_bios_serialcom(_COM_STATUS,SERIAL_PORT,u));
  }
  if ((u&0x8000)!=0) printf("timed out on error\n");
  if ((u&0xff)!=0x55) {
    printf("SPARClite failed to start execution and returned 0x%x instead of 0x55...\n",u);
  } else printf("SPARClite reports execution started.\n");

#ifndef MSDOS
  printf("To execute, type \"tip hardwire\".  After the \"connected\"\n");
  printf("message appears, type Control-B a few times to begin execution.\n");
#endif
  exit(0);
}

/*
* FUNCTION
*   load_sr()
*
* DESCRIPTION
*   Load the MRI monitor program, begin executing, then quit
*
*
* RETURN
*   - 0 = failure.
*   - program exits if successful
*/
load_sr() {
char inbuf[100];
long staddr;

  printf("Name of file to load: ");
  gets(inbuf);
  if (loadsr(inbuf,&staddr)==0)
    return 0;
  sleep(2);
  printf("The monitor program has been successfully downloaded.\n");
}

/*
* FUNCTION
*   execute()
*
* DESCRIPTION
*   Load the MRI monitor program, begin executing, then quit
*
*
* RETURN
*   - 0 = failure.
*   - program exits if successful
*/
execute() {
unsigned u;
int i;
char inbuf[100],ch;
unsigned char *p;
unsigned  val;
long staddr;

  /* Send EXEC command to SPARClite */
  if (_bios_serialcom(_COM_SEND,SERIAL_PORT,EXEC)&0x8000) {
    printf("transmit error...\n");
    return(0);
  }

  /* Prompt user for start address */
  printf("<EXEC> Start address: ");
  gets(inbuf);
  sscanf(inbuf,"%lx",&staddr);

  printf("Beginning execution at %lx\n",staddr);
  serial_putword(staddr);      /* Send address to EVIL */
  /* Receive SPARClite start reply byte */
  delay();
  delay();
  for (i=0; i<10; i++) {
    if (((u=_bios_serialcom(_COM_RECEIVE,SERIAL_PORT,u))&0x8000)==0x0) break;
    printf("status: %x\n",_bios_serialcom(_COM_STATUS,SERIAL_PORT,u));
  }
  if ((u&0x8000)!=0) printf("timed out on error\n");
  if ((u&0xff)!=0x55) {
    printf("SPARClite failed to start execution and returned 0x%x instead of 0x55...\n",u);
    return 0;
  } else printf("SPARClite reports execution started.\n");

  sleep(2);
  return 1;
}


#ifndef MSDOS
#  include <fcntl.h>
#  include <termio.h>
#if _MCCSP
  NOTE: The ioctl macros 'TCGETA' and 'TCSETAF' are expanded incorrectly
  when using sun4 include files and an ANSI compiler!!!
#endif
    struct termio oldsetup;
    struct termio newsetup;
     unsigned long status = B9600 | CS8 | CREAD; 
     char *portdev = "/dev/ttyb";
     int comPort;

/*
* FUNCTION
*   _bios_serialcom()
*
* DESCRIPTION
*   replaces the Microsoft version of '_bios_serialcom'
*
*
* RETURN
*   - subset of Microsoft _bios_serialcom.
*/
/* this replaces the Microsoft version of '_bios_serialcom' so that */
/* this program can be ported to other hosts */
_bios_serialcom( service,  port,  data)
  unsigned service,port,data;
{
    int ret;
    unsigned char chdata;
    int cnt;
    switch (service) {
    case _COM_INIT:
      printf("Using serial port %s\n",portdev);
      comPort = open(portdev, O_RDWR | O_NDELAY);
      if (comPort == -1)
          error("Cannot open device: '%s'\n", portdev);
      printf("'%s' openned with fd = %d\n", portdev, comPort);
      ret = ioctl( comPort, TCGETA, &newsetup );/* Take old setup.. */
      if (ret < 0) {
      perror(0);
          error("Cannot ioctl-1 device: '%s'\n", portdev);
      }
      newsetup.c_iflag   = 0L;          /*  and set new values  */
      newsetup.c_oflag   = 0L;
      newsetup.c_cflag   = (long) status;
      newsetup.c_lflag     = 0L;
      newsetup.c_cc[VTIME] = 0;         /* No timeout       */
      newsetup.c_cc[VMIN]  = 0;         /* No char neccerilly.. */
      ret = ioctl( comPort, TCSETAF, &newsetup );/* Write it back.  */
      if (ret < 0) {
      perror(0);
          error("Cannot ioctl-2 device: '%s'\n", portdev);
      }
      ret = 0;
      break;

    case _COM_SEND:
      chdata = data;
      cnt = 0;
      do {
         ret = write(comPort, &chdata, 1);
      } while (++cnt < 200 && ret != 1);
      ret = ret==1 ? 0 : 0x8000;
      delay();
      break;

    case _COM_SEND_FAST:
      chdata = data;
      cnt = 0;
      do {
         ret = write(comPort, &chdata, 1);
      } while (++cnt < 200 && ret != 1);
      ret = ret==1 ? 0 : 0x8000;
      break;

    case _COM_RECEIVE:
      ret = read(comPort, &chdata, 1);
      if (ret == 1) /* success */
    ret = chdata;
      else
    ret = 0x8000; /* nothing to read */
      break;

    case _COM_STATUS:
      ret = 0;
      break;
    }
    return ret;
}



/*
* FUNCTION
*   error()
*
* DESCRIPTION
*   issue error message and quit
*
*
* RETURN
*   - none
*/
error(msg, arg)
  char *msg;
{
  printf(msg, arg);
  exit(1);
}
#endif
delay()
{
#ifdef MSDOS
   static long wait_var, one=1;
   for (wait_var=0; wait_var < 100; wait_var++) wait_var *= one;
#else /* SUN */
   usleep(500);
#endif
}

/*
* FUNCTION
*   dumpinp()
*
* DESCRIPTION
*   read any characters coming in from ser
*
*
* RETURN
*   - none
*/
dumpinp()
/* read and report characters from serial port */
{
  int ret;
  delay();
  ret = _bios_serialcom(_COM_RECEIVE,SERIAL_PORT,0);
  while ((ret&0x8000)==0) {
    printf("..received 0x%x",ret);
    if (ret >= 32 && ret <= 127) printf(" ('%c')",ret);
    printf("\n");
    delay();
    ret = _bios_serialcom(_COM_RECEIVE,SERIAL_PORT,0);
  }
}

/*
* FUNCTION
*   main()
*
* DESCRIPTION
*   prompt for command and dispatch
*
*
* RETURN
*   - none.
*/
main() {
    char        buf[100];
    unsigned int    status, u, n;
    int         codevalid=0;
    long        addr;

    /* Copyright Notice */
    printf("SPARClite Evaluation Board User Interface\n");
    printf("Copyright (c) 1993, Fujitsu Microelectronics, Inc.  All rights reserved.\n\n");

    /* Initialize COM */
    status=_bios_serialcom(_COM_INIT,SERIAL_PORT,
        _COM_CHR8|_COM_STOP1|_COM_NOPARITY|_COM_9600);
    if (status&0x8000!=0)
            printf("Not able to initialize COM%d\nBIOS returns: %x\n",SERIAL_PORT+1,status);
    printf("\n");

    /* Report SPARClite version number */
    if ((status = rep_version()) == 2)  /* new version PROM */
        printf("SPARClite EPROM Version %d.%d\n\n", v1, v2);
    else 
    if (status == 1)
        mv_trap2sram();
    else
        printf("Failed to connect to SPARClite\n");

    /* 1st prompt to user */
    printf("Valid command list -\n");
    printf("<q> = quit to OS\n");
    printf("<s> = get evaluation board status\n");
    printf("<l> = load Srecord file\n");
    printf("<a> = load a.out file\n");
    printf("<r> = run a program\n");
    printf("<b> = run a benchmark\n");
    printf("<e> = execute a program without waiting for SPARClite to finish\n");
    printf("<h> = load traphandler to SRAM\n");
    printf("Enter a command: \n");

    /* Command enter loop */
    gets(buf);
    status = *buf;
    while (status!='q') {
        switch(status) {
        case 's' : getstatus();
               break;
        case 'l' : loadsr();
               codevalid=1;
               break;
        case 'a' : loadaout();
               codevalid=1;
               break;
        case 'r' : if (codevalid) 
                runsr();
               else 
                printf("No valid code loaded...\n");
               break;
        case 'b' : if (codevalid) 
                runbm();
               else 
                printf("No valid code loaded...\n");
               break;
        case 'e' : execute();
               break;
        case 'h' : if ((n=rep_version())==2)    /* success */
                move_trap();
               else           
               if (n==1)            /* old version PROM */
                mv_trap2sram();
               else
                printf("Connecting to SPARClite failed\n");
               break;
        default  : printf("???\n");
        }

        /* User prompt */
        printf("\nValid command list -\n");
        printf("<q> = quit to OS\n");
        printf("<s> = get evaluation board status\n");
        printf("<l> = load Srecord file\n");
        printf("<a> = load a.out file\n");
        printf("<r> = run a program\n");
        printf("<b> = run a benchmark\n");
        printf("<e> = execute a program without waiting for SPARClite to finish\n");
        printf("<h> = load traphandler to SRAM\n");
        printf("Enter a command: \n");
        gets(buf);
        status = *buf;
    }

}


recv_seri_byte(c)
unsigned char   *c;
{
    unsigned int    u, i;

    for (i=0; i<10; i++) {  /* try 10 times before giving up */
        if (((u=_bios_serialcom(_COM_RECEIVE,SERIAL_PORT,u))
            & 0x8000) == 0x0) 
            break;
        delay();
    }
    if ((u & 0x8000) != 0) 
        return(0);
    else {
        *c = u & 0xFF;
        return(1);
    }
}



/*
* FUNCTION
*       mv_trap2sram()
*
* DESCRIPTION
*       This function loads the EPROM code into 0x30000000, which is SRAM.
*
* INTERNAL DESCRIPTION
*       - Send LOAD command.
*       - Send file size.
*       - Send Srecord file one byte at a time.
*       - Return to caller.
*
* RETURN
*       - 1 = success.
*       - 0 = failure.
*/
mv_trap2sram() {
  unsigned u,stat;
  char filename[40];
  FILE *sfile, *tmpfile;
  int i, j;
#define SBUFSIZ 256
  char sfile_buf[SBUFSIZ], *fstat, rectype;
  long fsize, bytecnt;
  unsigned long address;

  /* Open files */
  sscanf("tbr0.sr","%s",filename);
  if ((sfile=fopen (filename, "rb+"))==NULL) {
    printf("Cannot open file...\n");
    return(0);
  }
  /* Create temporary binary data file */
#if MSDOS
  if ((tmpfile=fopen("srecord.tmp", "wb+"))==NULL) {
#else
  if ((tmpfile=fopen("srecord.tmp", "w+"))==NULL) {
#endif
    printf("Cannot create temporary file...\n");
    fclose(sfile);
    return(0);
  }

  /* Create temporary binary data file */
  do { /* Find start of real data */
    fstat = fgets(sfile_buf, SBUFSIZ, sfile);
  } while (fstat && sfile_buf[1] < '1');

  rectype = sfile_buf[1];
  while (fstat && sfile_buf[0]=='S' && rectype <= '5') {
    /* num data bytes = total - num bytes for address */
    int bytecnt = htob(&sfile_buf[2]) - (2 + (rectype-'0'));
    char *p = &sfile_buf[6+((rectype-'0')*2)]; /* start of data (ignore addr) */    while (bytecnt-- > 0) {
      putc(htob(p), tmpfile);
      p += 2;
    }
    fstat = fgets(sfile_buf, SBUFSIZ, sfile);
    rectype = sfile_buf[1];
  }
  fclose(sfile);

  rewind(tmpfile); /* Go to beginning of file */

  /* Send LOAD command */
  if((stat=_bios_serialcom(_COM_SEND,SERIAL_PORT,LOAD))&0x8000) {
    printf("transmit error (%x)\n",stat);
    return(0);
  }

  /* Receive SPARClite reply byte */
  for (i=0; i<10; i++) {
    delay();
    if (((u=_bios_serialcom(_COM_RECEIVE,SERIAL_PORT,u))&0x8000)==0x0) break;
    printf("status: %x\n",_bios_serialcom(_COM_STATUS,SERIAL_PORT,u));
  }
  if ((u&0x8000)!=0)
    printf("COM port timed out on error\n");
  else if ((u&0xff)!=0x5a) {
    printf("SPARClite failed and returned 0x%x instead of 0x5a on load command reply...\n",u );
    return(0);
  }

  /* Set start address */
  printf("Loading tbr0 modification routine to DRAM address 0x40000000...\n");
  address=0x40000000;
  serial_putword(address);      /* Send address to EVIL */

  /* Send actual file */
  if (sdtr_sendfile(tmpfile)==1)
  fclose(sfile);
  fclose(tmpfile);
  unlink("srecord.tmp");

  /* Send RUN_SR command to SPARClite */
  if (_bios_serialcom(_COM_SEND,SERIAL_PORT,RUN_SR)&0x8000) {
    printf("PC COM%d transmit error...\n",SERIAL_PORT+1);
    return(0);
  }

  /* Set start address */
  address=0x40000000;
  serial_putword(address);      /* Send address to EVIL */

  /* Receive SPARClite start reply byte */
  delay();
  delay();
  for (i=0; i<10; i++) {
    if (((u=_bios_serialcom(_COM_RECEIVE,SERIAL_PORT,u))&0x8000)==0x0) break;
    printf("COM status: %x\n",_bios_serialcom(_COM_STATUS,SERIAL_PORT,u));
  }
  if ((u&0x8000)!=0) printf("COM port timed out on error\n");
  if ((u&0xff)!=0x55) {
    printf("SPARClite failed to start execution and returned 0x%x instead of 0x55...\n",u);
    return(0);
  } else printf("SPARClite reports tbr0 modification started...\n");

  /* Send start byte received reply to SPARClite */
  if (_bios_serialcom(_COM_SEND,SERIAL_PORT,0xaa)&0x8000) {
    printf("PC COM%d transmit error...\n",SERIAL_PORT+1);
    return(0);
  }

  /* Receive SPARClite stop reply byte */
  delay();
  delay();
  delay();
  delay();
  delay();
  for (i=0; i<10; i++) {
    if (((u=_bios_serialcom(_COM_RECEIVE,SERIAL_PORT,u))&0x8000)==0x0) break;
    printf("COM status: %x\n",_bios_serialcom(_COM_STATUS,SERIAL_PORT,u));
  }
  if ((u&0x8000)!=0) printf("COM port timed out on error\n");
  if ((u&0xff)!=0xaa) {
    printf("SPARClite execution failed to end and returned 0x%x instead of 0xaa...\n",u);
    return(0);
  } else printf("SPARClite reports tbr0 modification completed.\n");

  /* Load traphandlers to 0x30000000 */
  /* Open files */
  sscanf("prom.sr","%s",filename);
  if ((sfile=fopen (filename, "rb+"))==NULL) {
    printf("Cannot open file...\n");
    return(0);
  }
  /* Create temporary binary data file */
#if MSDOS
  if ((tmpfile=fopen("srecord.tmp", "wb+"))==NULL) {
#else
  if ((tmpfile=fopen("srecord.tmp", "w+"))==NULL) {
#endif
    printf("Cannot create temporary file...\n");
    fclose(sfile);
    return(0);
  }

  /* Create temporary binary data file */
  do { /* Find start of real data */
    fstat = fgets(sfile_buf, SBUFSIZ, sfile);
  } while (fstat && sfile_buf[1] < '1');

  rectype = sfile_buf[1];
  while (fstat && sfile_buf[0]=='S' && rectype <= '5') {
    /* num data bytes = total - num bytes for address */
    int bytecnt = htob(&sfile_buf[2]) - (2 + (rectype-'0'));
    char *p = &sfile_buf[6+((rectype-'0')*2)]; /* start of data (ignore addr) */    while (bytecnt-- > 0) {
      putc(htob(p), tmpfile);
      p += 2;
    }
    fstat = fgets(sfile_buf, SBUFSIZ, sfile);
    rectype = sfile_buf[1];
  }
  fclose(sfile);

  rewind(tmpfile); /* Go to beginning of file */

  /* Send LOAD command */
  if((stat=_bios_serialcom(_COM_SEND,SERIAL_PORT,LOAD))&0x8000) {
    printf("transmit error (%x)\n",stat);
    return(0);
  }

  /* Receive SPARClite reply byte */
  for (i=0; i<10; i++) {
    delay();
    if (((u=_bios_serialcom(_COM_RECEIVE,SERIAL_PORT,u))&0x8000)==0x0) break;
    printf("status: %x\n",_bios_serialcom(_COM_STATUS,SERIAL_PORT,u));
  }
  if ((u&0x8000)!=0)
    printf("COM port timed out on error\n");
  else if ((u&0xff)!=0x5a) {
    printf("SPARClite failed and returned 0x%x instead of 0x5a on load command reply...\n",u );
    return(0);
  }

  /* Set start address */
  printf("Loading traphandlers, prom.sr, to SRAM address 0x30000000...\n");
  address=0x30000000;
  serial_putword(address);      /* Send address to EVIL */

  /* Send actual file */
  if (sdtr_sendfile(tmpfile)==1)
    printf("Traphandler file, prom.sr, successfully loaded.\n");
  fclose(sfile);
  fclose(tmpfile);
  unlink("srecord.tmp");

  /* Load file to change tbr */
  /* Open files */
  sscanf("tbr.sr","%s",filename);
  if ((sfile=fopen (filename, "rb+"))==NULL) {
    printf("Cannot open file...\n");
    return(0);
  }
  /* Create temporary binary data file */
#if MSDOS
  if ((tmpfile=fopen("srecord.tmp", "wb+"))==NULL) {
#else
  if ((tmpfile=fopen("srecord.tmp", "w+"))==NULL) {
#endif
    printf("Cannot create temporary file...\n");
    fclose(sfile);
    return(0);
  }

  /* Create temporary binary data file */
  do { /* Find start of real data */
    fstat = fgets(sfile_buf, SBUFSIZ, sfile);
  } while (fstat && sfile_buf[1] < '1');

  rectype = sfile_buf[1];
  while (fstat && sfile_buf[0]=='S' && rectype <= '5') {
    /* num data bytes = total - num bytes for address */
    int bytecnt = htob(&sfile_buf[2]) - (2 + (rectype-'0'));
    char *p = &sfile_buf[6+((rectype-'0')*2)]; /* start of data (ignore addr) */    while (bytecnt-- > 0) {
      putc(htob(p), tmpfile);
      p += 2;
    }
    fstat = fgets(sfile_buf, SBUFSIZ, sfile);
    rectype = sfile_buf[1];
  }
  fclose(sfile);

  rewind(tmpfile); /* Go to beginning of file */

  /* Send LOAD command */
  if((stat=_bios_serialcom(_COM_SEND,SERIAL_PORT,LOAD))&0x8000) {
    printf("transmit error (%x)\n",stat);
    return(0);
  }

  /* Receive SPARClite reply byte */
  for (i=0; i<10; i++) {
    delay();
    if (((u=_bios_serialcom(_COM_RECEIVE,SERIAL_PORT,u))&0x8000)==0x0) break;
    printf("status: %x\n",_bios_serialcom(_COM_STATUS,SERIAL_PORT,u));
  }
  if ((u&0x8000)!=0)
    printf("COM port timed out on error\n");
  else if ((u&0xff)!=0x5a) {
    printf("SPARClite failed and returned 0x%x instead of 0x5a on load command reply...\n",u );
    return(0);
  }

  /* Set start address */
  printf("Loading tbr modification routine to DRAM address 0x40000000...\n");
  address=0x40000000;
  serial_putword(address);      /* Send address to EVIL */

  /* Send actual file */
  if (sdtr_sendfile(tmpfile)==1)
  fclose(sfile);
  fclose(tmpfile);
  unlink("srecord.tmp");

  /* Send RUN_SR command to SPARClite */
  if (_bios_serialcom(_COM_SEND,SERIAL_PORT,RUN_SR)&0x8000) {
    printf("PC COM%d transmit error...\n",SERIAL_PORT+1);
    return(0);
  }

  /* Set start address */
  address=0x40000000;
  serial_putword(address);      /* Send address to EVIL */

  /* Receive SPARClite start reply byte */
  delay();
  delay();
  for (i=0; i<10; i++) {
    if (((u=_bios_serialcom(_COM_RECEIVE,SERIAL_PORT,u))&0x8000)==0x0) break;
    printf("COM status: %x\n",_bios_serialcom(_COM_STATUS,SERIAL_PORT,u));
  }
  if ((u&0x8000)!=0) printf("COM port timed out on error\n");
  if ((u&0xff)!=0x55) {
    printf("SPARClite failed to start execution and returned 0x%x instead of 0x55...\n",u);
    return(0);
  } else printf("SPARClite reports tbr modification started...\n");

  /* Send start byte received reply to SPARClite */
  if (_bios_serialcom(_COM_SEND,SERIAL_PORT,0xaa)&0x8000) {
    printf("PC COM%d transmit error...\n",SERIAL_PORT+1);
    return(0);
  }

  /* Receive SPARClite stop reply byte */
  delay();
  delay();
  delay();
  delay();
  delay();
  for (i=0; i<10; i++) {
    if (((u=_bios_serialcom(_COM_RECEIVE,SERIAL_PORT,u))&0x8000)==0x0) break;
    printf("COM status: %x\n",_bios_serialcom(_COM_STATUS,SERIAL_PORT,u));
  }
  if ((u&0x8000)!=0) printf("COM port timed out on error\n");
  if ((u&0xff)!=0xaa) {
    printf("SPARClite execution failed to end and returned 0x%x instead of 0xaa...\n",u);
    return(0);
  } else printf("SPARClite reports tbr modification completed.\n");

  return(1); /* Success */

}

