/*
 * resproc.c -- pci device driver module for Responsive Processor Eva. Board
 *			made from Main.c
 *
 * $Id: resproc.c,v 1.2 2001/03/02 08:16:56 kobahide Exp $
 *********/

#ifndef __KERNEL__
#  define __KERNEL__
#endif
#ifndef MODULE
#  define MODULE
#endif

/*
 * to use resproc module with version support
 */

#include <linux/autoconf.h>
#if defined(CONFIG_MODVERSIONS) && !defined(MODVERSIONS)
#	define MODVERSIONS
#endif

#ifdef 	MODVERSIONS
#	include	<linux/modversions.h>
#endif

#include <linux/module.h>

#include <linux/ioport.h>	/* check_region() */
#include <linux/kernel.h> /* printk() */
#include <linux/malloc.h> /* kmalloc() */
#include <linux/fs.h>     /* everything... */
#include <linux/errno.h>  /* error codes */
#include <linux/types.h>  /* size_t */
#include <linux/fcntl.h>        /* O_ACCMODE */
#include <linux/pci.h>

#include <asm/system.h>   /* cli(), *_flags */
#include <asm/io.h>			/* inb, ... */
#include <asm/uaccess.h>	/* __copy_to_user() */

#include "resproc.h"        /* local definitions */


/*
 * I don't use static symbols here, because I export no symbols
 */

int resproc_major =   RESPROC_MAJOR;
int resproc_nr_devs = RESPROC_INIT_NR_DEVS; /* num. of bare resproc devices */

MODULE_PARM(resproc_major,"i");
MODULE_AUTHOR("Hidenori Kobayashi");

Resproc_Dev *resproc_devices; /* allocated in init_module */

/*
 * Interrupt Handler
 */
void resproc_intr( int irq, void *dev_id, struct pt_regs *regs)
{
	u8 byte;
	Resproc_Dev *dev = (Resproc_Dev *)dev_id;
	unsigned long ioaddr = dev->port;
	unsigned short head = dev->mail_head;
	unsigned long *mail = &dev->mail[head];

	byte = inb(ioaddr+OS_CMD1);

	/* PIS==1 then get mailbox B and clear PIS/PIC */
	if ( byte & CMD1_PIS ) {
		*mail = inl(ioaddr+OS_MB_B);
		dev->mail_head = ( head == (MAIL_NUM-1) ) ? 0 : head + 1;

		wake_up_interruptible(&dev->wq);
		outb( byte|CMD1_PIS, ioaddr+OS_CMD1 );
	}
}

/*
 * Open and close
 */

int resproc_open (struct inode *inode, struct file *filp)
{
    int num;	/* minor number */
	unsigned int irq;
	unsigned long ioaddr;
	unsigned long flags;

    Resproc_Dev *dev; /* device information */

    /* check the device number */
	num = MINOR(inode->i_rdev);

    if (num >= resproc_nr_devs) return -ENODEV;
    dev = &resproc_devices[num];

    /* and use filp->private_data to point to the device data */
    filp->private_data = dev;

	/* register irq, if opened for the first time */
	if ( !dev->usage ){
		u8 b;

		irq = dev->irq;
		ioaddr = dev->port & PCI_BASE_ADDRESS_IO_MASK;

		/* disable and clear interrupt before requesting irq */
		b = inb( ioaddr+OS_CMD1 );
		outb( b | ~CMD1_PIE | CMD1_PIC, ioaddr+OS_CMD1 );
		save_flags(flags);
		cli();	

		/* request irq */
		if( request_irq( irq, resproc_intr, SA_INTERRUPT | SA_SHIRQ, dev->name,
					dev) )
			printk(KERN_WARNING "Can't get IRQ.\n");

		/* enable pci bus interrupt */
		outb( b|CMD1_PIE, ioaddr+OS_CMD1 );

		/* and initialize buffer index */
		dev->mail_head = dev->mail_tail = 0;
		restore_flags(flags);
	}
	dev->usage++;

    MOD_INC_USE_COUNT;

    return 0;          /* success */
}

int resproc_release (struct inode *inode, struct file *filp)
{
	Resproc_Dev *dev;

	dev = filp->private_data;

	/* free irq, if closed for the last time */
	dev->usage--;
	if ( !dev->usage )
		free_irq(dev->irq, dev);

    MOD_DEC_USE_COUNT;
    return (0);
}

/*
 * Data management: read and write
 */

int send_mail( Resproc_Dev *dev, unsigned long x )
{
	int timer = TIMER_COUNT;
	unsigned long ioaddr = dev->port;

	/* wait for LIS down */
	/*
	 * FIXME: this is really stupid.
	 * "up" is need to avoid reading unsettled flag.
	 * 		may be caused by SPARClite reading the mailbox.
	 * 	Using ID1 may help.
	 */
up:
	while ( timer > 0 ){
		if ( (inb(ioaddr+OS_CMD1) & CMD1_LIS) == 0 )
			break;
		timer--;
	}

	if ( timer <= 0 )
		return -ETIMEDOUT;

	if ( (inb(ioaddr+OS_CMD1) & CMD1_LIS) != 0 )
		goto up;

	PDEBUG("send_mail: after OS_CMD1 = (%x)\n", 
			inb(ioaddr+OS_CMD1) & CMD1_LIS);

	/* write to MailBox A */
	outl( x, ioaddr+OS_MB_A );

	PDEBUG("send_mail: x = %lx\n", x);

	return 0;	/* succeed */
}

int recv_mail( Resproc_Dev *dev, unsigned long *x )
{
	unsigned short tail;
	unsigned short head;
	unsigned long flags;

	save_flags(flags);
	cli();
	head = dev->mail_head;
	tail = dev->mail_tail;
	if ( head == tail){
		interruptible_sleep_on(&dev->wq);
	}
	*x = dev->mail[tail];

	PDEBUG("recv_mail: *x = %lx\n", *x);

	dev->mail_tail = (tail == (MAIL_NUM-1)) ? 0 : tail +  1;
	restore_flags(flags);

	return 0;	/* succeed */
}

int wait_ID0_clr( Resproc_Dev *dev )
{
	int timer = TIMER_COUNT;
	unsigned long ioaddr = dev->port;

id0again:
	while( (inb(ioaddr+OS_CMD1) & CMD1_ID0) && (timer > 0) )
		timer--;

	if( timer <= 0 )
		return -1;	/* fail */
	else{
		if (inb(ioaddr+OS_CMD1) & CMD1_ID0)
			goto id0again;
		return 0;	/* succeed */
	}
}
	
int wait_ID1_clr( Resproc_Dev *dev )
{
	int timer = TIMER_COUNT;
	unsigned long ioaddr = dev->port;

	while( (inb(ioaddr+OS_CMD1) & CMD1_ID1) && (timer > 0) )
		timer--;

	if( timer <= 0 )
		return -1;	/* fail */
	else
		return 0;	/* succeed */
}
			
ssize_t resproc_read (struct file *filp, char *buf, size_t count,
                loff_t *f_pos)
{
    Resproc_Dev *dev;
	unsigned long ioaddr;
	unsigned long *data;

	int res;
	unsigned int size;
	unsigned int mailsize;
	unsigned long m;
	unsigned long maildata[3];

	dev = filp->private_data; 
	ioaddr = dev->port;
	data = (unsigned long*)dev->Rdma_data;

    if (down_interruptible (&dev->sem))
    	return (-ERESTARTSYS);

	outb( CMD2_TAIE | CMD2_MAIE, ioaddr+OS_CMD2 );

	/*
	 * interactively setup a connection
	 */
	/* possible to read data? */
	res = send_mail( dev, C_S2P );
	if ( res < 0 )
		goto timeout;

	/* wait for ack */
	recv_mail( dev, &m );

	/* tell the board where the buffer is */
	res = send_mail( dev, virt_to_bus(data));
	if ( res < 0 )
		goto timeout;

	/* determine the size ( in byte )*/
	if ( count <= PAGE_SIZE ){
		size = count;
	} else {
		size = PAGE_SIZE;
	}

	/* tell the board about the size */
	res = send_mail( dev, size );
	if ( res < 0 )
		goto timeout;

	/*
	 * small data are faster via mailbox
	 */
	if ( (mailsize = size % 0x10) > 0 ){
		int i;

#if 0
		/* fix size */
		if ( size > 0x10 )
			mailsize = size % 0x4;
#endif

		for ( i = 0; i < mailsize; i+= 4){
			recv_mail( dev, &maildata[i>>2] );
		}

		do{
			int j;
			for ( j = 0; j < mailsize; j+=4 )
				PDEBUG("resproc_read: maildata[%d] = %lx\n", j>>2, maildata[j>>2] );
		}while(0);
	}

	/* 
	 * wait for transfer completion. waiting for dma to complete
	 */
	res = wait_ID0_clr( dev );
	if ( res < 0 )
		goto timeout;

	do{
		int i;
		for( i = 0; i < (size>>2) && i < 0x40; i ++)
			PDEBUG("resproc_read: data[%d] = %lx\n",i, 
					*(volatile unsigned long*)(data+i));
	}while(0);

	/* copy kernel-space buffer to user-space buffer */
    res = copy_to_user (buf, data, size-mailsize);
	PDEBUG("to user[dma]: %dB, res = %d\n", size-mailsize, res);
	if ( mailsize > 0 )
    	res = copy_to_user (buf+size-mailsize, maildata, mailsize);
	PDEBUG("to user[mail]: %dB, res = %d\n", mailsize, res);

    up (&dev->sem);
    return size;

  timeout:
    up (&dev->sem);
    return (-ETIMEDOUT);
}

ssize_t resproc_write (struct file *filp, const char *buf, size_t count,
                loff_t *f_pos)
{
    Resproc_Dev *dev;
	unsigned long ioaddr;
	unsigned long *data;

	int res;
	unsigned long m;
	unsigned int size;
	unsigned int mailsize;

	dev = filp->private_data;
	ioaddr = dev->port;
	data = (unsigned long *)dev->Wdma_data;

	/* determine the size ( in byte )*/
	if ( count <= PAGE_SIZE ){
		size = count;
	} else {
		size = PAGE_SIZE;
	}

    if (down_interruptible (&dev->sem))
            return (-ERESTARTSYS);

	PDEBUG("resproc_write : trying to send %x\n", buf[0]);
	PDEBUG("resproc_write : size %x\n", count);

	outb( CMD2_TAIE | CMD2_MAIE, ioaddr+OS_CMD2 );

	/*
	 * interactively setup a connection
	 */
	/* possible to write data? */
	res = send_mail( dev, C_P2S );
	if ( res < 0 )
		goto timeout;

	/* wait for ack */
	recv_mail( dev, &m );

	/* copy user-space buffer to kernel-space buffer */
    copy_from_user (data, buf, size);

	/* tell the board where the buffer is */
	res = send_mail( dev, virt_to_bus(data));
	if ( res < 0 )
		goto timeout;

	/* tell the board about the size */
	res = send_mail( dev, size );
	if ( res < 0 )
		goto timeout;

	/* wait a bit to send data via mailbox (only here) */
	res = wait_ID1_clr( dev );
	if ( res < 0 )
		goto timeout;

	/*
	 * small data are faster via mailbox
	 */
	if ( (mailsize = size % 0x10) > 0 ) {
		int i;
		unsigned long *p;

#if 0
		/* fix size */
		if ( size > 0x10 )
			mailsize = size % 0x4;
#endif

		//p = data + size - mailsize;
		p = data + ((size - mailsize)>>2);

		do{
			int j;
			for ( j = 0; j < mailsize; j+=4 )
				PDEBUG("resproc_write: mail_data = %lx\n", 
						*(unsigned long *)(p+(j>>2)));
		}while(0);

		for ( i = 0; i < mailsize; i+=4 ){
			send_mail( dev, *(unsigned long *)(p+(i>>2)));
		}
	}

	/* 
	 * wait for transfer completion. waiting for dma to complete
	 */
	res = wait_ID0_clr( dev );
	if ( res < 0 )
		goto timeout;

    up (&dev->sem);
    return size;

timeout:
    up (&dev->sem);
    return (-ETIMEDOUT);
}

/*
 * The different file operations
 */

struct file_operations resproc_fops = {
    read:       resproc_read,
    write:      resproc_write,
    open:       resproc_open,
    release:    resproc_release,
};
/*
 * Finally, the module stuff
 */

int configure_resproc( struct pci_dev *rsp )
{
	unsigned long ioaddr;
	int err;
	static int ith = 0;	/* i-th found device */
	Resproc_Dev *cur_dev = &resproc_devices[ith];

	if( ith >= DEV_RESPROC_MAXNUM ) {
		printk(KERN_WARNING "too much Responsive Processor Board on PCI\n");
		return -EBUSY;
	}

	/* set name of device (pci[0-9])*/
	sprintf(cur_dev->name, "resproc%d", ith);

#if 0
	//(3)
	pci_write_config_dword(rsp, PCI_COMMAND, 
			PCI_COMMAND_SERR | PCI_COMMAND_PARITY 
			| PCI_COMMAND_MASTER | PCI_COMMAND_MEMORY | PCI_COMMAND_IO);
#else
	pci_write_config_dword(rsp, PCI_COMMAND,
			PCI_COMMAND_SERR | PCI_COMMAND_PARITY | PCI_COMMAND_IO );

	pci_set_master( rsp );
#endif

	/* check, request I/O port */
	ioaddr = rsp->base_address[0] & PCI_BASE_ADDRESS_IO_MASK;

	if ( (err = check_region(ioaddr, PCI_RESPROC_IO_RANGE)) < 0 ){
		PDEBUG("ioaddr=%lx, range = %x : in use\n",
				ioaddr, PCI_RESPROC_IO_RANGE);

		while (ith--)
			release_region(resproc_devices[ith].port, PCI_RESPROC_IO_RANGE);

		return err;
	}

	request_region(ioaddr, PCI_RESPROC_IO_RANGE, cur_dev->name);

	/* record the address of the port */
	cur_dev->port = ioaddr;

	/* get irqline ( Resposive Processor is a PnP board ) */
	cur_dev->irq = rsp->irq;

	//(5)
	pci_write_config_dword(rsp, 0x0c, 0x00004000);

	/* clear master abort status */
	pci_write_config_word( rsp, PCI_STATUS, PCI_STATUS_REC_MASTER_ABORT );

	/* allocate a page for the dma transfer buffer */
	if ( !(cur_dev->Rdma_data = get_free_page(GFP_KERNEL)) ){
		goto Rfail;
	}
	if ( !(cur_dev->Wdma_data = get_free_page(GFP_KERNEL)) ){
		goto Wfail;
	}

	PDEBUG("returning from init\n");
	ith++;
	return 0;	/* succeed */

Wfail:
	free_page(resproc_devices[ith].Rdma_data);
Rfail:
	printk(KERN_INFO "resproc.o: NO more memory for dma buffer");
	release_region(ioaddr, PCI_RESPROC_IO_RANGE);

	while(--ith > 0){
		release_region(resproc_devices[ith].port, PCI_RESPROC_IO_RANGE);
		free_page(resproc_devices[ith].Rdma_data);
		free_page(resproc_devices[ith].Wdma_data);
	}

	resproc_devices[ith].port = 0;
	return -ENOMEM;
}

int init_module(void)
{
	struct pci_dev *dev = NULL;
    int result, i;

	/* test pci device presence */
	if ( !pci_present() )
		return -ENODEV;

    /*
     * Register Resproc major, and accept a dynamic number
     */
    result = register_chrdev(resproc_major, "resproc", &resproc_fops);
    if (result < 0) {
        printk(KERN_WARNING "resproc: can't get major %d\n",resproc_major);
        return result;
    }
    if (resproc_major == 0) resproc_major = result; /* dynamic */

	/* Count the number of the Responsive Processor (PCI) Evaluation Board */
	while ( (dev = pci_find_device( PCI_VENDOR_ID_RESPROC, 
					PCI_DEVICE_ID_RESPROC, dev)) )
		resproc_nr_devs++;

	if (!resproc_nr_devs ){
		unregister_chrdev( resproc_major, "resproc");
		return -ENODEV;
	}

    /* 
     * allocate the devices -- we can't have them static, as the number
     * can be specified at load time
     */
    resproc_devices = kmalloc(resproc_nr_devs * sizeof (Resproc_Dev), 
			GFP_KERNEL);

    if (!resproc_devices) {
		unregister_chrdev(resproc_major, "resproc");
		return -ENOMEM;
    }

    memset(resproc_devices, 0, resproc_nr_devs * sizeof (Resproc_Dev));
    for (i=0; i < resproc_nr_devs; i++) {
        sema_init (&resproc_devices[i].sem, 1);
    }

	/* configure all Responsive Processor (PCI) Evaluation Board  */
	dev = NULL;
	while ( (dev = pci_find_device( PCI_VENDOR_ID_RESPROC, 
					PCI_DEVICE_ID_RESPROC, dev)) ){
		result = configure_resproc(dev);
		if ( result < 0 ){
			PDEBUG("configure_resproc returned an error\n");
			unregister_chrdev(resproc_major, "resproc");
			return result;	/* error */
		}
	}

    EXPORT_NO_SYMBOLS; 

    return 0; /* succeed */

}

void cleanup_module(void)
{
	unsigned long ioaddr;
	int i;
	
	for (i=0; i < resproc_nr_devs; i++){
		if ( resproc_devices[i].usage )
			return;

		ioaddr = resproc_devices[i].port;
		release_region(ioaddr, PCI_RESPROC_IO_RANGE);
		free_page( resproc_devices[i].Rdma_data );
		free_page( resproc_devices[i].Wdma_data );
	}

    unregister_chrdev(resproc_major, "resproc");

    kfree(resproc_devices);

    PDEBUG("module resproc cleaned\n");
}
