/*
 * resproc.c -- pci device driver module for Responsive Processor Eva. Board
 *			made from Main.c
 *
 * $Id: resproc.c,v 1.3 2003/08/21 12:38:40 kobahide Exp $
 *********/

#ifndef __KERNEL__
#  define __KERNEL__
#endif
#ifndef MODULE
#  define MODULE
#endif

/*
 * to use resproc module with version support
 */

#include <linux/autoconf.h>
#if defined(CONFIG_MODVERSIONS) && !defined(MODVERSIONS)
#	define MODVERSIONS
#endif

#ifdef 	MODVERSIONS
#	include	<linux/modversions.h>
#endif

#include <linux/module.h>

#include <linux/ioport.h>	/* check_region() */
#include <linux/kernel.h>	/* printk() */
#include <linux/slab.h>		/* kmalloc() */
#include <linux/fs.h>		/* everything... */
#include <linux/errno.h>	/* error codes */
#include <linux/types.h>	/* size_t */
#include <linux/fcntl.h>	/* O_ACCMODE */
#include <linux/pci.h>
#include <linux/ioctl.h>

#include <asm/system.h>		/* cli(), *_flags */
#include <asm/io.h>		/* inb, ... */
#include <asm/uaccess.h>	/* __copy_to_user() */

#include "resproc.h"		/* local definitions */

/*
 * I don't use static symbols here, because I export no symbols
 */

int resproc_major = RESPROC_MAJOR;
int resproc_irq = 0;
int resproc_nr_devs = RESPROC_INIT_NR_DEVS;	/* num. of bare resproc devices */
int dma_page_order = 0;

MODULE_DESCRIPTION("Responsive Processor PCI Driver ver. 4.5");
MODULE_AUTHOR("Hidenori Kobayashi");
MODULE_LICENSE("GPL");
MODULE_PARM(resproc_major, "i");
MODULE_PARM(resproc_irq, "i");
MODULE_PARM(dma_page_order, "i");
MODULE_PARM_DESC(dma_page_order, "order of pages used for read/write (=<9)");

#define	dma_page_size	(PAGE_SIZE << dma_page_order)

DECLARE_WAIT_QUEUE_HEAD(resproc_queue);

Resproc_Dev *resproc_devices;	/* allocated in init_module */

/*
 * Interrupt Handler
 */
void resproc_intr(int irq, void *dev_id, struct pt_regs *regs)
{
	u8 byte;
	Resproc_Dev *dev = (Resproc_Dev *) dev_id;
	unsigned long ioaddr = dev->port;
	unsigned short head = dev->mail_head;
	unsigned long *mail = &dev->mail[head];

	byte = inb(ioaddr + OS_CMD1);

	/* PIS==1 then get mailbox B and clear PIS/PIC */
	if (byte & CMD1_PIS) {
		*mail = inl(ioaddr + OS_MB_B);
		dev->mail_head = (head == (MAIL_NUM - 1)) ? 0 : head + 1;

		wake_up_interruptible(&resproc_queue);
		outb(byte | CMD1_PIS, ioaddr + OS_CMD1);
	}
}

/*
 * Open and close
 */

int resproc_open(struct inode *inode, struct file *filp)
{
	int num;		/* minor number */
	unsigned int irq;
	unsigned long ioaddr;
	unsigned long flags;

	Resproc_Dev *dev;	/* device information */

	/* check the device number */
	num = MINOR(inode->i_rdev);

	if (num >= resproc_nr_devs)
		return -ENODEV;
	dev = &resproc_devices[num];

	/* and use filp->private_data to point to the device data */
	filp->private_data = dev;

	/* register irq, if opened for the first time */
	if (!dev->usage) {
		u8 b;

		irq = dev->irq;
		ioaddr = dev->port & PCI_BASE_ADDRESS_IO_MASK;

		/* disable and clear interrupt before requesting irq */
		b = inb(ioaddr + OS_CMD1);
		outb(b | ~CMD1_PIE | CMD1_PIC, ioaddr + OS_CMD1);
		save_flags(flags);
		cli();

		/* request irq */
		if (request_irq(irq, resproc_intr, SA_INTERRUPT | SA_SHIRQ,
				dev->name, dev))
			printk(KERN_WARNING "Can't get IRQ.\n");

		/* enable pci bus interrupt */
		outb(b | CMD1_PIE, ioaddr + OS_CMD1);

		/* and initialize buffer index */
		dev->mail_head = dev->mail_tail = 0;
		restore_flags(flags);
	}
	dev->usage++;

	MOD_INC_USE_COUNT;

	return 0;		/* success */
}

int resproc_release(struct inode *inode, struct file *filp)
{
	Resproc_Dev *dev;

	dev = filp->private_data;

	/* free irq, if closed for the last time */
	dev->usage--;
	if (!dev->usage)
		free_irq(dev->irq, dev);

	MOD_DEC_USE_COUNT;
	return (0);
}

/*
 * Data management: read and write
 */

int send_mail(Resproc_Dev * dev, unsigned long x)
{
	int timer = TIMER_COUNT;
	unsigned long ioaddr = dev->port;

	/* wait for LIS down */
	/*
	 * FIXME: this is really stupid.
	 * "up" is need to avoid reading unsettled flag.
	 *              may be caused by SPARClite reading the mailbox.
	 *      Using ID1 may help.
	 */
      //up:
	while (timer > 0) {
#if 1
		if ((inb(ioaddr + OS_CMD1) & CMD1_LIS) == 0)
			break;
#else
		u8 tmp;
		tmp = inb(ioaddr + OS_CMD1);
		if (!(tmp & CMD1_LIS))
			break;
		rmb();
#endif
		timer--;
	}

	if (timer <= 0)
		return -ETIMEDOUT;

#if 0
	if ((inb(ioaddr + OS_CMD1) & CMD1_LIS) != 0)
		goto up;
#endif

	/* write to MailBox A */
	outl(x, ioaddr + OS_MB_A);

	PDEBUG("send_mail: x = %lx\n", x);

	return 0;		/* succeed */
}

int recv_mail(Resproc_Dev * dev, unsigned long *x)
{
	unsigned short tail;

	if (wait_event_interruptible(resproc_queue,
				     (dev->mail_head != dev->mail_tail)))
		return -ERESTARTSYS;

	tail = dev->mail_tail;
	*x = dev->mail[tail];
	barrier();
	dev->mail_tail = (tail == (MAIL_NUM - 1)) ? 0 : tail + 1;

	PDEBUG("recv_mail: *x = %lx\n", *x);
	return 0;		/* succeed */
}

/*
 * This function assumes that the transferred data fit within the page
 */
int resproc_transfer(Resproc_Dev *dev, void *dma_data, int dir, size_t count)
{
	unsigned long tmp;
	int res;
	dma_addr_t bus_addr;

	//outb( CMD2_TAIE | CMD2_MAIE, dev->port+OS_CMD2 );
	bus_addr = pci_map_single(dev->pci_dev, dma_data, PAGE_SIZE, dir);

	/* command, address, size */
	res = send_mail(dev, dir);	/* New Definition */
	if (res < 0) {
		count = res;
		goto error;
	}
	res = send_mail(dev, bus_addr);
	if (res < 0) {
		count = res;
		goto error;
	}
	res = send_mail(dev, ((count + 3) & ~0x3));
	if (res < 0) {
		count = res;
		goto error;
	}

	/* wait for ack: notification of completion */
	recv_mail(dev, &tmp);

error:
	pci_unmap_single(dev->pci_dev, bus_addr, PAGE_SIZE, dir);

	/* stabilizer (though bugus :) */
	*(volatile unsigned long *)dma_data = tmp;

	return count;
}


ssize_t resproc_read(struct file *filp, char *buf, size_t count, loff_t *f_pos)
{
	Resproc_Dev *dev;
	unsigned long *dma_data;
	int res;

	if (count <= 0)
		return 0;

	if ((dma_page_size) < count)
		count = dma_page_size;

	dev = filp->private_data;
	dma_data = (unsigned long *)dev->Rdma_data;

	if (down_interruptible(&dev->sem))
		return (-ERESTARTSYS);

	res = resproc_transfer(dev, dma_data, PCI_DMA_FROMDEVICE, count);
	if (res > 0) {
		if (copy_to_user (buf, dma_data, count)) {
			res = -EFAULT;
		}
	}

	PDEBUG("read: %08x\n", buf[0]);

	up(&dev->sem);
	return res;
}

ssize_t resproc_write(struct file *filp, const char *buf, size_t count,
		      loff_t *f_pos)
{
	Resproc_Dev *dev;
	unsigned long *dma_data;
	int res;

	if (count <= 0)
		return 0;

	if ((dma_page_size) < count)
		count = dma_page_size;

	PDEBUG("write: %08x\n", buf[0]);

	dev = filp->private_data;
	dma_data = (unsigned long *)dev->Wdma_data;

	if (down_interruptible(&dev->sem))
		return (-ERESTARTSYS);

	/* copy user-space buffer to kernel-space buffer */
	if (copy_from_user (dma_data, buf, count))
		res = -EFAULT;
	else
		res = resproc_transfer(dev, dma_data, PCI_DMA_TODEVICE, count);

	up(&dev->sem);
	return res;
}

/*
 * ioctl
 */
int resproc_rhandshake(Resproc_Dev * dev)
{
	int res;
	res = resproc_transfer(dev, &dev->Rdma_data, PCI_DMA_FROMDEVICE, 1);
	return res;
}

int resproc_shandshake(Resproc_Dev * dev)
{
	int res;
	res = resproc_transfer(dev, &dev->Wdma_data, PCI_DMA_TODEVICE, 1);
	return res;
}

unsigned long resproc_read_local(Resproc_Dev * dev, unsigned long addr)
{
	unsigned long data;

	outl(0xb, dev->port + OS_ASIR);	//SDATA
	outl(addr, dev->port + OS_L_AD);

	data = inl(dev->port + OS_LBAP);

	return data;
}

int resproc_ioctl(struct inode *inode, struct file *filp,
		  unsigned int cmd, unsigned long arg)
{
	int ret = 0, err = 0;
	Resproc_Dev *dev = filp->private_data;

	if (_IOC_TYPE(cmd) != RESPROC_IOC_MAGIC)
		return -ENOTTY;
	if (_IOC_NR(cmd) > RESPROC_IOC_MAXNR)
		return -ENOTTY;

	if (_IOC_DIR(cmd) & _IOC_READ)
		err = !access_ok(VERIFY_WRITE, (void *) arg, _IOC_SIZE(cmd));
	if (_IOC_DIR(cmd) & _IOC_WRITE)
		err = !access_ok(VERIFY_READ, (void *) arg, _IOC_SIZE(cmd));
	if (err)
		return -EFAULT;

	switch (cmd) {
	case RESPROC_BUF:
		printk(KERN_INFO "Rdma_data: %08lx, Wdma_data: %08lx\n",
		       virt_to_bus((long *) dev->Rdma_data),
		       virt_to_bus((long *) dev->Wdma_data));
		break;
	case RESPROC_RBUF:
		printk(KERN_INFO "resp: R %08lx %08lx %08lx %08lx\n",
		       *(long *) dev->Rdma_data,
		       *(long *) (dev->Rdma_data + 4),
		       *(long *) (dev->Rdma_data + 8),
		       *(long *) (dev->Rdma_data + 12));
		printk(KERN_INFO "resp: R %08lx %08lx %08lx %08lx\n",
		       *(long *) (dev->Rdma_data + 0x10),
		       *(long *) (dev->Rdma_data + 0x14),
		       *(long *) (dev->Rdma_data + 0x18),
		       *(long *) (dev->Rdma_data + 0x1c));
		break;
	case RESPROC_WBUF:
		printk(KERN_INFO "resp: W %08lx %08lx %08lx %08lx\n",
		       *(long *) dev->Wdma_data,
		       *(long *) (dev->Wdma_data + 4),
		       *(long *) (dev->Wdma_data + 8),
		       *(long *) (dev->Wdma_data + 12));
		printk(KERN_INFO "resp: W %08lx %08lx %08lx %08lx\n",
		       *(long *) (dev->Wdma_data + 0x10),
		       *(long *) (dev->Wdma_data + 0x14),
		       *(long *) (dev->Wdma_data + 0x18),
		       *(long *) (dev->Wdma_data + 0x1c));
		break;
	case RESPROC_RHANDSHAKE:
		ret = resproc_rhandshake(dev);
		break;
	case RESPROC_SHANDSHAKE:
		ret = resproc_shandshake(dev);
		break;
	case RESPROC_READ_LOCAL:
		ret = resproc_read_local(dev, arg);
		break;
	default:
		return -ENOTTY;
	}
	return ret;
}

/*
 * The different file operations
 */

struct file_operations resproc_fops = {
	read:resproc_read,
	write:resproc_write,
	open:resproc_open,
	release:resproc_release,
	ioctl:resproc_ioctl,
};
/*
 * Finally, the module stuff
 */

int configure_resproc(struct pci_dev *rsp)
{
	int err;
	static int ith = 0;	/* i-th found device */
	Resproc_Dev *cur_dev = &resproc_devices[ith];

	if (ith >= DEV_RESPROC_MAXNUM) {
		printk(KERN_WARNING
		       "too much Responsive Processor Board on PCI\n");
		return -EBUSY;
	}

	/* set name of device (pci[0-9]) */
	sprintf(cur_dev->name, "resproc%d", ith);

	err = pci_enable_device(rsp);
	if (err) {
		printk(KERN_INFO "failed to enable %s\n", cur_dev->name);
		return err;
	}

	pci_write_config_dword(rsp, PCI_COMMAND,
			       PCI_COMMAND_SERR | PCI_COMMAND_PARITY
			       | PCI_COMMAND_IO);

	pci_set_master(rsp);

	/* check, request I/O port */
	if (pci_resource_len(rsp, 0) == 0) {
		printk(KERN_WARNING "no resource\n");
		return -1;
	}
	cur_dev->port = pci_resource_start(rsp, 0);
	cur_dev->port_len = pci_resource_len(rsp, 0);
	if (!request_region(cur_dev->port,
			    pci_resource_len(rsp, 0), cur_dev->name)) {
		printk(KERN_WARNING "can not reserve i/o resource\n");
	}

	/* get irqline ( Resposive Processor is a PnP board ) */
	if (resproc_irq != 0)
		cur_dev->irq = resproc_irq;
	else
		cur_dev->irq = rsp->irq;

	pci_write_config_byte(rsp, PCI_LATENCY_TIMER, 0x40);

	/* clear master abort status */
	pci_write_config_word(rsp, PCI_STATUS, PCI_STATUS_REC_MASTER_ABORT);

	/* allocate a page for the dma transfer buffer */
	if (!(cur_dev->Rdma_data = __get_free_pages(GFP_KERNEL | GFP_DMA, dma_page_order))) {
		goto Rfail;
	}
	if (!(cur_dev->Wdma_data = __get_free_pages(GFP_KERNEL | GFP_DMA, dma_page_order))) {
		goto Wfail;
	}

	resproc_devices[ith].pci_dev = rsp;
	PDEBUG("returning from init\n");
	ith++;
	return 0;		/* succeed */

      Wfail:
	free_page(resproc_devices[ith].Rdma_data);
      Rfail:
	printk(KERN_INFO "resproc.o: NO more memory for dma buffer");
	//release_region(ioaddr, PCI_RESPROC_IO_RANGE);

	while (--ith > 0) {
		//release_region(resproc_devices[ith].port, PCI_RESPROC_IO_RANGE);
		free_pages(resproc_devices[ith].Rdma_data, dma_page_order);
		free_pages(resproc_devices[ith].Wdma_data, dma_page_order);
	}

	resproc_devices[ith].port = 0;
	return -ENOMEM;
}

int init_module(void)
{
	struct pci_dev *dev;
	int result, i;

	/* test pci device presence */
	if (!pci_present())
		return -ENODEV;

	/* Register Resproc major, and accept a dynamic number */
	result = register_chrdev(resproc_major, "resproc", &resproc_fops);
	if (result < 0) {
		printk(KERN_WARNING
		       "resproc: can't get major %d\n", resproc_major);
		return result;
	}
	if (resproc_major == 0)
		resproc_major = result;	/* dynamic */

	/* Count the number of the Responsive Processor Evaluation Board 
	 *      on PCI bus. */
	dev = NULL;
	while ((dev = pci_find_device(PCI_VENDOR_ID_RESPROC,
				      PCI_DEVICE_ID_RESPROC, dev))) {
		resproc_nr_devs++;
	}

	if (!resproc_nr_devs) {
		unregister_chrdev(resproc_major, "resproc");
		return -ENODEV;
	}

	/* 
	 * allocate the devices -- we can't have them static, as the number
	 * can be specified at load time
	 */
	resproc_devices
	    = kmalloc(resproc_nr_devs * sizeof (Resproc_Dev), GFP_KERNEL);

	if (!resproc_devices) {
		unregister_chrdev(resproc_major, "resproc");
		return -ENOMEM;
	}

	memset(resproc_devices, 0, resproc_nr_devs * sizeof (Resproc_Dev));
	for (i = 0; i < resproc_nr_devs; i++) {
		sema_init(&resproc_devices[i].sem, 1);
	}

	/* configure all Responsive Processor (PCI) Evaluation Board  */
	dev = NULL;
	for (i = 0; i < resproc_nr_devs; i++) {
		dev = pci_find_device(PCI_VENDOR_ID_RESPROC,
				      PCI_DEVICE_ID_RESPROC, dev);
		result = configure_resproc(dev);
		if (result < 0) {
			PDEBUG("configure_resproc returned error\n");
			unregister_chrdev(resproc_major, "resproc");
			return result;	/* error */
		}
	}

	EXPORT_NO_SYMBOLS;

	return 0;		/* succeed */
}

void cleanup_module(void)
{
	unsigned long ioaddr, iolen;
	int i;

	for (i = 0; i < resproc_nr_devs; i++) {
		if (resproc_devices[i].usage)
			return;

		ioaddr = resproc_devices[i].port;
		iolen = resproc_devices[i].port_len;
		release_region(ioaddr, iolen);
		free_page(resproc_devices[i].Rdma_data);
		free_page(resproc_devices[i].Wdma_data);
	}

	unregister_chrdev(resproc_major, "resproc");

	kfree(resproc_devices);

	PDEBUG("module resproc cleaned\n");
}
