/*
 * RMT RT Thread support
 */

#include <linux/config.h>
#include <linux/time.h>
#include <linux/module.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <asm/time.h>
#include <asm/rtthread.h>

#undef RTT_DEBUG

#ifdef RTT_DEBUG
#define RTT_PRINTK(...)	printk(__VA_ARGS__)
#else
#define RTT_PRINTK(...)	do{}while(0)
#endif

#define RTT_NT_QUEUE_LEN CONFIG_NR_CPUS
#define RTT_NT_QUEUE_HEAD CONFIG_NR_CPUS
#define RTT_NT_QUEUE_FREELIST (CONFIG_NR_CPUS+1)
#define RTT_NT_QUEUE_ARRAY_LEN (CONFIG_NR_CPUS+2)

struct rtt_enqueue_kick {
	int request;
	int rtt_num;
};

struct rtt_queue_1 {
	volatile task_t *rq_run;
};

struct rtt_queue_n {
	spinlock_t rqn_lock;
	volatile struct rtt_queue_elm {
		task_t *run;
		int next;
	} queue[RTT_NT_QUEUE_ARRAY_LEN];
};

struct rtthread_info {
	atomic_t user_count;
	struct timeval start;
	struct timeval end;
	struct timeval last;
	unsigned last_offset;
	unsigned period;
	unsigned sched_count;
	unsigned drop_count;
	unsigned drop_acc;
	unsigned drop_min;
	unsigned drop_max;
	unsigned error_count;
};

static struct rtthread_info rtt_info[NRTTHREADS];
static struct rtt_queue_1 rtt_rq[NRTTHREADS];
static struct rtt_queue_n rtt_nt_rq;
static struct rtt_enqueue_kick rtt_enq_kick[NR_CPUS];

static int rtthread_activate(void);


static int rttq_empty(struct rtt_queue_n *rq)
{
	return rq->queue[RTT_NT_QUEUE_HEAD].next == -1;
}

static int rttq_full(struct rtt_queue_n *rq)
{
	return rq->queue[RTT_NT_QUEUE_FREELIST].next == -1;
}

static void rttq_enqueue(struct rtt_queue_n *rq,
			 struct task_struct *tsk)
{
	int elm;

	if (rttq_full(rq))
		panic("RT-Thread queue is over flow");
	elm = rq->queue[RTT_NT_QUEUE_FREELIST].next;
	rq->queue[RTT_NT_QUEUE_FREELIST].next = rq->queue[elm].next;
	rq->queue[elm].next = rq->queue[RTT_NT_QUEUE_HEAD].next;
	rq->queue[RTT_NT_QUEUE_HEAD].next = elm;
	rq->queue[elm].run = tsk;
}

static struct task_struct *rttq_dequeue(struct rtt_queue_n *rq)
{
	int elm;
	struct task_struct *tsk;

	if (rttq_empty(rq))
		return NULL;
	
	elm = rq->queue[RTT_NT_QUEUE_HEAD].next;
	rq->queue[RTT_NT_QUEUE_HEAD].next = rq->queue[elm].next;
	rq->queue[elm].next = rq->queue[RTT_NT_QUEUE_FREELIST].next;
	rq->queue[RTT_NT_QUEUE_FREELIST].next = elm;
	tsk = rq->queue[elm].run;
	rq->queue[elm].run = NULL;
	return tsk;
}

/*******/

#define rtti_running(rtt_num)	(atomic_read(&rtt_info[rtt_num].user_count))

static int rtti_set_run(int rtt_num)
{
	int n;

	if (rtti_running(rtt_num))
		return 0;
	n = atomic_inc_return(&rtt_info[rtt_num].user_count);
	if (n > 1) {
		atomic_dec(&rtt_info[rtt_num].user_count);
		return 0;
	}
	return 1;
}

#define rtti_unset_run(rtt_num) atomic_set(&rtt_info[rtt_num].user_count,0)

static int rtti_find_and_set_run(void)
{
	int n, rval;

	for (n=0; n<NRTTHREADS; n++)
		if (rtti_set_run(n))
			return n;
	return -1;
}

/*******/

static int cmp_tv(struct timeval *a, struct timeval *b)
{
	if (a->tv_sec < b->tv_sec)
		return -1;
	if (a->tv_sec > b->tv_sec)
		return 1;
	if (a->tv_usec < b->tv_usec)
		return -1;
	if (a->tv_usec > b->tv_usec)
		return 1;
	return 0;
}

static int cmp_tv_offset(struct timeval *a, unsigned a_ns,
			 struct timeval *b, unsigned b_ns)
{
	unsigned a_s, a_us, b_s, b_us;

	a_us = a->tv_usec + a_ns / 1000;
	a_ns %= 1000;
	a_s = a->tv_sec + a_us / 1000000;
	a_us %= 1000000;

	b_us = b->tv_usec + b_ns / 1000;
	b_ns %= 1000;
	b_s = b->tv_sec + b_us / 1000000;
	b_us %= 1000000;

	if ((int)(a_s - b_s) < 0)
		return -1;
	if (a_s > b_s)
		return 1;
	if (a_us < b_us)
		return -1;
	if (a_us > b_us)
		return 1;
	if (a_ns < b_ns)
		return -1;
	if (a_ns > b_ns)
		return 1;
	return 0;
}


/* a - b => c */
static unsigned sub_tv_offset(struct timeval *a, unsigned a_ns,
			      struct timeval *b, unsigned b_ns,
			      struct timeval *c)
{
	unsigned v;
	time_t a_sec;

	a_ns += a->tv_usec * 1000;
	b_ns += b->tv_usec * 1000;
	a_sec = a->tv_sec;
	if (a_ns < b_ns) {
		a_sec--;
		a_ns += 1000000000;
	}
	v = a_ns - b_ns;
	c->tv_sec = a_sec - b->tv_sec;
	c->tv_usec = v / 1000;
	return v % 1000;
}

static unsigned sub_tv_offset_ns(struct timeval *a, unsigned a_ns,
				 unsigned ns)
{
	unsigned v;

	a_ns += a->tv_usec * 1000;

	if (a_ns <ns) {
		a->tv_sec--;
		a_ns += 1000000000;
	}
	v = a_ns - ns;
	a->tv_usec = v / 1000;
	return v % 1000;
}

static unsigned add_tv_ns(struct timeval *v, unsigned ns)
{
	v->tv_usec += ns / 1000;
	ns %= 1000;
	v->tv_sec += v->tv_usec / 1000000;
	v->tv_usec %= 1000000;
	return ns;
}

static unsigned div_tv_ns(struct timeval *v, unsigned off_ns, unsigned ns,
			  unsigned *mod)
{
	unsigned tgt, div;

	if (v->tv_sec >= 4)
		return (unsigned)-1;	/* XXXX */

	tgt = v->tv_sec * 1000000000 + v->tv_usec * 1000 + off_ns;

	div = tgt / ns;
	*mod = tgt % ns;
	return div;
}

static int calc_pass_deadline(struct timeval *now, unsigned ns_off,
			      unsigned period,
			      struct timeval *last_deadline,
			      unsigned *last_ns_off)
{
	struct timeval d;
	unsigned tgt, div, d_ns_off, mod;
	unsigned pass;

	d_ns_off = sub_tv_offset(now, ns_off, last_deadline, *last_ns_off, &d);

	pass = div_tv_ns(&d, d_ns_off, period, &mod);
	if (pass == (unsigned)-1)
		return (unsigned)-1;
	if (pass == 0) {
		RTT_PRINTK("calc_pass_deadline: no over deadline (cid=%d)\n",
			   rmt_mycnum());
		return 0;
	}
	last_deadline->tv_sec = now->tv_sec;
	last_deadline->tv_usec = now->tv_usec;
	*last_ns_off = sub_tv_offset_ns(last_deadline, ns_off, mod);
	return pass;
}

static inline unsigned nsec_to_clock_tick(unsigned nsec)
{
#define _CGETA 10000UL
#if CLOCK_TICK_RATE % _CGETA != 0
#warning "nsec_to_clock_tick() is inexact"
#endif
	unsigned tick;
	unsigned sec10u, tick_per_10us;

	sec10u = nsec / (1000000000UL / _CGETA);
	nsec = nsec % (1000000000UL / _CGETA);

	tick_per_10us = CLOCK_TICK_RATE / _CGETA;

	tick = sec10u * tick_per_10us;
	tick += nsec * tick_per_10us / (1000000000UL / _CGETA);
	return tick;
}

static void hw_timer_set_period(unsigned count)
{
	unsigned status;

	status = read_c0_status();
	status &= ~ST0_TS;		/* timer stop */
	write_c0_status(status);
	status |= (ST0_PE);
	status &= ~ST0_IM_TIMER;;	/* Unmask TimerInturrption */
	write_c0_status(status);

	write_c0_count(0);
	write_c0_compare(count);
		/* timer period is (count / CLOCK_TICK_RATE) */
}

static void hw_timer_start(void)
{
	set_c0_status(ST0_TS);		/* timer start */
}

static void hw_timer_stop(void)
{
	clear_c0_status(ST0_TS);		/* timer stop */
}

static void rtthread_stop(void)
{
	int rval;
	int rtt_num;

	rtt_num = rtt_current();

	hw_timer_stop();
	rval = rtthread_deactivate();
	if (rval != 0) {
		RTT_PRINTK("kern: rtthread_stop: failed\n");
	} else {
		rtti_unset_run(rtt_num);
	}
}

static inline void update_drop_info(struct rtthread_info *p, int drop)
{
	if (p->drop_count == 0) {
		p->drop_min = drop;
		p->drop_max = drop;
	} else {
		if (p->drop_min > drop)
			p->drop_min = drop;
		if (p->drop_max < drop)
			p->drop_max = drop;
	}
	p->drop_count++;
	p->drop_acc += drop;
}

int do_rtthread_start_periodic_thread(struct timeval *start,
				      struct timeval *end,
				      unsigned period)
{
	struct timeval now;
	int cmp, cur, pass;
	unsigned long d;

	RTT_PRINTK("kern: start_periodic_thread() %d.%06d - %d.%06d %dns cpu=%d\n",
		   start->tv_sec, start->tv_usec,
		   end->tv_sec, end->tv_usec,
		   period, smp_processor_id());

	cmp = cmp_tv(start, end);
	if (cmp >= 0) {
		RTT_PRINTK("kern: error: start >= end\n");
		return -EINVAL;
	}

	if (rtthread_activate() != 0) {
		RTT_PRINTK("kern: activate FAIL cpu=%d\n", smp_processor_id());
		return -EBUSY;
	}
	RTT_PRINTK("kern: activate SUCCESS cpu=%d\n", smp_processor_id());

	hw_timer_set_period(nsec_to_clock_tick(period));
	RTT_PRINTK("kern: set CP0_COMPARE=%d (%dns)\n", nsec_to_clock_tick(period), period);

	cur = rtt_current();
	rtt_info[cur].start = *start;
	rtt_info[cur].end = *end;
	rtt_info[cur].period = period;

	rtt_info[cur].last = *start;
	rtt_info[cur].last_offset = 0;

	do {
		do_gettimeofday(&now);
		cmp = cmp_tv(&now, end);
		if (cmp >= 0) {
			rtthread_stop();
			RTT_PRINTK("kern: stop\n");
			rtt_info[cur].error_count++;
			return -EIO;
		}

		cmp = cmp_tv(&now, start);
		/* printk("kern: start wait %d.%06d - %d.%06d cmp=%d\n",
		   now.tv_sec, now.tv_usec, start->tv_sec, start->tv_usec,
		   cmp); */

		if (cmp == 0) {
			hw_timer_start();
			RTT_PRINTK("kern: 0\n");
			return 0;
		}
	} while (cmp < 0);	/* wait start time XXXX */

	/* already pass start time */
	pass = 1;
	while (1) {
		cmp =  cmp_tv_offset(&now, 0, &rtt_info[cur].last,
				     rtt_info[cur].last_offset +
				     period);
		if (cmp < 0) {
			RTT_PRINTK("kern: %d\n", pass);
			hw_timer_start();
			if (pass > 0)
				update_drop_info(&rtt_info[cur], pass);
			return pass;
		}
		rtt_info[cur].last_offset =
			add_tv_ns(&rtt_info[cur].last, period);
		pass++;
	}
	return -EIO;	/* not reached */
}

void do_rtthread_stop_periodic_thread(void)
{
	RTT_PRINTK("kern: stop_periodic_thread()\n");

	if (rtti_running(rtt_current())) {
		rtthread_stop();
	}
}

int do_rtthread_next_period(void)
{
	struct timeval now;
	int cmp, cur, pass, drop;
	unsigned long d;

//	RTT_PRINTK("kern: next_period()\n");
	if (!(rtti_running(rtt_current())))
		return -EIO;
	
	cur = rtt_current();

#ifdef CALC_PASS_BY_INTERATION
	cmp = cmp_tv_offset(&rtt_info[cur].last,
			    rtt_info[cur].last_offset +
			    rtt_info[cur].period,
			    &rtt_info[cur].end, 0);
	if (cmp >= 0) {
		rtthread_stop();
		RTT_PRINTK("kern: stop\n");
		rtt_info[cur].error_count++;
		return -EIO;
	}
#endif

	do_gettimeofday(&now);
	cmp = cmp_tv_offset(&now, 0, &rtt_info[cur].last,
			    rtt_info[cur].last_offset +
			    rtt_info[cur].period);
#ifdef CALC_PASS_BY_INTERATION
	if (cmp < 0) {
		rmt_stopslf();
		pass = 0;
	} else
		pass = 1;

	do_gettimeofday(&now);

	rtt_info[cur].last_offset =
		add_tv_ns(&rtt_info[cur].last, rtt_info[cur].period);
	while (1) {
		cmp =  cmp_tv_offset(&now, 0,
				     &rtt_info[cur].last,
				     rtt_info[cur].last_offset +
				     rtt_info[cur].period);
		if (cmp < 0) {
			RTT_PRINTK("kern: %d\n", pass);
			if (pass > 0) 
				update_drop_info(&rtt_info[cur], pass);
			return pass;
		}
		rtt_info[cur].last_offset =
			add_tv_ns(&rtt_info[cur].last,
				  rtt_info[cur].period);
		pass++;
	}
	return -EIO;	/* not reached */
#else	/* !CALC_PASS_BY_INTERATION */
	if (cmp < 0) {
	stop_self:
		rmt_stopslf();
	}

	do_gettimeofday(&now);

	if (cmp_tv_offset(&now, 0, &rtt_info[cur].end, 0) >= 0 ||
	    cmp_tv_offset(&now, 0, &rtt_info[cur].start, 0) < 0) {
		rtthread_stop();
		RTT_PRINTK("kern: stop\n");
		rtt_info[cur].error_count++;
		return -EIO;
	}

	pass = calc_pass_deadline(&now, 0, rtt_info[cur].period,
				  &rtt_info[cur].last,
				  &rtt_info[cur].last_offset);
	if (pass == 0)
		goto stop_self;
	if (pass == (unsigned)-1) {
		RTT_PRINTK("kern: rtthread_next_period: counter over flow\n");
		rtthread_stop();
		RTT_PRINTK("kern: stop\n");
		rtt_info[cur].error_count++;
		return -EIO;
	}
	if (pass - 1 > 0)
		update_drop_info(&rtt_info[cur], pass - 1);
	return pass - 1;
#endif	/* !CALC_PASS_BY_INTERATION */
}

void rtthread_init(void)
{
	int n;

	for (n=0; n<NRTTHREADS; n++) {
		rtt_rq[n].rq_run = NULL;
		rtt_info[n].error_count = 0;
		rtt_info[n].sched_count = 0;
		rtt_info[n].drop_count = 0;
		rtt_info[n].drop_acc = 0;
		rtt_info[n].drop_min = 0;
		rtt_info[n].drop_max = 0;
	}

	rtt_nt_rq.rqn_lock = SPIN_LOCK_UNLOCKED;
	for (n=0; n<RTT_NT_QUEUE_LEN; n++) {
		rtt_nt_rq.queue[n].run = NULL;
		rtt_nt_rq.queue[n].next = n + 1;
	}
	rtt_nt_rq.queue[RTT_NT_QUEUE_LEN-1].next = -1;
	rtt_nt_rq.queue[RTT_NT_QUEUE_HEAD].next = -1;
	rtt_nt_rq.queue[RTT_NT_QUEUE_FREELIST].next = 0;

	printk("Real-time thread support cpu:%d-%d(available:%d)\n",
	       rtt_smp_processor_id(0), rtt_smp_processor_id(NRTTHREADS-1),
	       NRTTHREADS);
}

static int rtthread_activate(void)
{
	int cpu, rtt_num;

	cpu = smp_processor_id();
	if (rmt_rtthread(smp_processor_id()))
		return -1;

	if (cpu >= NNTHREADS)
		return -1;

	rtt_num = rtti_find_and_set_run();
	if (rtt_num == -1)
		return -1;

	rtt_enq_kick[cpu].request = 1;
	rtt_enq_kick[cpu].rtt_num = rtt_num;
	current->state = TASK_UNINTERRUPTIBLE;
	schedule();	/* Activate sequance:
                            <Normal Thread>
			     rtthread_activate()
			       schedule()
				 context_switch()
				 finish_task_switch()
				   rtthread_request_send()
				     rtthread_rq_enqueue()

                            <RT Thread>
			       schedule()
			         rtthread_request_recv()
				   rtthread_rq_dequeue()
                                :
				:
			      (return rtthread_activate() function)
			 */
	return 0;
}

static int rtthread_deactivate(void)
{
	int cpu;

	cpu = smp_processor_id();
	if (!rmt_rtthread(smp_processor_id()))
		return -1;

	if (cpu < NNTHREADS)
		return -1;

	rtt_enq_kick[cpu].request = 1;
	current->state = TASK_UNINTERRUPTIBLE;
	schedule();	/* Deactivate sequance:
                            <RT Thread>
			     rtthread_deactivate()
			       schedule()
				 context_switch()
				 finish_task_switch()
				   rtthread_request_send()
				     rtthread_rq_enqueue_to_normal()

                            <Normal thread>
			       schedule()
			         rtthread_request_recv()
				   rttq_dequeue(&rtt_nt_rq)
                                :
				:
			      (return rtthread_deactivate() function)
			 */
	return 0;
}

static void rtthread_rq_enqueue(int rtt_n, task_t *task)
{
	if (rtt_rq[rtt_n].rq_run)
		panic("rtthread_rq_enqueue: no rtthread resources");

	rtt_rq[rtt_n].rq_run = task;
	rmt_print_val_dec("rtthread_rq_enqueue:rmt_runth",
			  rtt_smp_processor_id(rtt_n));
	rmt_runth(rtt_smp_processor_id(rtt_n));
}

static void rtthread_rq_enqueue_to_normal(task_t *task)
{
	spin_lock(&rtt_nt_rq.rqn_lock);
	if (!rttq_full(&rtt_nt_rq)) {
		rttq_enqueue(&rtt_nt_rq, task);
		spin_unlock(&rtt_nt_rq.rqn_lock);
		rmt_print_val_dec("rtthread_rq_enqueue_normal: enqueue",
				  rmt_mycnum());
	} else {
		spin_unlock(&rtt_nt_rq.rqn_lock);
		panic("rtthread_rq_enqueue_normal: no rtthread resources");
	}
}


/* RT-Thread migration request send */
int rtthread_request_send(task_t *task)
{
	int cpu;

	cpu = smp_processor_id();
	if (!rmt_rtthread(cpu)) {	/* normal thread -> realtime thread */
		if (rtt_enq_kick[cpu].request) {
			rmt_str_printf("rtthread_request_send: to realtime thread\n");
			rtt_enq_kick[cpu].request = 0;
			rtthread_rq_enqueue(rtt_enq_kick[cpu].rtt_num, task);
		}
	} else {			/* realtime thread -> normal thread */
		if (rtt_enq_kick[cpu].request) {
			rmt_str_printf("rtthread_request_send: to normal thread\n");
			rtt_enq_kick[cpu].request = 0;
			rtthread_rq_enqueue_to_normal(task);
		}
	}
}

static task_t *rtthread_rq_dequeue(int rtt_n)
{
	task_t *task;

	task = rtt_rq[rtt_n].rq_run;
	rtt_rq[rtt_n].rq_run = NULL;
	return task;
}

static int rtthread_rq_check(int rtt_n)
{
	int rval;

	rval = (rtt_rq[rtt_n].rq_run != NULL);
	return rval;
}

/* RT-Thread migration request receive */
void rtthread_request_recv(void)
{
	int cpu;
	task_t *task;

	cpu = smp_processor_id();
	if (rmt_rtthread(cpu)) {
		task = rtthread_rq_dequeue(rtt_current());
		if (task) {
			rmt_str_printf("rtthread_request_recv: waku_up_process\n realtime");
			set_task_cpu(task, cpu);
			wake_up_process(task);
		} else {
			rtt_info[rtt_current()].sched_count++;
		}
	} else {
		spin_lock(&rtt_nt_rq.rqn_lock);
		while ((task = rttq_dequeue(&rtt_nt_rq)) != NULL) {
			spin_unlock(&rtt_nt_rq.rqn_lock);
			rmt_str_printf("rtthread_request_recv: waku_up_process\n normal");
			set_task_cpu(task, cpu);
			wake_up_process(task);
			spin_lock(&rtt_nt_rq.rqn_lock);
		}
		spin_unlock(&rtt_nt_rq.rqn_lock);
	}
}

int rtthread_request_check(void)
{
	int cpu, rval;
	task_t *task;

	cpu = smp_processor_id();
	if (rmt_rtthread(cpu)) {
		return rtthread_rq_check(rtt_current());
	} else {
		spin_lock(&rtt_nt_rq.rqn_lock);
		rval = !rttq_empty(&rtt_nt_rq);
		spin_unlock(&rtt_nt_rq.rqn_lock);
		return rval;
	}
}

#ifdef CONFIG_PROC_FS
static void *rtt_seq_start(struct seq_file *seq, loff_t *pos)
{
	if (*pos >= NRTTHREADS)
		return NULL;
	return &rtt_info[*pos];
}

static void *rtt_seq_next(struct seq_file *seq, void *v, loff_t *pos)
{
	if (++*pos >= NRTTHREADS)
		return NULL;
	return &rtt_info[*pos];
}

static void rtt_seq_stop(struct seq_file *seq, void *v)
{
}

static int rtt_seq_show(struct seq_file *seq, void *v)
{
	const struct rtthread_info *p = v;
	int rtt_num, cpu;

	rtt_num = p - &rtt_info[0];
	cpu = rtt_smp_processor_id(rtt_num);

	seq_printf(seq, "%d cpu:%d a:%d start:%d.%06d end:%d.%06d period_ns:%d cur:%d.%06d cur_ns:%d drop:%d,acc:%d,min:%d,max:%d err:%d\n",
		   rtt_num, cpu, atomic_read(&p->user_count),
		   p->start.tv_sec, p->start.tv_usec, 
		   p->end.tv_sec, p->end.tv_usec, 
		   p->period,
		   p->last.tv_sec, p->last.tv_usec,
		   p->last_offset, p->drop_count, p->drop_acc,
		   p->drop_min, p->drop_max,
		   p->error_count);
	return 0;
}


static struct seq_operations rtt_seq_ops = {
	.start = rtt_seq_start,
	.next  = rtt_seq_next,
	.stop  = rtt_seq_stop,
	.show  = rtt_seq_show,
};

static int rtt_seq_open(struct inode *inode, struct file *file)
{
	return seq_open(file, &rtt_seq_ops);
}

static struct file_operations rtt_proc_fops = {
	.owner	 = THIS_MODULE,
	.open    = rtt_seq_open,
	.read    = seq_read,
	.llseek  = seq_lseek,
	.release = seq_release,
};
#endif /* CONFIG_PROC_FS */

static int rtt_init2_module(void)
{

#ifdef CONFIG_PROC_FS
	struct proc_dir_entry *ent;

	ent = create_proc_entry("rttinfo", 0, NULL);
	if (ent)
		ent->proc_fops = &rtt_proc_fops;
#endif /* CONFIG_PROC_FS */
}

module_init(rtt_init2_module);
