/*
 * Handle unaligned accesses by emulation.
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file "COPYING" in the main directory of this archive
 * for more details.
 *
 * Copyright (C) 1996, 1998, 1999, 2002 by Ralf Baechle
 * Copyright (C) 1999 Silicon Graphics, Inc.
 *
 * This file contains exception handler for address error exception with the
 * special capability to execute faulting instructions in software.  The
 * handler does not try to handle the case when the program counter points
 * to an address not aligned to a word boundary.
 *
 * Putting data to unaligned addresses is a bad practice even on Intel where
 * only the performance is affected.  Much worse is that such code is non-
 * portable.  Due to several programs that die on MIPS due to alignment
 * problems I decided to implement this handler anyway though I originally
 * didn't intend to do this at all for user code.
 *
 * For now I enable fixing of address errors by default to make life easier.
 * I however intend to disable this somewhen in the future when the alignment
 * problems with user programs have been fixed.  For programmers this is the
 * right way to go.
 *
 * Fixing address errors is a per process option.  The option is inherited
 * across fork(2) and execve(2) calls.  If you really want to use the
 * option in your user programs - I discourage the use of the software
 * emulation strongly - use the following code in your userland stuff:
 *
 * #include <sys/sysmips.h>
 *
 * ...
 * sysmips(MIPS_FIXADE, x);
 * ...
 *
 * The argument x is 0 for disabling software emulation, enabled otherwise.
 *
 * Below a little program to play around with this feature.
 *
 * #include <stdio.h>
 * #include <sys/sysmips.h>
 *
 * struct foo {
 *         unsigned char bar[8];
 * };
 *
 * main(int argc, char *argv[])
 * {
 *         struct foo x = {0, 1, 2, 3, 4, 5, 6, 7};
 *         unsigned int *p = (unsigned int *) (x.bar + 3);
 *         int i;
 *
 *         if (argc > 1)
 *                 sysmips(MIPS_FIXADE, atoi(argv[1]));
 *
 *         printf("*p = %08lx\n", *p);
 *
 *         *p = 0xdeadface;
 *
 *         for(i = 0; i <= 7; i++)
 *         printf("%02x ", x.bar[i]);
 *         printf("\n");
 * }
 *
 * Coprocessor loads are not supported; I think this case is unimportant
 * in the practice.
 *
 * TODO: Handle ndc (attempted store to doubleword in uncached memory)
 *       exception for the R6000.
 *       A store crossing a page boundary might be executed only partially.
 *       Undo the partial store in this case.
 */
#include <linux/config.h>
#include <linux/mm.h>
#include <linux/module.h>
#include <linux/signal.h>
#include <linux/smp.h>
#include <linux/smp_lock.h>

#include <asm/asm.h>
#include <asm/branch.h>
#include <asm/byteorder.h>
#include <asm/inst.h>
#include <asm/uaccess.h>
#include <asm/system.h>

#define STR(x)  __STR(x)
#define __STR(x)  #x

#ifdef CONFIG_PROC_FS
unsigned long unaligned_instructions;
#endif

#ifdef CONFIG_RMT_TLBFLUSH_UACCESS        
extern atomic_t flag_wptlb_exist;
#endif

static  unsigned long get_phys_page (unsigned long addr,
					   struct mm_struct *mm)
{
	pgd_t *pgd;
	pmd_t *pmd;
	pte_t *pte;
	unsigned long physpage;

	pgd = pgd_offset(mm, addr);
	pmd = pmd_offset(pgd, addr);
	pte = pte_offset(pmd, addr);

	if ((physpage = pte_val(*pte)) & _PAGE_VALID)
		return KSEG0ADDR(physpage & PAGE_MASK);

	return 0;
}

static inline int emulate_load_store_insn(struct pt_regs *regs,
	void *addr, unsigned long pc,
	unsigned long **regptr, unsigned long *newvalue)
{
	union mips_instruction insn;
	unsigned long value;
	unsigned int res;
#ifdef CONFIG_RMTBUG_LWLLWR
	unsigned int tmp1, tmp2;
#endif

	regs->regs[0] = 0;
	*regptr=NULL;

#ifdef CONFIG_RMT_TLBFLUSH_UACCESS
        if (atomic_read(&flag_wptlb_exist)) {
                local_flush_dtlb_all();
        }
#endif
#ifdef CONFIG_RMT_TESTPAGE_UACCESS
	if (addr < __UA_LIMIT) {
		if (pretest_page_waccess(addr)) {
			printk("addr = %08x\n", addr);
			panic(__FILE__":%d", __LINE__); 
		}
	}
#endif

	/*
	 * This load never faults.
	 */
	__get_user(insn.word, (unsigned int *)pc);

	switch (insn.i_format.opcode) {
	/*
	 * These are instructions that a compiler doesn't generate.  We
	 * can assume therefore that the code is MIPS-aware and
	 * really buggy.  Emulating these instructions would break the
	 * semantics anyway.
	 */
	case ll_op:
	case lld_op:
	case sc_op:
	case scd_op:

	/*
	 * For these instructions the only way to create an address
	 * error is an attempted access to kernel/supervisor address
	 * space.
	 */
	case ldl_op:
	case ldr_op:
	case lwl_op:
	case lwr_op:
	case sdl_op:
	case sdr_op:
	case swl_op:
	case swr_op:
	case lb_op:
	case lbu_op:
	case sb_op:
		goto sigbus;

	/*
	 * The remaining opcodes are the ones that are really of interest.
	 */
	case lh_op:
		if (verify_area(VERIFY_READ, addr, 2))
			goto sigbus;

		__asm__ __volatile__ (".set\tnoat\n"
#ifdef __BIG_ENDIAN
			"1:\tlb\t%0, 0(%2)\n"
			"2:\tlbu\t$1, 1(%2)\n\t"
#endif
#ifdef __LITTLE_ENDIAN
			"1:\tlb\t%0, 1(%2)\n"
			"2:\tlbu\t$1, 0(%2)\n\t"
#endif
			"sll\t%0, 0x8\n\t"
			"or\t%0, $1\n\t"
			"li\t%1, 0\n"
			"3:\t.set\tat\n\t"
			".section\t.fixup,\"ax\"\n\t"
			"4:\tli\t%1, %3\n\t"
			"j\t3b\n\t"
			".previous\n\t"
			".section\t__ex_table,\"a\"\n\t"
			STR(PTR)"\t1b, 4b\n\t"
			STR(PTR)"\t2b, 4b\n\t"
			".previous"
			: "=&r" (value), "=r" (res)
			: "r" (addr), "i" (-EFAULT));
		if (res)
			goto fault;
		*newvalue = value;
		*regptr = &regs->regs[insn.i_format.rt];
		break;

	case lw_op:
		if (verify_area(VERIFY_READ, addr, 4))
			goto sigbus;
			
#ifdef CONFIG_RMTBUG_LWLLWR
		__asm__ __volatile__ (
			"1:\n\t"
			"addiu	%3, %4, 0x0\n\t"
			"ori	%2, %3, 0x3\n\t"
			"xori	%2, %2, 0x3\n\t"
			"lw	%0, 0x0(%2)\n\t"
			"andi	%2, %3, 0x3\n\t"
			"sll	%2, %2, 0x3\n\t"
			"sllv	%0, %0, %2\n"
			"2:\n\t"
			"addiu	%3, %4, 0x3\n\t"
			"ori	%2, %3, 0x3\n\t"
			"xori	%2, %2, 0x3\n\t"
			"lw	%2, 0x0(%2)\n\t"
			"andi	%3, %3, 0x3\n\t"
			"xori	%3, %3, 0x3\n\t"
			"sll	%3, %3, 0x3\n\t"
			"srlv	%2, %2, %3\n\t"
			"or	%0, %0, %2\n\t"
			"li\t%1, 0\n"
			"3:\t.section\t.fixup,\"ax\"\n\t"
			"4:\tli\t%1, %5\n\t"
			"j\t3b\n\t"
			".previous\n\t"
			".section\t__ex_table,\"a\"\n\t"
			STR(PTR)"\t1b, 4b\n\t"
			STR(PTR)"\t2b, 4b\n\t"
			".previous"
			: "=&r" (value), "=r" (res),
			  "=&r" ((unsigned int)tmp1),
			  "=&r" ((unsigned int)tmp2)
			: "r" (addr), "i" (-EFAULT));
#else /* CONFIG_RMTBUG_LWLLWR */
		__asm__ __volatile__ (
#ifdef __BIG_ENDIAN
			"1:\tlwl\t%0, (%2)\n"
			"2:\tlwr\t%0, 3(%2)\n\t"
#endif
#ifdef __LITTLE_ENDIAN
			"1:\tlwl\t%0, 3(%2)\n"
			"2:\tlwr\t%0, (%2)\n\t"
#endif
			"li\t%1, 0\n"
			"3:\t.section\t.fixup,\"ax\"\n\t"
			"4:\tli\t%1, %3\n\t"
			"j\t3b\n\t"
			".previous\n\t"
			".section\t__ex_table,\"a\"\n\t"
			STR(PTR)"\t1b, 4b\n\t"
			STR(PTR)"\t2b, 4b\n\t"
			".previous"
			: "=&r" (value), "=r" (res)
			: "r" (addr), "i" (-EFAULT));
#endif

		if (res)
			goto fault;
		*newvalue = value;
		*regptr = &regs->regs[insn.i_format.rt];
		break;

	case lhu_op:
		if (verify_area(VERIFY_READ, addr, 2))
			goto sigbus;

		__asm__ __volatile__ (
			".set\tnoat\n"
#ifdef __BIG_ENDIAN
			"1:\tlbu\t%0, 0(%2)\n"
			"2:\tlbu\t$1, 1(%2)\n\t"
#endif
#ifdef __LITTLE_ENDIAN
			"1:\tlbu\t%0, 1(%2)\n"
			"2:\tlbu\t$1, 0(%2)\n\t"
#endif
			"sll\t%0, 0x8\n\t"
			"or\t%0, $1\n\t"
			"li\t%1, 0\n"
			"3:\t.set\tat\n\t"
			".section\t.fixup,\"ax\"\n\t"
			"4:\tli\t%1, %3\n\t"
			"j\t3b\n\t"
			".previous\n\t"
			".section\t__ex_table,\"a\"\n\t"
			STR(PTR)"\t1b, 4b\n\t"
			STR(PTR)"\t2b, 4b\n\t"
			".previous"
			: "=&r" (value), "=r" (res)
			: "r" (addr), "i" (-EFAULT));
		if (res)
			goto fault;
		*newvalue = value;
		*regptr = &regs->regs[insn.i_format.rt];
		break;

	case lwu_op:
		/* Cannot handle 64-bit instructions in 32-bit kernel */
		goto sigill;

	case ld_op:
		/* Cannot handle 64-bit instructions in 32-bit kernel */
		goto sigill;

	case sh_op:
		if (verify_area(VERIFY_WRITE, addr, 2))
			goto sigbus;

		value = regs->regs[insn.i_format.rt];
		__asm__ __volatile__ (
			".set\tnoat\n"
			"1:\tsb\t%1, 1(%2)\n\t"
			"srl\t$1, %1, 0x8\n"
			"2:\tsb\t$1, 0(%2)\n\t"
			".set\tat\n\t"
			"li\t%0, 0\n"
			"3:\n\t"
			".section\t.fixup,\"ax\"\n\t"
			"4:\tli\t%0, %3\n\t"
			"j\t3b\n\t"
			".previous\n\t"
			".section\t__ex_table,\"a\"\n\t"
			STR(PTR)"\t1b, 4b\n\t"
			STR(PTR)"\t2b, 4b\n\t"
			".previous"
			: "=r" (res)
			: "r" (value), "r" (addr), "i" (-EFAULT));
		if (res)
			goto fault;
		break;

	case sw_op:
		if (verify_area(VERIFY_WRITE, addr, 4))
			goto sigbus;

		value = regs->regs[insn.i_format.rt];
		__asm__ __volatile__ (
			"1:\tswl\t%1,(%2)\n"
			"2:\tswr\t%1, 3(%2)\n\t"
			"li\t%0, 0\n"
			"3:\n\t"
			".section\t.fixup,\"ax\"\n\t"
			"4:\tli\t%0, %3\n\t"
			"j\t3b\n\t"
			".previous\n\t"
			".section\t__ex_table,\"a\"\n\t"
			STR(PTR)"\t1b, 4b\n\t"
			STR(PTR)"\t2b, 4b\n\t"
			".previous"
		: "=r" (res)
		: "r" (value), "r" (addr), "i" (-EFAULT));
		if (res)
			goto fault;
		break;

	case sd_op:
		/* Cannot handle 64-bit instructions in 32-bit kernel */
		goto sigill;

	case lwc1_op:
	case ldc1_op:
	case swc1_op:
	case sdc1_op:
		/*
		 * I herewith declare: this does not happen.  So send SIGBUS.
		 */
		goto sigbus;

	case lwc2_op:
	case ldc2_op:
	case swc2_op:
	case sdc2_op:
		/*
		 * These are the coprocessor 2 load/stores.  The current
		 * implementations don't use cp2 and cp2 should always be
		 * disabled in c0_status.  So send SIGILL.
                 * (No longer true: The Sony Praystation uses cp2 for
                 * 3D matrix operations.  Dunno if that thingy has a MMU ...)
		 */
	default:
		/*
		 * Pheeee...  We encountered an yet unknown instruction or
		 * cache coherence problem.  Die sucker, die ...
		 */
		goto sigill;
	}

#ifdef CONFIG_PROC_FS
	unaligned_instructions++;
#endif

	return 0;

fault:
	/* Did we have an exception handler installed? */
	if (fixup_exception(regs))
		return 1;

	die_if_kernel ("Unhandled kernel unaligned access", regs);
	send_sig(SIGSEGV, current, 1);

	return 0;

sigbus:
	die_if_kernel("Unhandled kernel unaligned access", regs);
	send_sig(SIGBUS, current, 1);

	return 0;

sigill:
	die_if_kernel("Unhandled kernel unaligned access or invalid instruction", regs);
	printk("%s; SIGILL\n", __FUNCTION__);
	send_sig(SIGILL, current, 1);

	return 0;
}

#ifdef CONFIG_RMT
asmlinkage void do_dataaddr_missalign(struct pt_regs *regs)
{
	unsigned long *regptr, newval;
	extern int do_dsemulret(struct pt_regs *);
	mm_segment_t seg;
	unsigned long pc;
	unsigned int address;
	unsigned int dum; 

	AXE_CP0DUMP(0);
	rmt_str_printf("### EXP: DO_DATAADDR_MISSALIGN ###\n");
#ifdef CONFIG_SMP
	printk("AXEAXE[%d] data-address misalign epc=%x\n",
		rmt_mycnum(), regs->cp0_epc);
#else
	printk("AXEAXE data-address misalign epc=%x\n", regs->cp0_epc);
#endif
	rmt_show_pt_regs(regs);
	show_regs(regs);

#if 1
{
		struct vm_area_struct *dbg_vma;
		int i, j;
		unsigned long psz;
		struct thread_info *ti;
		char	*comm;
		extern unsigned int kernelsp[NR_CPUS];

		for(i=0; i<NR_CPUS; i++) {
			ti = (struct thread_info *)(kernelsp[i] & 0xffffe000);
			if (ti == NULL) { // This modification is temporary
				continue;
			}
			comm = ti->task->comm;
			if(comm != NULL)
			printk("%d: %s[%d]\n",
				i, ti->task->comm, ti->task->pid);
			
			if(ti->task->mm)
			dbg_vma = ti->task->mm->mmap;
			else
			dbg_vma = 0;
			while(dbg_vma) {
				for(j=dbg_vma->vm_start; j<dbg_vma->vm_end;
				    j+=PAGE_SIZE) {
				    	psz = get_phys_page(j, current->mm);
					if(psz)
						printk("cpu%d: %08x (phy:%8x)\n",
							i, j, psz);
				}
				dbg_vma = dbg_vma->vm_next;
			}
		}
		
#ifdef CONFIG_RMT_INSTSIM
		__axe_write_c0_register(0x09000000); // data TLB dump
		__axe_write_c0_register(0x08000000); // instruction TLB dump
		AXE_MONITOR();
#endif
}
#endif

	/*
	 * Address errors may be deliberately induced by the FPU emulator to
	 * retake control of the CPU after executing the instruction in the
	 * delay slot of an emulated branch.
	 */
	/* Terminate if exception was recognized as a delay slot return */
	if (do_dsemulret(regs))
		return;

	/* Otherwise handle as normal */

	/*
	 * Did we catch a fault trying to load an instruction?
	 * Or are we running in MIPS16 mode?
	 */
	if ((regs->mmu_spr_exp_addr == regs->cp0_epc) || (regs->cp0_epc & 0x1))
		goto sigbus;

	pc = exception_epc(regs);
	if ((current->thread.mflags & MF_FIXADE) == 0)
		goto sigbus;

	/*
	 * Do branch emulation only if we didn't forward the exception.
	 * This is all so but ugly ...
	 */
	seg = get_fs();
	if (!user_mode(regs))
		set_fs(KERNEL_DS);
	address = get_fault_address(regs, &dum);
	if (!emulate_load_store_insn(regs, (void *)address, pc,
	//	if (!emulate_load_store_insn(regs, (void *)regs->mmu_spr_exp_addr, pc,
	                             &regptr, &newval)) {
		compute_return_epc(regs);
		/*
		 * Now that branch is evaluated, update the dest
		 * register if necessary
		 */
		if (regptr)
			*regptr = newval;
	}
	set_fs(seg);

	return;

sigbus:
	die_if_kernel("Kernel unaligned instruction access", regs);
	force_sig(SIGBUS, current);

	/*
	 * XXX On return from the signal handler we should advance the epc
	 */
}
#else
asmlinkage void do_ade(struct pt_regs *regs)
{
	unsigned long *regptr, newval;
	extern int do_dsemulret(struct pt_regs *);
	mm_segment_t seg;
	unsigned long pc;

	/*
	 * Address errors may be deliberately induced by the FPU emulator to
	 * retake control of the CPU after executing the instruction in the
	 * delay slot of an emulated branch.
	 */
	/* Terminate if exception was recognized as a delay slot return */
	if (do_dsemulret(regs))
		return;

	/* Otherwise handle as normal */

	/*
	 * Did we catch a fault trying to load an instruction?
	 * Or are we running in MIPS16 mode?
	 */
	if ((regs->mmu_spr_exp_addr == regs->cp0_epc) || (regs->cp0_epc & 0x1))
		goto sigbus;

	pc = exception_epc(regs);
	if ((current->thread.mflags & MF_FIXADE) == 0)
		goto sigbus;

	/*
	 * Do branch emulation only if we didn't forward the exception.
	 * This is all so but ugly ...
	 */
	seg = get_fs();
	if (!user_mode(regs))
		set_fs(KERNEL_DS);
	if (!emulate_load_store_insn(regs, (void *)regs->mmu_spr_exp_addr, pc,
	                             &regptr, &newval)) {
		compute_return_epc(regs);
		/*
		 * Now that branch is evaluated, update the dest
		 * register if necessary
		 */
		if (regptr)
			*regptr = newval;
	}
	set_fs(seg);

	return;

sigbus:
	die_if_kernel("Kernel unaligned instruction access", regs);
	force_sig(SIGBUS, current);

	/*
	 * XXX On return from the signal handler we should advance the epc
	 */
}
#endif
