/*
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file "COPYING" in the main directory of this archive
 * for more details.
 *
 * Copyright (C) 1994 - 2000 Ralf Baechle
 * Copyright (C) 1999, 2000 Silicon Graphics, Inc.
 * Kevin D. Kissell, kevink@mips.com and Carsten Langgaard, carstenl@mips.com
 * Copyright (C) 2000 MIPS Technologies, Inc.  All rights reserved.
 */
#include <linux/config.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/signal.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <linux/types.h>
#include <linux/pagemap.h>
#include <linux/ptrace.h>
#include <linux/mman.h>
#include <linux/mm.h>
#include <linux/bootmem.h>
#include <linux/highmem.h>
#include <linux/swap.h>

#include <asm/bootinfo.h>
#include <asm/cachectl.h>
#include <asm/cpu.h>
#include <asm/dma.h>
#include <asm/pgalloc.h>
#include <asm/mmu_context.h>
#include <asm/sections.h>
#include <asm/tlb.h>

DEFINE_PER_CPU(struct mmu_gather, mmu_gathers);

unsigned long highstart_pfn, highend_pfn;

/*
 * We have up to 8 empty zeroed pages so we can map one of the right colour
 * when needed.  This is necessary only on R4000 / R4400 SC and MC versions
 * where we have to avoid VCED / VECI exceptions for good performance at
 * any price.  Since page is never written to after the initialization we
 * don't have to care about aliases on other CPUs.
 */
unsigned long empty_zero_page, zero_page_mask;

/*
 * Not static inline because used by IP27 special magic initialization code
 */
unsigned long setup_zero_pages(void)
{
	rmt_str_printf("+ setup_zero_pages()\n");
	unsigned long order, size;
	struct page *page;

	if (cpu_has_vce)
		order = 3;
	else
		order = 0;

	rmt_str_printf("__get_free_pages()\n");
	empty_zero_page = __get_free_pages(GFP_KERNEL, order);
	if (!empty_zero_page)
		panic("Oh boy, that early out of memory?");

	rmt_str_printf("virt_to_page()\n");
	rmt_str_printf("empty_zero_page: 0x"); rmt_hex_printf(empty_zero_page);
	page = virt_to_page(empty_zero_page);
	while (page < virt_to_page(empty_zero_page + (PAGE_SIZE << order))) {
		set_bit(PG_reserved, &page->flags);
		set_page_count(page, 0);
		page++;
	}

	size = PAGE_SIZE << order;
	zero_page_mask = (size - 1) & PAGE_MASK;
	//printk(KERN_DEBUG "memset(%x, 0, %x)\n", empty_zero_page, size);
	memset((void *)empty_zero_page, 0, size);

	return 1UL << order;
}

#ifdef CONFIG_HIGHMEM
pte_t *kmap_pte;
pgprot_t kmap_prot;

EXPORT_SYMBOL(kmap_prot);
EXPORT_SYMBOL(kmap_pte);

#define kmap_get_fixmap_pte(vaddr)					\
	pte_offset_kernel(pmd_offset(pgd_offset_k(vaddr), (vaddr)), (vaddr))

static void __init kmap_init(void)
{
	unsigned long kmap_vstart;

	/* cache the first kmap pte */
	kmap_vstart = __fix_to_virt(FIX_KMAP_BEGIN);
	kmap_pte = kmap_get_fixmap_pte(kmap_vstart);

	kmap_prot = PAGE_KERNEL;
}
#endif /* CONFIG_HIGHMEM */

#ifndef CONFIG_DISCONTIGMEM
extern void pagetable_init(void);

void __init paging_init(void)
{
	unsigned long zones_size[MAX_NR_ZONES] = {0, 0, 0};
	unsigned long max_dma, high, low;

	pagetable_init();

#ifdef CONFIG_HIGHMEM
	kmap_init();
#endif

	max_dma = phys_to_pfn(virt_to_phys((char *)MAX_DMA_ADDRESS));
	low = max_low_pfn;
	high = highend_pfn;

#ifdef CONFIG_ISA
	if (low < max_dma)
		zones_size[ZONE_DMA] = low;
	else {
		zones_size[ZONE_DMA] = max_dma;
		zones_size[ZONE_NORMAL] = low - max_dma;
	}
#else
	zones_size[ZONE_DMA] = low;
#endif
#ifdef CONFIG_HIGHMEM
	if (cpu_has_dc_aliases) {
		printk(KERN_WARNING "This processor doesn't support highmem.");
		if (high - low)
			printk(" %dk highmem ignored", high - low);
		printk("\n");
	} else
		zones_size[ZONE_HIGHMEM] = high - low;
#endif

	free_area_init(zones_size);
}

#define PFN_UP(x)	(((x) + PAGE_SIZE - 1) >> PAGE_SHIFT)
#define PFN_DOWN(x)	((x) >> PAGE_SHIFT)

static inline int page_is_ram(unsigned long pagenr)
{
	int i;

	for (i = 0; i < boot_mem_map.nr_map; i++) {
		unsigned long addr, end;

		if (boot_mem_map.map[i].type != BOOT_MEM_RAM)
			/* not usable memory */
			continue;

		addr = PFN_UP(boot_mem_map.map[i].addr);
		end = PFN_DOWN(boot_mem_map.map[i].addr +
			       boot_mem_map.map[i].size);

		if (pagenr >= addr && pagenr < end)
			return 1;
	}

	return 0;
}

void __init mem_init(void)
{
	unsigned long codesize, reservedpages, datasize, initsize;
	unsigned long tmp, ram;

	//printk(KERN_DEBUG "status: 0x%x\n", read_c0_status());
#ifdef CONFIG_HIGHMEM
	highstart_pfn = (KSEG1 - KSEG0) >> PAGE_SHIFT;
	highmem_start_page = mem_map + highstart_pfn;
#ifdef CONFIG_DISCONTIGMEM
#error "CONFIG_HIGHMEM and CONFIG_DISCONTIGMEM dont work together yet"
#endif
	max_mapnr = num_physpages = highend_pfn;
#else
#ifdef CONFIG_RMT
	num_physpages = max_low_pfn - min_low_pfn;
	max_mapnr = max_low_pfn;
#else
	max_mapnr = num_physpages = max_low_pfn;
#endif
#endif
	high_memory = (void *) __va(max_low_pfn << PAGE_SHIFT);

	rmt_str_printf("free_all_bootmem()\n");
	rmt_str_printf("status=0x"); rmt_hex_printf(read_c0_status());
	totalram_pages += free_all_bootmem();
	rmt_str_printf("tmp totalram_pages: 0x"); rmt_hex_printf(totalram_pages);
	rmt_str_printf("setup_zero_pages()\n");
	rmt_str_printf("status=0x"); rmt_hex_printf(read_c0_status());
	//printk(KERN_DEBUG "status: 0x%x\n", read_c0_status());
	totalram_pages -= setup_zero_pages();	/* Setup zeroed pages.  */
	rmt_str_printf("totalram_pages: 0x"); rmt_hex_printf(totalram_pages);
	rmt_str_printf("status=0x"); rmt_hex_printf(read_c0_status());

	reservedpages = ram = 0;
	//printk(KERN_DEBUG "max_low_pfn: %lu\n", max_low_pfn);
	//printk(KERN_DEBUG "status: 0x%x\n", read_c0_status());
	for (tmp = 0; tmp < max_low_pfn; tmp++)
		if (page_is_ram(tmp)) {
			ram++;
			if (PageReserved(mem_map+tmp))
				reservedpages++;
		}
	
#ifdef CONFIG_HIGHMEM
	for (tmp = highstart_pfn; tmp < highend_pfn; tmp++) {
		struct page *page = mem_map + tmp;

		if (!page_is_ram(tmp)) {
			SetPageReserved(page);
			continue;
		}
		ClearPageReserved(page);
		set_bit(PG_highmem, &page->flags);
		atomic_set(&page->count, 1);
		__free_page(page);
		totalhigh_pages++;
	}
	totalram_pages += totalhigh_pages;
#endif

	codesize =  (unsigned long) &_etext - (unsigned long) &_text;
	datasize =  (unsigned long) &_edata - (unsigned long) &_etext;
	initsize =  (unsigned long) &__init_end - (unsigned long) &__init_begin;

	//printk(KERN_DEBUG "status: 0x%x\n", read_c0_status());
	rmt_str_printf("status=0x"); rmt_hex_printf(read_c0_status());
	rmt_str_printf("call printk()\n");
	printk(KERN_INFO "Memory: %luk/%luk available (%ldk kernel code, "
	       "%ldk reserved, %ldk data, %ldk init, %ldk highmem)\n",
	       (unsigned long) nr_free_pages() << (PAGE_SHIFT-10),
	       ram << (PAGE_SHIFT-10),
	       codesize >> 10,
	       reservedpages << (PAGE_SHIFT-10),
	       datasize >> 10,
	       initsize >> 10,
	       (unsigned long) (totalhigh_pages << (PAGE_SHIFT-10)));
}
#endif /* !CONFIG_DISCONTIGMEM */

#ifdef CONFIG_BLK_DEV_INITRD
void free_initrd_mem(unsigned long start, unsigned long end)
{
#ifdef CONFIG_RMT64
	/* Switch from KSEG0 to XKPHYS addresses */
	start = (unsigned long)phys_to_virt(CPHYSADDR(start));
	end = (unsigned long)phys_to_virt(CPHYSADDR(end));
#endif
	rmt_str_printf("free_initrd_mem: start=0x"); rmt_hex_printf(start);
	rmt_str_printf("free_initrd_mem: end=0x"); rmt_hex_printf(end);
	if (start < end)
		printk(KERN_INFO "Freeing initrd memory: %ldk freed\n",
		       (end - start) >> 10);

	for (; start < end; start += PAGE_SIZE) {
		ClearPageReserved(virt_to_page(start));
		set_page_count(virt_to_page(start), 1);
		free_page(start);
		totalram_pages++;
	}
}
#endif

extern unsigned long prom_free_prom_memory(void);

void free_initmem(void)
{
	unsigned long addr, page, freed;

	freed = prom_free_prom_memory();

	addr = (unsigned long) &__init_begin;
	while (addr < (unsigned long) &__init_end) {
#ifdef CONFIG_RMTS64
		page = PAGE_OFFSET | CPHYSADDR(addr);
#else
		page = addr;
#endif
		ClearPageReserved(virt_to_page(page));
		set_page_count(virt_to_page(page), 1);
		free_page(page);
		totalram_pages++;
		freed += PAGE_SIZE;
		addr += PAGE_SIZE;
	}
	printk(KERN_INFO "Freeing unused kernel memory: %ldk freed\n",
	       freed >> 10);
}
