/*
 * This file is subject to the terms and conditions of the GNU General
 * Public License.  See the file "COPYING" in the main directory of this
 * archive for more details.
 *
 * Copyright (C) 2000 - 2001 by Kanoj Sarcar (kanoj@sgi.com)
 * Copyright (C) 2000 - 2001 by Silicon Graphics, Inc.
 * Copyright (C) 2000, 2001, 2002 Ralf Baechle
 * Copyright (C) 2000, 2001 Broadcom Corporation
 */
#ifndef __ASM_SMP_H
#define __ASM_SMP_H

#include <linux/config.h>

#ifdef CONFIG_SMP

#include <linux/bitops.h>
#include <linux/linkage.h>
#include <linux/threads.h>
#include <linux/cpumask.h>
#include <asm/atomic.h>
#include <asm/thread_control.h>

#define smp_processor_id()	(current_thread_info()->cpu)

/* Map from cpu id to sequential logical cpu number.  This will only
   not be idempotent when cpus failed to come on-line.  */
extern int __cpu_number_map[NR_CPUS];
#define cpu_number_map(cpu)  __cpu_number_map[cpu]

/* The reverse map from sequential logical cpu number to cpu id.  */
extern int __cpu_logical_map[NR_CPUS];
#define cpu_logical_map(cpu)  __cpu_logical_map[cpu]

#define NO_PROC_ID	(-1)

struct call_data_struct {
	void		(*func)(void *);
	void		*info;
	atomic_t	started;
	atomic_t	finished;
	int		wait;
};

extern struct call_data_struct *call_data;

#define SMP_RESCHEDULE_YOURSELF	CAUSE_PND_S0
#define SMP_CALL_FUNCTION	CAUSE_PND_S1

extern cpumask_t phys_cpu_present_map;
extern cpumask_t cpu_online_map;
#define cpu_possible_map	phys_cpu_present_map

#define cpu_online(cpu)		cpu_isset(cpu, cpu_online_map)

extern cpumask_t cpu_callout_map;
/* We don't mark CPUs online until __cpu_up(), so we need another measure */
static inline int num_booting_cpus(void)
{
	return cpus_weight(cpu_callout_map);
}

/* These are defined by the board-specific code. */

/*
 * Cause the function described by call_data to be executed on the passed
 * cpu.  When the function has finished, increment the finished field of
 * call_data.
 */
extern void core_send_ipi(int cpu, unsigned int action);

/*
 * Detect available CPUs, populate phys_cpu_present_map
 */
extern void prom_build_cpu_map(void);

/*
 * Firmware CPU startup hook
 */
extern void prom_boot_secondary(int cpu, struct task_struct *idle);

/*
 *  After we've done initial boot, this function is called to allow the
 *  board code to clean up state, if needed
 */
extern void prom_init_secondary(void);

/*
 * Callout to firmware before smp_init
 */
extern void prom_prepare_cpus(unsigned int max_cpus);

/*
 * Do whatever setup needs to be done for SMP at the board level.  Return
 * the number of cpus in the system, including this one
 */
extern int prom_setup_smp(void);

/*
 * Last chance for the board code to finish SMP initialization before
 * the CPU is "online".
 */
extern void prom_smp_finish(void);

/* Hook for after all CPUs are online */
extern void prom_cpus_done(void);

extern void asmlinkage smp_bootstrap(void);

/*
 * this function sends a 'reschedule' IPI to another CPU.
 * it goes straight through and wastes no time serializing
 * anything. Worst case is that we lose a reschedule ...
 */
static inline void smp_send_reschedule(int cpu)
{
	core_send_ipi(cpu, SMP_RESCHEDULE_YOURSELF);
}

#ifdef CONFIG_RMT_RTTHREAD
#define rmt_rtthread(cpu)	((unsigned)(cpu) - (NR_CPUS-CONFIG_NR_RTTHREADS) < CONFIG_NR_RTTHREADS)
#else
#define rmt_rtthread(cpu)	0
#endif

#if defined(CONFIG_RMT_RTTHREAD) || defined(CONFIG_RMTTIMER_NOUSE)
#define rmt_timer_count_thread()	NR_CPUS
#endif

#endif /* CONFIG_SMP */

#endif /* __ASM_SMP_H */
