/*
 *  linux/init/main.c
 *
 *  Copyright (C) 1991, 1992  Linus Torvalds
 *
 *  GK 2/5/95  -  Changed to support mounting root fs via NFS
 *  Added initrd & change_root: Werner Almesberger & Hans Lermen, Feb '96
 *  Moan early if gcc is old, avoiding bogus kernels - Paul Gortmaker, May '96
 *  Simplified starting of init:  Michael A. Griffith <grif@acm.org> 
 */

#define __KERNEL_SYSCALLS__

#include <linux/config.h>
#include <linux/types.h>
#include <linux/module.h>
#include <linux/proc_fs.h>
#include <linux/devfs_fs_kernel.h>
#include <linux/kernel.h>
#include <linux/syscalls.h>
#include <linux/string.h>
#include <linux/ctype.h>
#include <linux/delay.h>
#include <linux/utsname.h>
#include <linux/ioport.h>
#include <linux/init.h>
#include <linux/smp_lock.h>
#include <linux/initrd.h>
#include <linux/hdreg.h>
#include <linux/bootmem.h>
#include <linux/tty.h>
#include <linux/gfp.h>
#include <linux/percpu.h>
#include <linux/kernel_stat.h>
#include <linux/security.h>
#include <linux/workqueue.h>
#include <linux/profile.h>
#include <linux/rcupdate.h>
#include <linux/moduleparam.h>
#include <linux/kallsyms.h>
#include <linux/writeback.h>
#include <linux/cpu.h>
#include <linux/efi.h>
#include <linux/unistd.h>

#include <asm/io.h>
#include <asm/bugs.h>
#include <asm/thread_control.h>
#include <asm/dbgout.h>

/*
 * This is one of the first .c files built. Error out early
 * if we have compiler trouble..
 */
#if __GNUC__ == 2 && __GNUC_MINOR__ == 96
#ifdef CONFIG_FRAME_POINTER
#error This compiler cannot compile correctly with frame pointers enabled
#endif
#endif

#ifdef CONFIG_X86_LOCAL_APIC
#include <asm/smp.h>
#endif

/*
 * Versions of gcc older than that listed below may actually compile
 * and link okay, but the end product can have subtle run time bugs.
 * To avoid associated bogus bug reports, we flatly refuse to compile
 * with a gcc that is known to be too old from the very beginning.
 */
#if __GNUC__ < 2 || (__GNUC__ == 2 && __GNUC_MINOR__ < 95)
#error Sorry, your GCC is too old. It builds incorrect kernels.
#endif

extern char *linux_banner;

static int init(void *);

extern void init_IRQ(void);
extern void sock_init(void);
extern void fork_init(unsigned long);
extern void mca_init(void);
extern void sbus_init(void);
extern void sysctl_init(void);
extern void signals_init(void);
extern void buffer_init(void);
extern void pidhash_init(void);
extern void pidmap_init(void);
extern void pte_chain_init(void);
extern void radix_tree_init(void);
extern void free_initmem(void);
extern void populate_rootfs(void);
extern void driver_init(void);

#ifdef CONFIG_TC
extern void tc_init(void);
#endif

/*
 * Are we up and running (ie do we have all the infrastructure
 * set up)
 */
int system_running;

/*
 * Boot command-line arguments
 */
#define MAX_INIT_ARGS 8
#define MAX_INIT_ENVS 8

extern void time_init(void);
/* Default late time init is NULL. archs can override this later. */
void (*late_time_init)(void);
extern void softirq_init(void);

static char *execute_command;

/* Setup configured maximum number of CPUs to activate */
static unsigned int max_cpus = NR_CPUS;

/*
 * Setup routine for controlling SMP activation
 *
 * Command-line option of "nosmp" or "maxcpus=0" will disable SMP
 * activation entirely (the MPS table probe still happens, though).
 *
 * Command-line option of "maxcpus=<NUM>", where <NUM> is an integer
 * greater than 0, limits the maximum number of CPUs activated in
 * SMP mode to <NUM>.
 */
static int __init nosmp(char *str)
{
	max_cpus = 0;
	return 1;
}

__setup("nosmp", nosmp);

static int __init maxcpus(char *str)
{
	get_option(&str, &max_cpus);
	return 1;
}

__setup("maxcpus=", maxcpus);

static char * argv_init[MAX_INIT_ARGS+2] = { "init", NULL, };
char * envp_init[MAX_INIT_ENVS+2] = { "HOME=/", "TERM=linux", NULL, };
static const char *panic_later, *panic_param;

__setup("profile=", profile_setup);

static int __init obsolete_checksetup(char *line)
{
	struct obs_kernel_param *p;
	extern struct obs_kernel_param __setup_start, __setup_end;

	p = &__setup_start;
	do {
		int n = strlen(p->str);
		if (!strncmp(line, p->str, n)) {
			if (!p->setup_func) {
				printk(KERN_WARNING "Parameter %s is obsolete, ignored\n", p->str);
				return 1;
			} else if (p->setup_func(line + n))
				return 1;
		}
		p++;
	} while (p < &__setup_end);
	return 0;
}

/* this should be approx 2 Bo*oMips to start (note initial shift), and will
   still work even if initially too large, it will just take slightly longer */
unsigned long loops_per_jiffy = (1<<12);

#ifndef __ia64__
EXPORT_SYMBOL(loops_per_jiffy);
#endif

/* This is the number of bits of precision for the loops_per_jiffy.  Each
   bit takes on average 1.5/HZ seconds.  This (like the original) is a little
   better than 1% */
#define LPS_PREC 8

#if defined(CONFIG_RMTTIMER_NOUSE) || defined(CONFIG_CALIBRATE_DELAY_AVOIDANCE)
void __init calibrate_delay(void)
{
	rmt_str_printf("skip calibrate_delay\n");
#ifdef CONFIG_DEFALUT_LOOPS_PER_JIFFY
	loops_per_jiffy = CONFIG_DEFALUT_LOOPS_PER_JIFFY;
#else
	loops_per_jiffy = 15 * (500000/HZ);
#endif
}
#else
void __init calibrate_delay(void)
{
	unsigned long ticks, loopbit;
	int lps_precision = LPS_PREC;

#ifdef CONFIG_SMP
	if(rmt_mycnum())
		return;
#endif

	loops_per_jiffy = (1<<12);

#ifndef CONFIG_AXE_DEBUG
	printk("Calibrating delay loop... ");
#endif

	rmt_str_printf("jiffies: ");
	rmt_hex_print(jiffies);
	ticks = jiffies;
	while (ticks == jiffies){
		rmt_print_val_hex("count", read_c0_count());
		rmt_print_val_hex("compare", read_c0_compare());
	}
	rmt_str_printf("=1\n");
	rmt_str_printf("jiffies: ");
	rmt_hex_print(jiffies);
	ticks = jiffies;
	while (ticks == jiffies) {
		rmt_print_val_hex("count", read_c0_count());
		rmt_print_val_hex("compare", read_c0_compare());
	}
	rmt_str_printf("=2\n");
	rmt_str_printf("jiffies: ");
	rmt_hex_print(jiffies);
	ticks = jiffies;
	while (ticks == jiffies){
		rmt_print_val_hex("count", read_c0_count());
		rmt_print_val_hex("compare", read_c0_compare());
	}
	rmt_str_printf("=3\n");
	rmt_str_printf("jiffies: ");
	rmt_hex_print(jiffies);
	ticks = jiffies;
	while (ticks == jiffies){
		rmt_print_val_hex("count", read_c0_count());
		rmt_print_val_hex("compare", read_c0_compare());
	}
	rmt_str_printf("=4\n");
	rmt_str_printf("jiffies: ");
	rmt_hex_print(jiffies);

	rmt_str_printf("calibrate_delay #1\n");
	while (loops_per_jiffy <<= 1) {
		rmt_str_printf("jiffies: ");
		rmt_hex_print(jiffies);
		rmt_str_printf(" loop: ");
		rmt_hex_print(loops_per_jiffy);

		/* wait for "start of" clock tick */
		ticks = jiffies;
		while (ticks == jiffies)
			/* nothing */;
		/* Go .. */
		ticks = jiffies;
		__delay(loops_per_jiffy);
		ticks = jiffies - ticks;
		rmt_str_printf(" ticks: ");
		rmt_print_dec(ticks);
		rmt_str_printf("\n");
		if (ticks)
			break;
	}
	rmt_str_printf("calibrate_delay #3\n");

/* Do a binary approximation to get loops_per_jiffy set to equal one clock
   (up to lps_precision bits) */
	loops_per_jiffy >>= 1;
	loopbit = loops_per_jiffy;
	while ( lps_precision-- && (loopbit >>= 1) ) {
		loops_per_jiffy |= loopbit;
		ticks = jiffies;
		while (ticks == jiffies);
		ticks = jiffies;
		__delay(loops_per_jiffy);
		if (jiffies != ticks)	/* longer than 1 tick */
			loops_per_jiffy &= ~loopbit;
	}
	rmt_str_printf("calibrate_delay #4\n");
	rmt_print_dec(loops_per_jiffy/(500000/HZ));
	rmt_str_printf(".");
	rmt_print_dec((loops_per_jiffy/(5000/HZ)) % 100);
	rmt_str_printf(" BogoMIPS\n");
	rmt_print_dec("loops_per_jiffy", loops_per_jiffy);

#ifdef CONFIG_AXE_DEBUG
	printk("Calibrating delay loop... %lu.%02lu BogoMIPS\n",
		loops_per_jiffy/(500000/HZ),
		(loops_per_jiffy/(5000/HZ)) % 100);
#else
	printk("%lu.%02lu BogoMIPS\n",
		loops_per_jiffy/(500000/HZ),
		(loops_per_jiffy/(5000/HZ)) % 100);
#endif
}
#endif /* CONFIG_RMTTIMER_NOUSE */

static int __init debug_kernel(char *str)
{
	if (*str)
		return 0;
	console_loglevel = 10;
	return 1;
}

static int __init quiet_kernel(char *str)
{
	if (*str)
		return 0;
	console_loglevel = 4;
	return 1;
}

__setup("debug", debug_kernel);
__setup("quiet", quiet_kernel);

/* Unknown boot options get handed to init, unless they look like
   failed parameters */
static int __init unknown_bootoption(char *param, char *val)
{
	/* Change NUL term back to "=", to make "param" the whole string. */
	if (val)
		val[-1] = '=';

	/* Handle obsolete-style parameters */
	if (obsolete_checksetup(param))
		return 0;

	/* Preemptive maintenance for "why didn't my mispelled command
           line work?" */
	if (strchr(param, '.') && (!val || strchr(param, '.') < val)) {
		printk(KERN_ERR "Unknown boot option `%s': ignoring\n", param);
		return 0;
	}

	if (panic_later)
		return 0;

	if (val) {
		/* Environment option */
		unsigned int i;
		for (i = 0; envp_init[i]; i++) {
			if (i == MAX_INIT_ENVS) {
				panic_later = "Too many boot env vars at `%s'";
				panic_param = param;
			}
		}
		envp_init[i] = param;
	} else {
		/* Command line option */
		unsigned int i;
		for (i = 0; argv_init[i]; i++) {
			if (i == MAX_INIT_ARGS) {
				panic_later = "Too many boot init vars at `%s'";
				panic_param = param;
			}
		}
		argv_init[i] = param;
	}
	return 0;
}

static int __init init_setup(char *str)
{
	unsigned int i;
	rmt_str_printf("+ init_setup()\n");

	execute_command = str;
	/* In case LILO is going to boot us with default command line,
	 * it prepends "auto" before the whole cmdline which makes
	 * the shell think it should execute a script with such name.
	 * So we ignore all arguments entered _before_ init=... [MJ]
	 */
	for (i = 1; i < MAX_INIT_ARGS; i++)
		argv_init[i] = NULL;
	return 1;
}
__setup("init=", init_setup);

extern void setup_arch(char **);
extern void cpu_idle(void);

#ifndef CONFIG_SMP

#ifdef CONFIG_X86_LOCAL_APIC
static void __init smp_init(void)
{
	APIC_init_uniprocessor();
}
#else
#define smp_init()	do { } while (0)
#endif

static inline void setup_per_cpu_areas(void) { }
static inline void smp_prepare_cpus(unsigned int maxcpus) { }

#else

#ifdef __GENERIC_PER_CPU
unsigned long __per_cpu_offset[NR_CPUS];

EXPORT_SYMBOL(__per_cpu_offset);

static void __init setup_per_cpu_areas(void)
{
	unsigned long size, i;
	char *ptr;
	/* Created by linker magic */
	extern char __per_cpu_start[], __per_cpu_end[];

	/* Copy section for each CPU (we discard the original) */
	size = ALIGN(__per_cpu_end - __per_cpu_start, SMP_CACHE_BYTES);
#ifdef CONFIG_MODULES
	if (size < PERCPU_ENOUGH_ROOM)
		size = PERCPU_ENOUGH_ROOM;
#endif

	ptr = alloc_bootmem(size * NR_CPUS);

	for (i = 0; i < NR_CPUS; i++, ptr += size) {
		__per_cpu_offset[i] = ptr - __per_cpu_start;
		memcpy(ptr, __per_cpu_start, __per_cpu_end - __per_cpu_start);
	}
}
#endif /* !__GENERIC_PER_CPU */

/* Called by boot processor to activate the rest. */
static void __init smp_init(void)
{
	unsigned int i;
	unsigned j = 1;

	/* FIXME: This should be done in userspace --RR */
	for (i = 0; i < NR_CPUS; i++) {
		if (num_online_cpus() >= max_cpus)
			break;

		rmt_str_printf("cpu=0x"); rmt_hex_printf(i);
		rmt_str_printf("possible=0x"); rmt_hex_printf(cpu_possible(i));
		rmt_str_printf("online=0x"); rmt_hex_printf(cpu_online(i));
		if (cpu_possible(i) && !cpu_online(i)) {
			cpu_up(i);
			j++;
		}
	}

	/* Any cleanup work */
	rmt_str_printf("- success second thread!!!\n");
	printk("Brought up %u Threads\n", j);
	smp_cpus_done(max_cpus);
#if 0
	/* Get other processors into their bootup holding patterns. */

	smp_threads_ready=1;
	smp_commence();
#endif
	rmt_str_printf("- smp_init\n");
}

#endif

/*
 * We need to finalize in a non-__init function or else race conditions
 * between the root thread and the init thread may cause start_kernel to
 * be reaped by free_initmem before the root thread has proceeded to
 * cpu_idle.
 *
 * gcc-3.4 accidentally inlines this function, so use noinline.
 */

static void noinline rest_init(void)
{
	rmt_str_printf("0:kernel_thread()\n");
	rmt_str_printf("0:current=0x"); rmt_hex_printf(current);
	kernel_thread(init, NULL, CLONE_FS | CLONE_SIGHAND);
	rmt_str_printf("unlock_kernel()\n");
	rmt_str_printf("1:current=0x"); rmt_hex_printf(current);
	unlock_kernel();
	rmt_str_printf("cpu_idle(): pid=0x"); rmt_hex_printf(current->pid);
	rmt_str_printf("2:current=0x"); rmt_hex_printf(current);
 	cpu_idle();
}

/*
 *	Activate the first processor.
 */
asmlinkage void __init start_kernel(void)
{
	char * command_line;
	extern char saved_command_line[];
	extern struct kernel_param __start___param[], __stop___param[];
/*
 * Interrupts are still disabled. Do necessary setups, then
 * enable them
 */
  	
	rmt_str_printf("start_kernel StatusRegister: 0x");
	rmt_hex_printf(read_c0_status());
	rmt_str_printf("exp base addr reg: 0x");
	rmt_hex_printf(__read_32bit_c0_register(CP0_EXP_BASE_ADDR, 0));
	rmt_str_printf("icache-on: 0x");
	rmt_hex_printf(__read_32bit_c0_register(0x80, 0));
	rmt_str_printf("dcache-on: 0x");
	rmt_hex_printf(__read_32bit_c0_register(0x86, 0));
	rmt_str_printf("current=0x"); rmt_hex_printf(current);

	/* Dummy Exception Table */
	extern void set_temporary_exception_handlers(void);
	set_temporary_exception_handlers();

#if 0
	rmt_str_printf("Try Reserved Instruction\n");
	__asm__ __volatile__ (".long 0x0000000a");
#endif
#if 0
	rmt_str_printf("Try syscall\n");
	__asm__ __volatile__ ("syscall");
#endif
	
	rmt_str_printf("lock_kernel()\n");
	lock_kernel();
	
//AXE_PUTCHAR('1'); AXE_PUTCHAR('\n');
	rmt_str_printf("page_address_init()\n");
	page_address_init();
	
//AXE_PUTCHAR('2'); AXE_PUTCHAR('\n');
	rmt_str_printf("printk(linux_banner)\n");
	rmt_str_printf("current=0x"); rmt_hex_printf(current);
	printk(linux_banner);
	
//AXE_PUTCHAR('3'); AXE_PUTCHAR('\n');
	rmt_str_printf("setup_arch()\n");
	setup_arch(&command_line);

	/* rmt_str_printf("free_pages: 0x");
	rmt_hex_printf(NODE_DATA(0)->node_zonelists[0].zones[0]->free_pages);
	*/
//AXE_PUTCHAR('4'); AXE_PUTCHAR('\n');
	rmt_str_printf("setup_per_cpu_areas()\n");
	setup_per_cpu_areas();
	
	/*
	 * Mark the boot cpu "online" so that it can call console drivers in
	 * printk() and can access its per-cpu storage.
	 */
	rmt_str_printf("smp_prepare_boot_cpu()\n");
	rmt_str_printf("status=0x"); rmt_hex_printf(read_c0_status());
	smp_prepare_boot_cpu();

	rmt_str_printf("build_all_zonelists()\n");
	build_all_zonelists();
	rmt_str_printf("page_alloc_init()\n");
	page_alloc_init();
	printk("Kernel command line: %s\n", saved_command_line);
	parse_args("Booting kernel", command_line, __start___param,
		   __stop___param - __start___param,
		   &unknown_bootoption);
	rmt_str_printf("sort_main_extable()\n");
	sort_main_extable();
	rmt_str_printf("trap_init()\n");
	rmt_str_printf("status=0x"); rmt_hex_printf(read_c0_status());
	trap_init();
	rmt_str_printf("rcu_init()\n");
	rmt_str_printf("status=0x"); rmt_hex_printf(read_c0_status());
	rcu_init();
	rmt_str_printf("init_IRQ()\n");
	init_IRQ();
	rmt_str_printf("pidhash_init()\n");
	pidhash_init();
	rmt_str_printf("sched_init()\n");
	sched_init();
	rmt_str_printf("softirq_init\n");
	softirq_init();
	rmt_str_printf("time_init()\n");
	rmt_str_printf("current=0x"); rmt_hex_printf(current);
	time_init();

	/*
	 * HACK ALERT! This is early. We're enabling the console before
	 * we've done PCI setups etc, and console_init() must be aware of
	 * this. But we do want output early, in case something goes wrong.
	 */
	rmt_str_printf("+ console_init()\n");
	rmt_str_printf("status=0x"); rmt_hex_printf(read_c0_status());
	console_init();
	rmt_str_printf("- console_init()\n");
	if (panic_later)
		panic(panic_later, panic_param);
	rmt_str_printf("profile_init()\n");
	rmt_str_printf("status=0x"); rmt_hex_printf(read_c0_status());
	profile_init();
	rmt_str_printf("local_irq_enable()\n");
	local_irq_enable();

	rmt_str_printf("status=0x"); rmt_hex_printf(read_c0_status());
#ifdef CONFIG_BLK_DEV_INITRD
	if (initrd_start && !initrd_below_start_ok &&
			initrd_start < pfn_to_phys(min_low_pfn)) {
		printk(KERN_CRIT "initrd overwritten (0x%08lx < 0x%08lx) - "
		    "disabling it.\n",initrd_start,pfn_to_phys(min_low_pfn));
		initrd_start = 0;
	}
#endif
	rmt_str_printf("mem_init()\n");
	rmt_str_printf("status=0x"); rmt_hex_printf(read_c0_status());
	mem_init();
	rmt_str_printf("kmem_cache_init()\n");
	rmt_str_printf("status=0x"); rmt_hex_printf(read_c0_status());
	kmem_cache_init();
	if (late_time_init)
		late_time_init();

	rmt_str_printf("calibrate_delay()\n");
	rmt_str_printf("status=0x"); rmt_hex_printf(read_c0_status());
	calibrate_delay();
	rmt_str_printf("pidmap_init()\n");
	pidmap_init();
	rmt_str_printf("pgtable_cache_init()\n");
	pgtable_cache_init();
	rmt_str_printf("pte_chain_init()\n");
	rmt_str_printf("current=0x"); rmt_hex_printf(current);
	pte_chain_init();
#ifdef CONFIG_X86
	if (efi_enabled)
		efi_enter_virtual_mode();
#endif

#if 0
	rmt_str_printf("########## + TEST vmalloc()\n");
	int i, n=0x10;
	unsigned long *buf;
	rmt_str_printf("VMALLOC_START=0x"); rmt_hex_printf(VMALLOC_START);
	rmt_str_printf("VMALLOC_END=0x"); rmt_hex_printf(VMALLOC_END);
	rmt_str_printf("vmalloc size=0x"); rmt_hex_printf(sizeof(unsigned long)*n);
	
	buf = vmalloc(sizeof(unsigned long) * n);
	rmt_str_printf("buf=0x"); rmt_hex_printf(buf);
	for(i=0; i<n; i++) {
		buf[i] = (unsigned long)i+1UL;
		rmt_str_printf(" buf[0x");
		rmt_hex_print(i);
		rmt_str_printf("]=0x");
		rmt_hex_printf(buf[i]);
	}
	rmt_str_printf("vfree()\n");
	vfree(buf);
	rmt_str_printf("########## - TEST vmalloc()\n");
#endif

	rmt_str_printf("fork_init(num_physpages)\n");
	fork_init(num_physpages);
	rmt_str_printf("proc_caches_init()\n");
	proc_caches_init();
	rmt_str_printf("buffer_init()\n");
	buffer_init();
	rmt_str_printf("unnamed_dev_init()\n");
	unnamed_dev_init();
	rmt_str_printf("security_scaffolding_startup()\n");
	security_scaffolding_startup();
	rmt_str_printf("vfs_caches_init()\n");
	vfs_caches_init(num_physpages);
	rmt_str_printf("radix_tree_init()\n");
	radix_tree_init();
	rmt_str_printf("signals_init()\n");
	signals_init();
	/* rootfs populating might need page-writeback */
	rmt_str_printf("page_writeback_init()\n");
	page_writeback_init();
	rmt_str_printf("populate_rootfs()\n");
	rmt_str_printf("current=0x"); rmt_hex_printf(current);
	populate_rootfs();
#ifdef CONFIG_PROC_FS
	rmt_str_printf("proc_root_init()\n");
	proc_root_init();
#endif
	rmt_str_printf("check_bugs()\n");
	check_bugs();
	printk("POSIX conformance testing by UNIFIX\n");

#if 0
	/*
	 * TEST vmalloc
	 */
	rmt_str_printf("########## + TEST vmalloc()\n");
	int i, n=0x3, m=0x10;
	unsigned long *buf, *buf2;
	rmt_str_printf("PGDIR_SHIFT=0x"); rmt_hex_printf(PGDIR_SHIFT);
	rmt_str_printf("PGDIR_SIZE=0x"); rmt_hex_printf(PGDIR_SIZE);
	rmt_str_printf("PGDIR_MASK=0x"); rmt_hex_printf(PGDIR_MASK);
	rmt_str_printf("VMALLOC_START=0x"); rmt_hex_printf(VMALLOC_START);
	rmt_str_printf("VMALLOC_END=0x"); rmt_hex_printf(VMALLOC_END);
	rmt_str_printf("vmalloc size=0x"); rmt_hex_printf(sizeof(unsigned long)*n);
	rmt_str_printf("vmalloc size=0x"); rmt_hex_printf(sizeof(unsigned long)*m);
	
	buf = vmalloc(sizeof(unsigned long) * n);
	buf2 = vmalloc(sizeof(unsigned long) * m);
	rmt_str_printf("buf=0x"); rmt_hex_printf(buf);
	rmt_str_printf("buf2=0x"); rmt_hex_printf(buf2);
	for(i=0; i<n; i++) {
		printk(" buf[%d]=0x%x\n", i, buf[i]);
		printk("buf2[%d]=0x%x\n", i, buf2[i]);
		buf[i] = (unsigned long)i+1UL;
		buf2[i] = (unsigned long)i+2UL;
		printk(" buf[%d]=0x%x\n", i, buf[i]);
		printk("buf2[%d]=0x%x\n", i, buf2[i]);
	}
	for(i=n; i<m; i++) {
		printk("buf2[%d]=0x%x\n", i, buf2[i]);
		buf2[i] = (unsigned long)i+2UL;
		printk("buf2[%d]=0x%x\n", i, buf2[i]);
	}
	
	rmt_str_printf("pgd_current=0x"); rmt_hex_printf(pgd_current[0]);
	rmt_str_printf("vfree()\n");
	vfree(buf);
	for(i=0; i<m; i++) {
		printk("buf2[%d]=0x%x\n", i, buf2[i]);
		buf2[i] = (unsigned long)i*10UL;
		printk("buf2[%d]=0x%x\n", i, buf2[i]);
	}
	vfree(buf2);
	/*
	for(i=0; i<n; i++) {
		rmt_str_printf("buf[i]=0x"); rmt_hex_printf(buf[i]);
		buf[i] = i+1;
		rmt_str_printf("buf[i]=0x"); rmt_hex_printf(buf[i]);
	}
	*/
	rmt_str_printf("########## - TEST vmalloc()\n");
#endif

	/* 
	 *	We count on the initial thread going ok 
	 *	Like idlers init is an unlocked kernel thread, which will
	 *	make syscalls (and thus be locked).
	 */
	rmt_str_printf("init_idle()\n");
	rmt_str_printf("current=0x"); rmt_hex_printf(current);
	init_idle(current, smp_processor_id());

	/* Do the rest non-__init'ed, we're now alive */
	rmt_str_printf("rest_init()\n");
	rmt_str_printf("current=0x"); rmt_hex_printf(current);
	rest_init();
}

static int __initdata initcall_debug;

static int __init initcall_debug_setup(char *str)
{
	initcall_debug = 1;
	return 1;
}
__setup("initcall_debug", initcall_debug_setup);

struct task_struct *child_reaper = &init_task;

extern initcall_t __initcall_start, __initcall_end;

static void __init do_initcalls(void)
{
	initcall_t *call;
	//printk(KERN_DEBUG "preempt_count: pid=0x%x\n", current->pid);
	int count = preempt_count();
#ifdef CONFIG_RMT
	int	initcall_num = (&__initcall_end-&__initcall_start);
	int	initcall_counter = 0;
#endif

	for (call = &__initcall_start; call < &__initcall_end; call++) {
		char *msg;

		if (initcall_debug) {
			printk(KERN_DEBUG "Calling initcall 0x%p", *call);
			print_symbol(": %s()", (unsigned long) *call);
			printk("\n");
		}

#ifdef CONFIG_AXE_DEBUG
		printk(KERN_DEBUG "Calling initcall 0x%p (%d/%d)\n",
			*call, initcall_counter+1, initcall_num);
		initcall_counter++;
#endif
		(*call)();

		msg = NULL;
		if (preempt_count() != count) {
			msg = "preemption imbalance";
			preempt_count() = count;
		}
		if (irqs_disabled()) {
			msg = "disabled interrupts";
			local_irq_enable();
		}
		if (msg) {
			printk("error in initcall at 0x%p: "
				"returned with %s\n", *call, msg);
		}
	}

	/* Make sure there is no pending stuff from the initcall sequence */
	rmt_str_printf("flush_shceduled_work(): pid=0x%x\n", current->pid);
	flush_scheduled_work();
}

/*
 * Ok, the machine is now initialized. None of the devices
 * have been touched yet, but the CPU subsystem is up and
 * running, and memory and process management works.
 *
 * Now we can finally start doing some real work..
 */
static void __init do_basic_setup(void)
{
	rmt_str_printf("driver_init()\n");
	driver_init();

#ifdef CONFIG_SYSCTL
	rmt_str_printf("sysctl_init()\n");
	sysctl_init();
#endif

	rmt_str_printf("sock_init()\n");
	/* Networking initialization needs a process context */ 
	sock_init();

	rmt_str_printf("init_workqueues()\n");
	init_workqueues();
	rmt_str_printf("do_initcalls()\n");
	do_initcalls();
	rmt_str_printf("- do_initcalls()\n");
}

static void do_pre_smp_initcalls(void)
{
	extern int spawn_ksoftirqd(void);
#ifdef CONFIG_SMP
	extern int migration_init(void);

	migration_init();
#endif
	rmt_str_printf("node_nr_running_init()\n");
	node_nr_running_init();
	rmt_str_printf("spawn_ksoftirqd()\n");
	spawn_ksoftirqd();
}

static void run_init_process(char *init_filename)
{
	int i;
	//rmt_str_printf("run_init_process()\n");
	//rmt_str_printf("smp_processor_id() = 0x");
	//rmt_hex_printf(smp_processor_id());
	
	argv_init[0] = init_filename;
	execve(init_filename, argv_init, envp_init);
	printk("-run_init_process()\n");
}

extern void prepare_namespace(void);
extern struct console rmtsim_console;

#ifdef NEVER
static void add_tv(struct timeval *tv, long sec, long usec)
{
	int carry;

	if (usec < 0 || sec < 0) {
		printk("add_tv: bad add value (sec=%d usec=%06d)\n",
		       sec, usec);
		return;
	}

	usec += tv->tv_usec;
	sec += usec / 1000000;
	usec = usec % 1000000;

	tv->tv_sec += sec;	
	tv->tv_usec = usec;	
}
static void test_rtthread(void)
{
	struct timeval start, end;
	int rval;
	unsigned period;


	period = 1000000;
	do_gettimeofday(&start);
	printk("xxx:gettimeofday: rval=%d tv=%d.%06d\n",
	       rval, start.tv_sec, start.tv_usec);
	end = start;
	add_tv(&start, 1, 0);
	add_tv(&end, 1 + 10, 0 + 6);
	printk("xxx:test_rtthread: %d.%06d -> %d.%06d period=%dns\n",
	       start.tv_sec, start.tv_usec, end.tv_sec, end.tv_usec,
	       period);
	rval = do_rtthread_start_periodic_thread(&start, &end, period);
	printk("xxx: start: rval=%d\n", rval);

	do {
		do_gettimeofday(&start);
		printk("xxx:gettimeofday: rval=%d tv=%d.%06d\n",
		       rval, start.tv_sec, start.tv_usec);
		rval = do_rtthread_next_period();
		printk("xxx: next: rval=%d\n", rval);
	} while (rval >= 0);

	do_gettimeofday(&start);
	printk("xxx:gettimeofday: rval=%d tv=%d.%06d\n",
	       rval, start.tv_sec, start.tv_usec);
}
#endif /* NEVER */

static int init(void * unused)
{
	rmt_str_printf("init()\n");
	lock_kernel();

	/*
	 * Tell the world that we're going to be the grim
	 * reaper of innocent orphaned children.
	 *
	 * We don't want people to have to make incorrect
	 * assumptions about where in the task array this
	 * can be found.
	 */
	child_reaper = current;

	/* Sets up cpus_possible() */
	rmt_str_printf("smp_prepare_cpus()\n");
	smp_prepare_cpus(max_cpus);

	rmt_str_printf("do_pre_smp_initcalls()\n");
	do_pre_smp_initcalls();

	rmt_str_printf("smp_init()\n");
	smp_init();
	rmt_str_printf("do_basic_setup()\n");
	do_basic_setup();

	rmt_str_printf("prepare_namespace()\n");
	prepare_namespace();

	/*
	 * Ok, we have completed the initial bootup, and
	 * we're essentially up and running. Get rid of the
	 * initmem segments and start the user-mode stuff..
	 */
	free_initmem();
	unlock_kernel();
	system_running = 1;

	rmt_str_printf("/dev/console is R/W\n");
	if (sys_open("/dev/console", O_RDWR, 0) < 0)
		printk("Warning: unable to open an initial console.\n");

	(void) sys_dup(0);
	(void) sys_dup(0);

#ifdef NEVER
	test_rtthread();
#endif
	
	/*
	 * We try each of these until one succeeds.
	 *
	 * The Bourne shell can be used instead of init if we are 
	 * trying to recover a really broken machine.
	 */

	if (execute_command) {
		printk("run_init_process(");
		printk(execute_command);
		printk(")\n");
		run_init_process(execute_command);
	}

	printk("run_init_process(\"/sbin/init\")\n");
	run_init_process("/sbin/init");
	printk("run_init_process(\"/etc/init\")\n");
	run_init_process("/etc/init");
	printk("run_init_process(\"/bin/init\")\n");
	run_init_process("/bin/init");
	printk("run_init_process(\"/bin/sh\")\n");
	run_init_process("/bin/sh");

	printk("No init found. panic\n");
	panic("No init found.  Try passing init= option to kernel.");
}
