/*
 * asmwrap.h: Wrapper functions of assembly instructins.
 *
 * Editors:
 *	Shinpei Kato <shinpei@ny.ics.keio.ac.jp>
 *
 * Comments:
 *	By Shinpei Kato On 7 Feb, 2005
 *	Originated this file.
 */

#ifndef __A_RMT_ASMWRAP_H__
#define __A_RMT_ASMWRAP_H__

#include <types.h>

#define nop7()			 \
	asm volatile("nop\n" \
				 "nop\n" \
				 "nop\n" \
				 "nop\n" \
				 "nop\n" \
				 "nop\n" \
				 "nop\n")
#define nop4()			 \
	asm volatile("nop\n" \
				 "nop\n" \
				 "nop\n" \
				 "nop\n")

#define sync()	asm volatile("sync")

/* System */
static inline ulong_t mfc0(ulong_t addr)
{
	ulong_t ret;

	asm volatile ("mfc0	%0,%1"
			: "=r"(ret)
			: "r"(addr));
	return ret;
}

static inline void mtc0(ulong_t data, ulong_t addr)
{
	asm volatile ("mtc0	%0,%1"
			:
			: "r"(data), "r"(addr));
}

static inline ulong_t mfimm(ulong_t addr)
{
	ulong_t ret;

	asm volatile ("mfimm	%0,%1"
			: "=r"(ret)
			: "r"(addr));
	return ret;
}

static inline void mtimm(ulong_t addr, ulong_t data)
{
	asm volatile ("mtimm	%0,%1"
			:
			: "r"(data), "r"(addr));
}

static inline ulong_t mfdmm(ulong_t addr)
{
	ulong_t ret;

	asm volatile ("mfdmm	%0,%1"
			: "=r"(ret)
			: "r"(addr));
	return ret;
}

static inline void mtdmm(ulong_t addr, ulong_t data)
{
	asm volatile ("mtdmm	%0,%1"
			:
			: "r"(data), "r"(addr));
}

static inline void eret(void)
{
	asm volatile ("eret");
}


/* Thread */
#define THREAD_OP_ERROR		0
#define THREAD_OP_SUCCESS	1

static inline ulong_t mkth(ulong_t id, ulong_t pc)
{
	register ulong_t state;

	asm volatile ("mkth	%0,%1,%2"
			: "=r" (state)
			: "r"(id), "r"(pc));

	return state;
}

static inline ulong_t delth(ulong_t id)
{
	register ulong_t state;

	asm volatile ("delth	%0,%1"
			: "=r" (state)
			: "r"(id));

	return state;
}

static inline ulong_t chgpr(ulong_t id, ulong_t prio)
{
	register ulong_t state;

	asm volatile ("chgpr	%0,%1,%2"
			: "=r" (state)
			: "r"(id), "r"(prio));
	return state;
}

static inline ulong_t chgst(ulong_t id, ulong_t status)
{
	register ulong_t state;

	asm volatile ("chgst	%0,%1,%2"
			: "=r" (state)
			: "r"(id), "r"(status));

	return state;
}

static inline ulong_t runth(ulong_t id)
{
	register ulong_t state;

	asm volatile ("runth	%0,%1"
			: "=r" (state)
			: "r"(id));

	return state;
}

static inline ulong_t stopth(ulong_t id)
{
	register ulong_t state;

	asm volatile ("stopth	%0,%1"
			: "=r" (state)
			: "r"(id));

	return state;
}

static inline ulong_t stopslf(void)
{
	register ulong_t state asm("t0");

	asm volatile ("stopslf	%0"
			: "=r" (state));

	return state;
}

static inline ulong_t bkupth(ulong_t id)
{
	register ulong_t state;

	asm volatile ("bkupth	%0,%1"
			: "=r" (state)
			: "r"(id));

	return state;
}

static inline ulong_t bkupslf(void)
{
	register ulong_t state;

	asm volatile ("bkupslf	%0"
			: "=r" (state));

	return state;
}

static inline ulong_t rstrth(ulong_t id)
{
	register ulong_t state;

	asm volatile ("rstrth	%0,%1"
			: "=r" (state)
			: "r"(id));

	return state;
}

static inline ulong_t swapth(ulong_t from, ulong_t to)
{
	register ulong_t state;

	asm volatile ("swapth	%0,%1,%2"
			: "=r" (state)
			: "r"(from), "r"(to));

	return state;
}

static inline ulong_t swapslf(ulong_t id)
{
	register ulong_t state;

	asm volatile ("swapslf	%0,%1"
			: "=r" (state)
			: "r"(id));

	return state;
}

static inline ulong_t cpthtoa(ulong_t from, ulong_t to)
{
	register ulong_t state;

	asm volatile ("cpthtoa	%0,%1,%2"
			: "=r" (state)
			: "r"(from), "r"(to));

	return state;
}

static inline ulong_t cpthtom(ulong_t from, ulong_t to)
{
	register ulong_t state;

	asm volatile ("cpthtom	%0,%1,%2"
			: "=r" (state)
			: "r"(from), "r"(to));

	return state;
}

static inline ulong_t gettt(ulong_t id)
{
	register ulong_t table;

	asm volatile ("gettt	%0,%1"
			: "=r"(table)
			: "r"(id));

	return table;
}

static inline ulong_t gettid(ulong_t context)
{
	register ulong_t id;

	asm volatile ("gettid	%0,%1"
		      : "=r"(id)
		      : "r"(context));

	return id;
}

static inline ulong_t getotid(void)
{
	register ulong_t id;

	asm volatile("getotid	%0"
			: "=r"(id));

	return id;
}

static inline ulong_t getmtid(ulong_t context)
{
	register ulong_t id;

	asm volatile("getmtid	%0,%1"
			: "=r"(id)
			: "r"(context));

	return id;
}

#define CNTXT_ACTIVE	0x100
#define CNTXT_CACHED	0x40
#define CNTXT_MASK	0x1f
static inline ulong_t getcnum(ulong_t id)
{
	register ulong_t context;

	asm volatile("getcnum	%0,%1"
			: "=r"(context)
			: "r"(id));

	return context;
}


/* I/O Load/Store */
static inline byte_t iolb(ulong_t addr)
{
	byte_t ret;

	asm volatile ("iolb	%0,%1"
			: "=r"(ret)
			: "r"(addr));
	return ret;
}

static inline hword_t iolh(ulong_t addr)
{
	hword_t ret;

	asm volatile ("iolh	%0,%1"
			: "=r"(ret)
			: "r"(addr));
	return ret;
}

static inline ulong_t iolw(ulong_t addr)
{
	ulong_t ret;

	asm volatile ("iolw	%0,%1"
			: "=r"(ret)
			: "r"(addr));
	return ret;
}

#endif	/* __RT_FRONTIER_RMT_OPS_H__ */
