#ifndef __PCI_H__
#define __PCI_H__

#include <asm.h>
#include <types.h>
#include <io.h>

#define RMT_PCI_BASE	0xffff3000
/* Interruption Control Register */
#define RMT_PCI_ICR 	(RMT_PCI_BASE + 0x0000)
#define ICR_ID 0x3 << 1
#define ICR_ID0 0x1 << 1
#define ICR_ID1 0x1 << 2
#define ICR_LS 0x3 << 3
#define ICR_LS_L 0x1 << 3
#define ICR_LS_H 0x1 << 4
#define ICR_LE 0x1 << 5
#define ICR_PS 0x1 << 6
#define ICR_PE 0x1 << 7 
#define ICR_LMS 0x1 << 9
#define ICR_LME 0x1 << 10
#define ICR_LTS 0x1 << 11
#define ICR_LTE 0x1 << 12
#define ICR_EE 0x1 << 13
#define ICR_ME 0x1 << 14
#define ICR_TE 0x1 << 15
/* Mail Boxes */
#define RMT_PCI_MAILBOX_A			(RMT_PCI_BASE + 0x0008)
#define RMT_PCI_MAILBOX_A_L			(RMT_PCI_BASE + 0x0008)
#define RMT_PCI_MAILBOX_A_H			(RMT_PCI_BASE + 0x000c)
//#define RMT_PCI_MAILBOX_A_L			(RMT_PCI_BASE + 0x000c)
//#define RMT_PCI_MAILBOX_A_H			(RMT_PCI_BASE + 0x0008)
#define RMT_PCI_MAILBOX_B			(RMT_PCI_BASE + 0x0010)
#define RMT_PCI_CH0_DAMR			(RMT_PCI_BASE + 0x004c)
#define RMT_PCI_CH0_DMCR			(RMT_PCI_BASE + 0x004d)
#define RMT_PCI_CH0_DCR				(RMT_PCI_BASE + 0x004e)
#define RMT_PCI_CH0_FAMR			(RMT_PCI_BASE + 0x005c)
#define RMT_PCI_CH0_FMCR			(RMT_PCI_BASE + 0x005d)
#define RMT_PCI_CH0_FCR				(RMT_PCI_BASE + 0x005e)
#define RMT_PCI_CH1_DAMR			(RMT_PCI_BASE + 0x00cc)
#define RMT_PCI_CH1_DMCR			(RMT_PCI_BASE + 0x00cd)
#define RMT_PCI_CH1_DCR				(RMT_PCI_BASE + 0x00ce)
#define RMT_PCI_CH1_FAMR			(RMT_PCI_BASE + 0x00dc)
#define RMT_PCI_CH1_FMCR			(RMT_PCI_BASE + 0x00dd)
#define RMT_PCI_CH1_FCR				(RMT_PCI_BASE + 0x00de)

/* Local AD */
#define RMT_PCI_LOCAL_AD			(RMT_PCI_BASE + 0x0018)

#if 0
/* this is little endian! */
#define RMT_PCI_CONF_OFFSET			(RMT_PCI_BASE + 0x100)
#define RMT_PCI_CONF_DEVICE_ID		(RMT_PCI_BASE + 0x100)
#define RMT_PCI_CONF_VENDOR_ID		(RMT_PCI_BASE + 0x102)
#define RMT_PCI_CONF_STATUS			(RMT_PCI_BASE + 0x104)
#define STATUS_66MHZ_CAPABLE		(0x0000 | (0x1 << 5))
#define RMT_PCI_CONF_COMMAND		(RMT_PCI_BASE + 0x106)
#define RMT_PCI_CONF_CLASS_CODE		(RMT_PCI_BASE + 0x108)
#define RMT_PCI_CONF_REVISION_ID	(RMT_PCI_BASE + 0x10a)
#define RMT_PCI_CONF_BIST			(RMT_PCI_BASE + 0x10c)
#define RMT_PCI_CONF_HEADER_TYPE	(RMT_PCI_BASE + 0x10d)
#define RMT_PCI_CONF_LATENCY_TIMER	(RMT_PCI_BASE + 0x10e)
#define RMT_PCI_CONF_CACHE_LINE_SIZE	(RMT_PCI_BASE + 0x10f)
#else
/* this is big endian! */
#define RMT_PCI_CONF_OFFSET			(RMT_PCI_BASE + 0x100)
#define RMT_PCI_CONF_DEVICE_ID		(RMT_PCI_BASE + 0x102)
#define RMT_PCI_CONF_VENDOR_ID		(RMT_PCI_BASE + 0x100)
#define RMT_PCI_CONF_STATUS			(RMT_PCI_BASE + 0x106)
#define STATUS_66MHZ_CAPABLE		(0x0000 | (0x1 << 5))
#define RMT_PCI_CONF_COMMAND		(RMT_PCI_BASE + 0x104)
#define RMT_PCI_CONF_CLASS_CODE		(RMT_PCI_BASE + 0x10a)
#define RMT_PCI_CONF_REVISION_ID	(RMT_PCI_BASE + 0x108)
#define RMT_PCI_CONF_BIST				(RMT_PCI_BASE + 0x10f)
#define RMT_PCI_CONF_HEADER_TYPE		(RMT_PCI_BASE + 0x10e)
#define RMT_PCI_CONF_LATENCY_TIMER		(RMT_PCI_BASE + 0x10d)
#define RMT_PCI_CONF_CACHE_LINE_SIZE	(RMT_PCI_BASE + 0x10c)
#endif

#define PCI_CMD_READ	0x11111111 /* PCI -> LOCAL */
#define PCI_CMD_WRITE	0x22222222 /* LOCAL -> PCI */
#define PCI_CMD_READ_BAP  0x22222223
#define PCI_CMD_READ_DMA  0x22222224
#define PCI_CMD_END		0x33333333
#define PCI_CMD_INIT	0x44444444
#define PCI_CMD_ACK		0xeeeeeeee
#define PCI_CMD_LOAD	0xffffffff

#define PCI_TEST_ADDR	0x80000000
#define PCI_TEST_DATA1	0xa5a5a5a5
#define PCI_TEST_DATA2	0x5a5a5a5a

#define MASK_BYTE	0x000000ff

static inline void pci_write_local_ad(ulong_t addr)
{
	unsigned char buf[4];
	ulong_t big_addr;
	buf[0] = (addr >> 24) & MASK_BYTE;
	buf[1] = (addr >> 16) & MASK_BYTE;
	buf[2] = (addr >> 8) & MASK_BYTE;
	buf[3] = (addr >> 0) & MASK_BYTE;
	big_addr = buf[0] | (buf[1] << 8) | (buf[2] << 16) | (buf[3] << 24);
	MEM_WR_W(big_addr, RMT_PCI_LOCAL_AD);
}

static inline void pci_read_mailbox(ulong_t *x, ulong_t *y)
{
	ulong_t ls;
	/* wait for LIS down */
	do {
		ls = iolw(RMT_PCI_ICR);
		nop7();
	} while(!(ls & ICR_LS));
	/* get the value from the mailbox */
	if (x) {
		*x = iolw(RMT_PCI_MAILBOX_A_L);
		nop7();
	}
	if (y) {
		*y = iolw(RMT_PCI_MAILBOX_A_H);
		nop7();
	}
	/* clear the LIS */
	MEM_WR_W(ls | ICR_LS, RMT_PCI_ICR);
}

static inline void pci_write_mailbox(ulong_t x)
{
	ulong_t ls;
	/* wait for PCI Bus Interrupt Status down */
	do {
		ls = iolw(RMT_PCI_ICR);
		nop7();
		sync();
	} while(ls & ICR_PS);
	MEM_WR_W(x, RMT_PCI_MAILBOX_B);
}

static inline void pci_ch0_reset(void)
{
	MEM_WR_B(0x0, RMT_PCI_CH0_DCR);
	MEM_WR_B(0x0, RMT_PCI_CH0_FCR);
}

static inline void pci_ch1_reset(void)
{
	MEM_WR_B(0x0, RMT_PCI_CH1_DCR);
	MEM_WR_B(0x0, RMT_PCI_CH1_FCR);
}

static inline void pci_reset(void)
{
	pci_ch0_reset();
	pci_ch1_reset();
}

static inline void pci_trans_ch0_start(void)
{
	MEM_WR_B(0x00, RMT_PCI_CH0_DMCR);
	MEM_WR_B(0x00, RMT_PCI_CH0_FMCR);
}

static inline void pci_trans_ch1_start(void)
{
	MEM_WR_B(0x00, RMT_PCI_CH1_DMCR);
	MEM_WR_B(0x00, RMT_PCI_CH1_FMCR);
}

static inline void pci_trans_stop(void)
{
	MEM_WR_B(0x1, RMT_PCI_CH0_DAMR);
	MEM_WR_B(0x1, RMT_PCI_CH0_FAMR);
}

static inline void pci_ls_clear(void)
{
	ulong_t tmp = iolw(RMT_PCI_ICR);
	MEM_WR_W(tmp | ICR_LS_H | ICR_LS_L, RMT_PCI_ICR);
}

static inline void pci_le_enable(void)
{
	ulong_t tmp = iolw(RMT_PCI_ICR);
	MEM_WR_W(tmp | ICR_LE, RMT_PCI_ICR);
}

static inline void pci_le_disable(void)
{
	ulong_t tmp = iolw(RMT_PCI_ICR);
	MEM_WR_W(tmp & ~ICR_LE, RMT_PCI_ICR);
}

extern int pci_setup(void);
extern ssize_t pci_write(int, const void *, size_t);
extern ssize_t pci_read(int, const void *, size_t);

#endif /* __PCI_H__ */
