#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <elf.h>
//#include <asm/page.h>
#include "rmtpcild.h"

int dma_page_order = 0;
#define	dma_page_size	(PAGE_SIZE << dma_page_order)

#define ERROR(...) printf("*Error: "  __VA_ARGS__)
#define BYTE_GET(field)  byte_get_big_endian(field, sizeof(field))

typedef struct {
	unsigned int offset;
	unsigned int size;
	unsigned int addr;
	unsigned int reminder;
} segment;

/* ELF Header (32-bit implementations) */
typedef struct {
  unsigned char e_ident[16];            /* ELF "magic number" */
  unsigned char e_type[2];              /* Identifies object file type */
  unsigned char e_machine[2];           /* Specifies required architecture */
  unsigned char e_version[4];           /* Identifies object file version */
  unsigned char e_entry[4];             /* Entry point virtual address */
  unsigned char e_phoff[4];             /* Program header table file offset */
  unsigned char e_shoff[4];             /* Section header table file offset */
  unsigned char e_flags[4];             /* Processor-specific flags */
  unsigned char e_ehsize[2];            /* ELF header size in bytes */
  unsigned char e_phentsize[2];         /* Program header table entry size */
  unsigned char e_phnum[2];             /* Program header table entry count */
  unsigned char e_shentsize[2];         /* Section header table entry size */
  unsigned char e_shnum[2];             /* Section header table entry count */
  unsigned char e_shstrndx[2];          /* Section header string table index */
} Elf32_External_Ehdr;


/* Program header */
typedef struct {
  unsigned char p_type[4];		/* Type of segment */
  unsigned char p_offset[4];	/* Offset of segment entry */
  unsigned char p_vaddr[4];		/* Virtual address of segment entry */
  unsigned char p_paddr[4];		/* Physical address of segment */
  unsigned char p_filesz[4];	/* Size of file image*/
  unsigned char p_memsz[4];		/* Size of memory image */
  unsigned char p_flags[4];		/* Miscellaneous segment attribute */
  unsigned char p_align[4];		/* Segment alignment */
} Elf32_External_Phdr;

int byte_get_big_endian (unsigned char* field, int size)
{
	switch (size) {
    case 1:
		return *field;

    case 2:
		return ((unsigned int) (field[1])) | (((int) (field[0])) << 8);

    case 4:
		return ((unsigned long) (field[3]))
			|   (((unsigned long) (field[2])) << 8)
			|   (((unsigned long) (field[1])) << 16)
			|   (((unsigned long) (field[0])) << 24);

    case 8:
		/* Although we are extracing data from an 8 byte wide field, we
		   are returning only 4 bytes of data.  */
		return ((unsigned long) (field[7]))
			|   (((unsigned long) (field[6])) << 8)
			|   (((unsigned long) (field[5])) << 16)
			|   (((unsigned long) (field[4])) << 24);

    default:
		printf ("Unhandled data length: %d\n", size);
		exit(1);
    }

	return 0;
}

int word_endian(int x)
{
	return (((x << 24) & 0xff000000) | ((x << 8) & 0x00ff0000)
			| ((x >> 8) & 0x0000ff00) | ((x >> 24) & 0x000000ff));
}



void show_usage(void) {
	printf("Usage: mips-rmt-elf-obj2prg SOURCE\n");
	printf("SOURCE is a elf format binary file.\n");
	printf("\n");
}

int main(int argc, char** argv)
{
	Elf32_Ehdr eh;
	Elf32_Phdr *ph = NULL;
	Elf32_Phdr *phdrs = NULL;
	Elf32_External_Ehdr xeh;
	Elf32_External_Phdr *xphdrs = NULL;
	segment *load_segments = NULL;
	unsigned int i, j, k;
	char *binary = NULL;
	FILE *fbin = NULL;
	int load_seg_num = 0;
	unsigned char **buf = NULL;
	unsigned char byte_data;
	int fd;
	int ret;
	int tmp;
	int addr, size;
	int cmd[2];

	/* elf program */
	binary = argv[1];

	/* open binary file. */
	if (!(fbin = fopen(binary, "rb"))) {
		printf("Failed to open %s\n", binary);
		goto end;
	}

	/* read elf header. */
	if (!(fread(&xeh, sizeof(xeh), 1, fbin))) {
		printf("Failed to read %s\n", binary);
		goto end;
	}

	/* transform to little endian byte */
	eh.e_type      = BYTE_GET (xeh.e_type);
	eh.e_machine   = BYTE_GET (xeh.e_machine);
	eh.e_version   = BYTE_GET (xeh.e_version);
	eh.e_entry     = BYTE_GET (xeh.e_entry);
	eh.e_phoff     = BYTE_GET (xeh.e_phoff);
	eh.e_shoff     = BYTE_GET (xeh.e_shoff);
	eh.e_flags     = BYTE_GET (xeh.e_flags);
	eh.e_ehsize    = BYTE_GET (xeh.e_ehsize);
	eh.e_phentsize = BYTE_GET (xeh.e_phentsize);
	eh.e_phnum     = BYTE_GET (xeh.e_phnum);
	eh.e_shentsize = BYTE_GET (xeh.e_shentsize);
	eh.e_shnum     = BYTE_GET (xeh.e_shnum);
	eh.e_shstrndx  = BYTE_GET (xeh.e_shstrndx);

	/* The head message */
	printf("//\n");
	printf("// Source file: %s\n", binary);
	/* print the entry point. */
	printf("// Entry point: 0x%08x\n", eh.e_entry);
	printf("//\n\n");

	/* allocate for external program headers */
	xphdrs = (Elf32_External_Phdr *) malloc (eh.e_phentsize * eh.e_phnum);

	/* seek file. */
	if (fseek(fbin, eh.e_phoff, SEEK_SET)) {
		printf("Failed to seek %s\n", binary);
		goto end;
	}

	/* read program headers. */
	if (!(fread(xphdrs, eh.e_phentsize * eh.e_phnum, 1, fbin))) {
		printf("Failed to read %s\n", binary);
		goto end;
	}

	/* allocate for program headers. */
	phdrs = (Elf32_Phdr *) 
		malloc (eh.e_phnum * sizeof (Elf32_Phdr));

	/* transfer to the little endian */
	for (i = 0, ph = phdrs; i < eh.e_phnum; i++, ph++) {
		ph->p_type		= BYTE_GET (xphdrs[i].p_type);
		ph->p_offset	= BYTE_GET (xphdrs[i].p_offset);
		ph->p_vaddr		= BYTE_GET (xphdrs[i].p_vaddr);
		ph->p_paddr		= BYTE_GET (xphdrs[i].p_paddr);
		ph->p_filesz	= BYTE_GET (xphdrs[i].p_filesz);
		ph->p_memsz		= BYTE_GET (xphdrs[i].p_memsz);
		ph->p_flags		= BYTE_GET (xphdrs[i].p_flags);
		ph->p_align		= BYTE_GET (xphdrs[i].p_align);
    }

	/* the number of loadable segments */
	for (i = 0, ph = phdrs; i < eh.e_phnum; i++, ph++) {
		if (PT_LOAD & ph->p_type)
			load_seg_num++;
	}
	load_segments = (segment*) malloc(sizeof(segment) * load_seg_num);

	printf("[Loadable segments]\n");
	buf = (unsigned char**) malloc(sizeof(unsigned char *) * load_seg_num);
	for (j = 0, i = 0, ph = phdrs; i < eh.e_phnum; i++, ph++) {
		if (PT_LOAD & ph->p_type) {
			load_segments[j].offset = ph->p_offset;
			load_segments[j].size = ph->p_filesz;
			load_segments[j].addr = ph->p_paddr;
			load_segments[j].reminder = load_segments[i].size % sizeof(long);
			printf("segment address: 0x%08x\n", load_segments[j].addr);
			printf("segment offset: 0x%x\n", load_segments[j].offset);
			printf("segment size: %d bytes\n", load_segments[j].size);
			printf("\n");
			if (load_segments[j].reminder != 0) {
				printf("This segment is not aligned!\n");
				printf("Forcefully aligning!\n");
			}

			tmp = load_segments[j].size
				+ load_segments[j].reminder 
				+ sizeof(long) * 2;
			/* the data of the loadable segment */
			buf[j] = (unsigned char*) malloc(tmp);
			memset(buf[j], 0, tmp);
			j++;
		} 
	}

	/* seek to the begining. */
	if (fseek(fbin, 0, SEEK_SET)) {
		printf("Failed to seek %s\n", binary);
		goto end;
	}

	/* copy all the loadble segments to the buffer */
	for (i = 0; i < load_seg_num; i++) {
		((unsigned long*)buf[i])[0] = load_segments[i].addr;
		((unsigned long*)buf[i])[1] = load_segments[i].size;
		/* seek to the offset of segment */
		fseek(fbin, load_segments[i].offset, SEEK_SET);
		for (j = 0; j < load_segments[i].size; j++) {
			/* read word as the format of big endian*/
			fread(&byte_data, sizeof(byte_data), 1, fbin);
			/* copy to the buffer */
			buf[i][j + sizeof(long) * 2] = byte_data;
		}
	}

#if 0
	for (i = 0; i < load_segments[0].size / 4; i++) {
		printf("0x%08lx\n", ((unsigned long*)buf[0])[i + 2]);
	}
#endif

	printf("Opening the device...\n");
	/* open the pci device */
	fd = open(DEVNAME, O_RDWR);
	if (fd < 0) {
		ERROR("Cannot open %s.\n", DEVNAME);
		goto end;
	}

	printf("Now loading...\n");
#if 0
	cmd[0] = RMT_CMD_LOAD;
	cmd[1] = 0;
	write(fd, (char*)cmd, 0);
#endif

	for (i = 0; i < load_seg_num; i++) {
		/* send the data */
		tmp = load_segments[i].size + load_segments[i].reminder;
		ret = write(fd, buf[i], tmp);
		printf("tried size = %d, sent size = %d\n", tmp, ret);
		if (ret < 0) {
			printf("Cannot write the data.\n");
			goto out;
		}
	}

	printf("\n");

	/* notify to finish loading */
	printf("Finishing...\n");

	/* entry point */

	//	cmd[0] = RMT_CMD_END;
	cmd[0] = RMT_CMD_LOAD;
	cmd[1] = word_endian(eh.e_entry);
	ret = write(fd, (char*)cmd, 0);
	if (ret < 0) {
		printf("Cannot write the command.\n");
		goto out;
	}
#if 0
	/* send the entry point */
	addr = word_endian(eh.e_entry);
	ret = write(fd, (char*)&addr, 0);
	if (ret < 0) {
		printf("Cannot write the entry.\n");
		goto out;
	}
#endif

 out:
	/* close the device */
	close(fd);

 end:
	/* 
	 * free memories 
	 */
	if (buf) {
		for (i = 0; i < load_seg_num; i++)
			free(buf[i]);
		free(buf);
	}

	if (xphdrs)
		free(xphdrs);

	if (phdrs)
		free(phdrs);

	if (load_segments)
		free(load_segments);

	/* close file stream. */
	if (fbin)
		fclose(fbin);

	return 0;
}
